/*---------------------------------------------
                 S I M P A C K
            Simulation Tool Package
        Copyright 1990, Paul A. Fishwick
---------------------------------------------*/
/* 
   Timed PETRI NET Simulator

   This program simulates any timed Petri net represented in the
   data structure given in array 't' (standing for 't'ransition).

C version of this program is written in C++ by S.Mukhopadhyay

*/

#include<stream.h>
#include<stdlib.h>
#include<stdio.h>
#include "../../queuing/queuing.h"

const BEGINFIRE= 1;
const ENDFIRE= 2;
const MAX_TRANS= 100;
const MAX_PLACES= 200;
const MAX_BRANCH= 20;


// transitions are facilities 
// class Petrinet inherits attributes of class Token and Events

class Petrinet: public Token, public Events {

public:
 
 void read_net();
 void create_out_set();
 void display_table_header();
 void display_table_trailer();
 void random_array(int max, int *array);
 void simulation_loop();
 void check_transition(int event, int transition_number);
 Petrinet();

private:

int transition[MAX_TRANS],tr_status[MAX_TRANS]; 
int t_in[MAX_TRANS][MAX_BRANCH],t_out[MAX_TRANS][MAX_BRANCH];
int out_set[MAX_TRANS][MAX_BRANCH],out_set_count[MAX_TRANS];
int p[MAX_PLACES],iteration_count;
int places,transitions,firings,array[500],number,events;
float trans_time[MAX_TRANS];
Token a_token;
};


Petrinet::Petrinet()   //class Petrinet constructor
{
 int i;
 events = 0;
 for (i=0;i<transitions;i++) out_set_count[i] = 0;
}


/* Format of the input file is as follows:
   #transitions #places #firings
   "#places for input to transition 0" place-0 place-1 ...
     "#places for output from transition 0" place-0 place-1 ...
   "#places for input to transition 1" place-0 place-1 ...
     "#places for output from transition 1" place-0 place-1 ...
   ...........
   "#places for input to transition (#transitions-1)" place-0 place-1 ...
     "#places for output from transition (#transitions-1)" place-0 place-1 ...
   tokens-in-place-0 tokens-in-place-1 ... tokens-in-place (#places-1)
*/


main()
/*
 Driver Program - Check each transition for some given number of
iterations (or until a given absolute simulation time has been
reached).
*/
{
  Petrinet pnet;

  init_simpack(HEAP); 
  pnet.read_net();        // read in network data
 
  pnet.create_out_set();  // create a transition out set table 
  
  pnet.display_table_header();    // Print Table Header
  
  pnet.simulation_loop(); // Simulate this run for a designated number 
                          // of transition firing

  pnet.display_table_trailer();   // Print Table Trailer
}


void Petrinet::display_table_header()
{
 int i;
  
  cout << form("Petri net contains %3d transitions and %3d places.\n",
         transitions,places);
  cout << form("Simulate for %d firings.\n\n",firings);

 // Print Table Header 
  cout << "TIME   " << "TRS ";
  for (i=0;i<places;i++)
      cout << form("%2d ",i);
  cout << "\n";
  for(i=0;i<(11+places*3);i++) cout << "-";
  cout << "\n";

 // print out initial place array 
  cout << form("%6.2f ",time());
  cout << "XX: ";
  for (i=0;i<places;i++)
      cout << form("%2d ",p[i]);
  cout << "\n";
}



void Petrinet::simulation_loop()
{
   int i,event,transition_number;

// Simulate this run for a designated number of transition firings

  for(i=0;i<transitions;i++) {
   a_token.attr[0] = (float) i;
   schedule(BEGINFIRE,0.0,a_token);
   events++;
  }
  iteration_count = firings;

/* continue to check all net transitions until a total of
     'firings' transitions have been fired - check transitions
     in a random fashion to insure fairness.
  */

   while (iteration_count != 0 && events > 0) {

// take one event from the event list and cause it to occur 

   next_event(event, a_token);
   transition_number = (int) a_token.attr[0];
   events--;
   check_transition(event,transition_number);
  } /* end while */
}




void Petrinet::display_table_trailer()
{
  // Print Table Trailer 
  int i;
  for(i=0;i<(11+places*3);i++) cout << "-";
  cout << "\n";
}



void Petrinet::check_transition(int event,  int transition_number)
{
 /*
 Check a transition for firing. If the transition is not already busy
 then fire it as long as at least one token exists in each input
 place for that transition.
*/
 
  int input_places,output_places,i,fire,tokens,out_set_number;
 
  switch(event) {

  case BEGINFIRE: // check transition for firing 

  input_places = t_in[transition_number][0];
  fire = TRUE;
  for(i=0;i<input_places;i++) {
    tokens = p[t_in[transition_number][i+1]];
    fire = fire && (tokens > 0);
  }
  if ((tr_status[transition_number] == FREE) && fire) {

// delete one token from each input place 
    for (i=0;i<input_places;i++) p[t_in[transition_number][i+1]] -= 1;
    tr_status[transition_number] = BUSY;
    a_token.attr[0] = (float) transition_number;
    schedule(ENDFIRE,trans_time[transition_number],a_token);
    events++;
  } // end if 
  break;

    case ENDFIRE: // end of transition fire 
  
    tr_status[transition_number] = FREE;
// add one token to each output place 
    input_places = t_in[transition_number][0];
    output_places = t_out[transition_number][0];
    for (i=0;i<output_places;i++) 
         p[t_out[transition_number][i+1]] += 1;

// firing just occurred, print out the 'p'lace array 
    cout << form("%6.2f ",time());
    cout << form("%2d: ",transition_number);
    for (i=0;i<places;i++)
      cout<< form("%2d ",p[i]);
      cout << "\n";

// sweep through outset transitions once to schedule new events 
   out_set_number = out_set_count[transition_number];
   for (i=0;i<out_set_number;i++) array[i] = out_set[transition_number][i];
   
   random_array(out_set_number-1, array); 
   
   for(number=0;number<out_set_count[transition_number];number++) {   
      a_token.attr[0] = (float) array[number];
      schedule(BEGINFIRE,0.0,a_token);
      events++;
    }
    iteration_count--;
    break;

} // end switch 
} // end check_transition 




void Petrinet::random_array(int max, int *array)
{
  int element,i,swap;
// rearrange array to yield a random ordering
  for(i=0;i<max;i++) {
    element = random(0,max-i);
    swap = array[element];
    array[element] = array[max-i];
    array[max-i] = swap;
  }
}


void Petrinet::read_net()
{
  int i,j,number_inputs,number_outputs;
  cin >> transitions >> places >> firings;
  for(i=0;i<transitions;i++) {
     cin >> trans_time[i] >> t_in[i][0];
     number_inputs = t_in[i][0];
     for(j=0;j<number_inputs;j++) cin >> t_in[i][j+1];
     cin >> t_out[i][0];
     number_outputs = t_out[i][0];
     for(j=0;j<number_outputs;j++) cin >> t_out[i][j+1];
   }
// Read place information 
   for(i=0;i<places;i++) cin >> p[i];
} // end read_net() 




void Petrinet::create_out_set()
{
  int tr,tr1,tr2,op,in,found;

  for (tr1=0;tr1<transitions;tr1++) 
    for (op=1;op<=t_out[tr1][0];op++)
      for (tr2=0;tr2<transitions;tr2++)
        for (in=1;in<=t_in[tr2][0];in++) 
          if (t_in[tr2][in] == t_out[tr1][op]) {
          // check to see if the transition is already stored 
            found = FALSE;
            for (tr=0;tr<out_set_count[tr1];tr++)
                if (out_set[tr1][tr] == tr2) found = TRUE;
            if (!found) {
                out_set[tr1][out_set_count[tr1]] = tr2;
                out_set_count[tr1]++;
            } // end if 
	   }
} // end out_set() 





