(*---RCS--- $Log:	_cexp.sml,v $
Revision 1.1  92/09/17  14:17:23  birkedal
Edinburgh Version 11Sep92
*)

(*$CExp: LVARS PRIMS FINMAP PRINTUTIL PRETTYPRINT CEXP*)
(* Continuation-passing expressions are a subset of lambda expressions.
   All applications must be tail-recursive (this is why we get away with
   no stack); it's convenient to make that a "syntactic" restriction in
   this datatype. Static scoping rules still apply when the CExp's are
   first generated, although closures will get eliminated later. *)

functor CExp(structure Lvars: LVARS
	     structure Prims: PRIMS
	     structure FinMap: FINMAP
	     structure PrintUtil: PRINTUTIL

	     structure PP: PRETTYPRINT
	       sharing type FinMap.StringTree = PP.StringTree
	    ): CEXP =
  struct
    type lvar = Lvars.lvar
    type prim = Prims.prim
    type (''a, 'b) map = (''a, 'b) FinMap.map

    datatype 'a option = NONE | SOME of 'a

    datatype CExp =
        FIX      of {f: lvar, formals: lvar list, body: CExp} list * CExp
      | APP      of {f: lvar, actuals: Simple list}
      | PRIM_APP of {cont: lvar, prim: prim, arg: lvar}
      | SELECT   of int * lvar
      | SWITCH_I of int Switch
      | SWITCH_S of string Switch
      | SWITCH_R of real Switch
      | SIMPLE	 of Simple

    and Simple =
        VAR      of lvar
      | INTEGER  of int
      | STRING   of string
      | REAL     of real
      | VECTOR	 of lvar list
      | VOID

    and 'a Switch = SWITCH of {arg: lvar,
			       selections: ('a, CExp) map,
			       wildcard: CExp option
				(* mandatory for REAL or STRING switches. *)
			      }


    type StringTree = PP.StringTree

    fun layoutSwitch(name, layoutCExp, pr,
		     (SWITCH{arg, selections, wildcard})
		    ) =
      let
	val selections =
	  FinMap.layoutMap {start="", eq=" -> ", sep="; ", finish=""}
	                   (PP.layoutAtom pr)
			   layoutCExp
			   selections
	val wildcard_L =
	  case wildcard
	    of NONE => nil
	     | SOME w =>
		 [PP.NODE{start="_ -> ", finish="", indent=3, childsep=PP.NONE,
			  children=[layoutCExp w]
			 }
		 ]
      in
	PP.NODE{start=name ^ "(" ^ Lvars.pr_lvar arg ^ ": ",
	        finish=")", indent=3,
		children=selections :: wildcard_L, childsep=PP.RIGHT "; "
	       }
      end

    fun layoutCExp cexp: StringTree =
      case cexp
	of FIX(bindings, scope) =>
	     layoutFix(bindings, scope)

         | APP{f, actuals} =>
	     PP.NODE{start=Lvars.pr_lvar f ^ "(", finish=")", indent=3,
		     children=map layoutSimple actuals, childsep=PP.RIGHT ", "
		    }

         | PRIM_APP{cont, prim, arg} =>
	     PP.NODE{start=Lvars.pr_lvar cont ^ "(", finish=")", indent=3,
		     children=[PP.layoutAtom Prims.pr_prim prim,
			       PP.layoutAtom Lvars.pr_lvar arg
			      ],
		     childsep=PP.RIGHT ", "
		    }

         | SELECT(int, lv) =>
	     PP.LEAF(implode ["SELECT(", PrintUtil.intToString int,
			      ", ", Lvars.pr_lvar lv, ")"
			     ]
		    )

	 | SWITCH_I sw =>
	     layoutSwitch("SWITCH_I", layoutCExp, PrintUtil.intToString, sw)

	 | SWITCH_S sw =>
	     layoutSwitch("SWITCH_S", layoutCExp, PrintUtil.Printable, sw)

	 | SWITCH_R sw =>
	     layoutSwitch("SWITCH_R", layoutCExp, PrintUtil.realToString, sw)

	 | SIMPLE x => layoutSimple x

    and layoutFix(bindings, scope) =
      PP.NODE{start="FIX(", finish=")", indent=4, childsep=PP.LEFT " in ",
	      children=[layoutBindings bindings, layoutCExp scope]
	     }

    and layoutBindings bindings =
      PP.NODE{start="", finish="", indent=0, childsep=PP.RIGHT ": ",
	      children=map layoutBinding bindings
	     }

    and layoutBinding{f, formals, body} =
      PP.NODE{start=implode [Lvars.pr_lvar f, "(", pr_formals formals, ") = "],
	      finish="", indent=0, childsep=PP.NONE,
	      children=[layoutCExp body]
	     }

    and pr_formals nil = ""
      | pr_formals [x] = Lvars.pr_lvar x
      | pr_formals(x :: xs) = Lvars.pr_lvar x ^ ", " ^ pr_formals xs

    and layoutSimple x =
      case x
	of VAR lv => PP.LEAF(Lvars.pr_lvar lv)
	 | INTEGER i => PP.LEAF(PrintUtil.intToString i)
	 | STRING s => PP.LEAF("\"" ^ PrintUtil.Printable s ^ "\"")
	 | REAL r => PP.LEAF(PrintUtil.realToString r)

	 | VECTOR lvars =>
	     PP.NODE{start="[", finish="]", indent=1, childsep=PP.RIGHT ", ",
		     children=map (PP.layoutAtom Lvars.pr_lvar) lvars
		    }

	 | VOID => PP.LEAF "VOID"
  end;
