(* Pure evaluator for declarations. *)

(*
$File: Interpreter/EvalDec.sml $
$Date: 1993/02/24 07:59:48 $
$Revision: 1.21 $
$Locker: birkedal $
*)

(*$EvalDec:
	DEC_GRAMMAR RESIDENT BASIC_VALUE SPECIAL_VALUE 
	OVERLOADING_INFO GRAMMAR_INFO CORE_DYNOBJECT MODULE_DYNOBJECT
	FINMAP SORTED_FINMAP PRETTYPRINT CRASH IOStreams Apply EVALDEC
 *)

functor EvalDec(structure Grammar : DEC_GRAMMAR

		structure ResIdent : RESIDENT
		  sharing type Grammar.longid = ResIdent.longid

		structure BasicValue: BASIC_VALUE
		structure SpecialValue: SPECIAL_VALUE

		structure OverloadingInfo: OVERLOADING_INFO
		structure GrammarInfo: GRAMMAR_INFO
		  sharing type GrammarInfo.PostElabGrammarInfo = Grammar.info
		      and type GrammarInfo.OverloadingInfo = OverloadingInfo.info

		structure CoreDynObject : CORE_DYNOBJECT
		  sharing CoreDynObject.Grammar = Grammar
		      and type CoreDynObject.Lab.lab = Grammar.lab
		      and type CoreDynObject.scon = Grammar.scon
		      and type CoreDynObject.Var.longvar = ResIdent.longvar
		      and type CoreDynObject.Con.longcon = ResIdent.longcon
		      and type CoreDynObject.Excon.longexcon
			       = ResIdent.longexcon
		      and type CoreDynObject.Excon.excon = Grammar.excon
		      and type CoreDynObject.BasVal = BasicValue.BasVal
		      and type CoreDynObject.SVal = SpecialValue.SVal
		      and type CoreDynObject.longstrid = Grammar.longstrid

	       (* Identifiers in the grammar are always
		  resolved to be variables: *)
		      and type CoreDynObject.Var.var = Grammar.id

		structure ModuleDynObject : MODULE_DYNOBJECT
		  sharing type ModuleDynObject.Env = CoreDynObject.Env

		structure SortedFinMap: SORTED_FINMAP
		  sharing type CoreDynObject.map = SortedFinMap.map

		structure FinMap: FINMAP

		structure PP : PRETTYPRINT
		  sharing type CoreDynObject.StringTree = PP.StringTree

		structure Crash : CRASH

		val USE: string -> unit	(* use a file, raise host ML's Io if
					   unable to open it. *)
	       ): EVALDEC = 
  struct
    structure Excon = CoreDynObject.Excon
    type dec = Grammar.dec
    type Env = CoreDynObject.Env
    type ExName = CoreDynObject.ExName.ExName
    type Val = CoreDynObject.Val

    structure C = CoreDynObject

    exception FAIL

   (* before we build Apply, we need a function that it can use to let
      APPLY raise exceptions. This is also used in evalExp(). *)

    fun RAISE_pack p =
      raise CoreDynObject.EXCEPTION p

    fun RAISE_val v =
      let
	open CoreDynObject
      in
	RAISE_pack(case v
		     of EXNAME0val name => EXNAME0pack name
		      | EXNAME1val(name, v) => EXNAME1pack(name, v)
		      | _ => Crash.impossible "EvalDec.RAISE_val"
		  )
      end

    structure Apply =
      Apply(structure CoreDynObject = CoreDynObject
	    structure SpecialValue = SpecialValue
	    structure BasicValue = BasicValue
	    structure SortedFinMap = SortedFinMap
	    structure IOStreams = IOStreams(structure FinMap = FinMap
					    structure Crash = Crash
					   )
	    structure Crash = Crash
	    val USE = USE
	    val RAISE = RAISE_val	(* Apply needs to raise values and
					   have them turned into packets. *)
	   )

    val FAIL_USE = Apply.FAIL_USE

    open Grammar ResIdent

   (* How to raise Match and Bind from FAIL: *)
    local
(*
      val B  = InitDynBasis.basis	(* Initial Basis *)
      val E  = ModuleDynObject.E_of_B B	(* Initial Environment *)
      val EE = CoreDynObject.EE_of_E E  (* Initial Excon environment *)

      val match_NAME =
	CoreDynObject.lookupEE_excon EE Excon.E_MATCH

      and bind_NAME =
	CoreDynObject.lookupEE_excon EE Excon.E_BIND
*)
    in
      fun raiseMatch() =
	let
	  open CoreDynObject
	in
	  raise EXCEPTION(EXNAME0pack(ExName.new "<void>"))
	end

	(* Crash.unimplemented "raiseMatch" *)
	(* raise DynObject.EXCEPTION(DynObject.EXNAMEpack match_NAME) *)

      and raiseBind() =
	Crash.unimplemented "raiseBind"
	(* raise DynObject.EXCEPTION(DynObject.EXNAMEpack bind_NAME) *)
    end

   (****** atomic expressions - Definition v3 pages 49-50 ******)

    fun evalAtexp(E, atexp) =
      case atexp
	of SCONatexp(_, scon) =>
	     C.Sval(C.mkSValSCon scon)

         | IDENTatexp(i, OP_OPT(LONGVAR lvar, _)) =>
	     let 
	       val v = (C.lookup_LongVar(E, lvar)) 
	       open CoreDynObject
	       fun resolve_int (b: BasicValue.BasVal) =
		 case b of
		     BasicValue.ABS       => BasicValue.ABS_INT
		   | BasicValue.NEG       => BasicValue.NEG_INT
		   | BasicValue.SUM       => BasicValue.SUM_INT
		   | BasicValue.DIFF      => BasicValue.DIFF_INT
		   | BasicValue.PROD      => BasicValue.PROD_INT
		   | BasicValue.LESS      => BasicValue.LESS_INT
		   | BasicValue.GREATER   => BasicValue.GREATER_INT
		   | BasicValue.LESSEQ    => BasicValue.LESSEQ_INT
		   | BasicValue.GREATEREQ => BasicValue.GREATEREQ_INT
		   | _ => Crash.impossible "resolve_int"

               fun resolve_real (b: BasicValue.BasVal) = 
		 case b of
		     BasicValue.ABS       => BasicValue.ABS_REAL
		   | BasicValue.NEG       => BasicValue.NEG_REAL
		   | BasicValue.SUM       => BasicValue.SUM_REAL
		   | BasicValue.DIFF      => BasicValue.DIFF_REAL
		   | BasicValue.PROD      => BasicValue.PROD_REAL
		   | BasicValue.LESS      => BasicValue.LESS_REAL
		   | BasicValue.GREATER   => BasicValue.GREATER_REAL
		   | BasicValue.LESSEQ    => BasicValue.LESSEQ_REAL
		   | BasicValue.GREATEREQ => BasicValue.GREATEREQ_REAL
		   | _ => Crash.impossible "resolve_real"

	       fun resolve (b : BasicValue.BasVal) i =
		 if (b = BasicValue.PRIM) then b
		 else
		   (case (GrammarInfo.getPostElabOverloadingInfo i) of
		      None => b
		    | Some (OverloadingInfo.RESOLVED_INT) => 
			(resolve_int b)
		    | Some (OverloadingInfo.RESOLVED_REAL) => 
			(resolve_real b)
		    | Some (OverloadingInfo.UNRESOLVED _) => 
			Crash.impossible "evalAtexp(IDENTatexp)-2"
		   )
	    in 
	       case v of 
	         BASval b => BASval (resolve b i)
               | _ => v
	     end

	 | IDENTatexp(_, OP_OPT(LONGCON lcon, _)) =>
(* XXX not correct, must check that lcon is in the domain of E, 
   c.f. rule 106 *)
	       C.CON0val(#2(C.Con.decompose lcon))

	 | IDENTatexp(_, OP_OPT(LONGEXCON lexcon, _)) =>
(* XXX not correct, must check that lcon is in the domain of E, 
   c.f. rule 106 *)
	     C.EXNAME0val(C.lookup_LongExcon(E, lexcon))

	 | RECORDatexp(_, None)=> 
	     C.RECORDval SortedFinMap.empty

	 | RECORDatexp(_, Some exprow) =>
	     C.RECORDval(evalExprow(E, exprow))

	 | LETatexp(_, dec, exp) =>
	     let
	       val E' = evalDec(E, dec)
	     in
	       evalExp(C.E_plus_E(E, E'), exp)
	     end

	 | PARatexp(_, exp) => evalExp(E, exp)


   (****** expression rows - Definition v3 page 50 ******)

    and evalExprow(E, EXPROW(_, lab, exp, exprow_opt)) =
      let
	val v = evalExp(E, exp)
	open CoreDynObject
      in
	case exprow_opt
	  of None =>
	       SortedFinMap.singleton(lab, v)

	   | Some exprow' =>
	       SortedFinMap.add (Lab.<) (lab, v, evalExprow(E, exprow'))
      end


   (****** expressions - Definition v3 pages 50-51 ******)

    and evalExp(E, exp) =
      case exp
	of ATEXPexp(_, atexp) => evalAtexp(E, atexp)

         | APPexp(_, exp, atexp) =>
	     let
	       val v1 = evalExp(E, exp) 
	       val v  = evalAtexp(E, atexp)

	       open CoreDynObject
	     in
	       case v1
		 of BASval b => Apply.APPLY(b, v)

		  | CLOSUREval closure =>
		      let
			val (match, E', VE) = unClosure closure
			val E'' = E_plus_E(E', VE_in_E(Rec VE))
		      in		(* Evaluate the match, transforming
					   any FAIL into [Match] (Semantics
					   V4 #118). *)
			evalMatch(E'', v, match)
			handle FAIL => raiseMatch()
		      end

		  | CON0val con =>	(* "ref" a special case. *)
		      if con = Con.con_REF then
			let
			  val addr = Store.unique()
			  val _ = Store.add(addr, v)
			in
			  ADDRval addr
			end
		      else
			CON1val(con, v)

		  | EXNAME0val name => EXNAME1val(name, v)
		  | _ => Crash.impossible "evalExp(APPexp(?, arg))"
	     end
                        
	 | TYPEDexp(_, exp, ty) => evalExp(E, exp)
                             
	 | HANDLEexp(_, exp, match) =>
	     (evalExp(E, exp)
	      handle exn as C.EXCEPTION pack =>
		let			(* some exn got raised... *)
		  val v =
		    case pack
		      of C.EXNAME0pack name =>
			   C.EXNAME0val name
		       | C.EXNAME1pack(name, v) =>
			   C.EXNAME1val(name, v)
		in
		  evalMatch(E, v, match)
		  handle FAIL => raise exn	(* not matched here: resignal *)
		end
	     )

	 | RAISEexp(_, exp) =>
	     RAISE_val(evalExp(E, exp))

	 | FNexp(_, match) =>
	     C.CLOSUREval(C.mkClosure(match, E, C.emptyVE))

	 | UNRES_INFIXexp _ =>
	     Crash.impossible "evalExp(UNRES_INFIX)"

   (****** matches - Definition v3 page 52 ******)

    and evalMatch(E, v, match) =
      case match
	of MATCH(_, mrule, None) => evalMrule(E, v, mrule)
         | MATCH(_, mrule, Some match') =>
	     evalMrule(E, v, mrule)
	     handle FAIL => evalMatch(E, v, match')


   (****** match rules - Definition v3 page 52 ******)

    and evalMrule(E, v, mrule) =
      case mrule
	of MRULE(_, pat, exp) =>
	     let
	       val VE = evalPat(E, v, pat)
	     in
	       evalExp(C.E_plus_E(E, C.VE_in_E VE), exp)
	     end
	     handle FAIL => raise FAIL


   (****** declarations - Definition v3 page 52 ******)

    and evalDec(E, dec) =
      case dec
	of VALdec(_,valbind) => C.VE_in_E(evalValbind(E,valbind))
	 | UNRES_FUNdec _ => Crash.impossible "evalDec(UNRES_FUN)"
	 | TYPEdec _ => C.emptyE
	 | DATATYPEdec _ => C.emptyE
	 | ABSTYPEdec(_, _, dec) => evalDec(E, dec)

	 | EXCEPTIONdec(_, exbind) =>
	     C.EE_in_E(evalExbind(E, exbind))

         | LOCALdec(_, dec, dec') =>
	     let
	       val E1 = evalDec(E, dec)
	       val E2 = evalDec(C.E_plus_E(E, E1), dec')
	     in
	       E2
	     end

	 | OPENdec(_, list) =>
	     let
	       val Es =
		 map (fn WITH_INFO(_, id) => C.lookup_LongStrId(E, id)) list
	     in
	       List.foldL (General.curry C.E_plus_E) C.emptyE Es
	     end

	 | SEQdec(_, dec, dec') =>
	     let
	       val E1 = evalDec(E, dec)
	       val E2 = evalDec(C.E_plus_E(E, E1), dec')
	     in
	       C.E_plus_E(E1, E2)
	     end

	 | INFIXdec _  => C.emptyE
	 | INFIXRdec _ => C.emptyE
	 | NONFIXdec _ => C.emptyE
	 | EMPTYdec _  => C.emptyE

   (****** value bindings - Definition v3 page 53 ******)

    and evalValbind(E, valbind) =
      (case valbind
	 of PLAINvalbind(_, pat, exp, None) =>
	      let
		val v = evalExp(E, exp)
	      in
		evalPat(E, v, pat)	(* FAIL => [Bind] (Sem. V4 #136) *)
		handle FAIL => raiseBind()
	      end

          | PLAINvalbind(_, pat, exp, Some valbind) =>
	      let
		val v  = evalExp(E, exp)
		val VE = evalPat(E, v, pat) handle FAIL => raiseBind()
					(* FAIL => [Bind] (Sem. V4 #136) *)
	      in
		C.VE_plus_VE(VE, evalValbind(E, valbind))
	      end

	  | RECvalbind(_, valbind) => C.Rec(evalValbind(E, valbind))
     )
     handle FAIL => raise FAIL (* MEMO: incorrect (huh?) *)

   (* exbind's - Definition v4 rules 138/139 *)

    and evalExbind(E, exbind) =
      case exbind
	of EXBIND(_, OP_OPT(excon, _), _, exbind_opt) =>
	     let
	       val name =
		 C.ExName.new(C.Excon.pr_excon excon)

	       val EE_this = C.singleEE(excon, name)

	       val EE_rest =
		 case exbind_opt
		   of Some exbind => evalExbind(E, exbind)
		    | None => C.emptyEE
	     in
	       C.EE_plus_EE(EE_this, EE_rest)
	     end

	 | EXEQUAL(_, OP_OPT(excon, _),
		      OP_OPT(LONGEXCON eqexcon, _), exbind_opt
		  ) =>
	     let
	       val EE_this =
		 C.singleEE(excon, C.lookup_LongExcon(E, eqexcon))

	       val EE_rest =
		 case exbind_opt
		   of Some exbind => evalExbind(E, exbind)
		    | None => C.emptyEE
	     in
	       C.EE_plus_EE(EE_this, EE_rest)
	     end

	 | EXEQUAL _ =>
	     Crash.impossible "evalExbind EXEQUAL(_, ?, ?, _)"

   (****** atomic patterns - Definition v3 pages 53,54 ******)

    and evalAtpat(E, v, atpat) =
      case atpat
	of WILDCARDatpat _ => C.emptyVE

         | SCONatpat(_, scon) =>
	     if Apply.equal(v, C.Sval(C.mkSValSCon scon))
	     then C.emptyVE
	     else raise FAIL

	 | LONGIDatpat(_, OP_OPT(LONGVAR longvar, _)) =>
	     (case (C.Var.decompose longvar) of
		([], var) => C.singleVE(var, v)
	      | (_ , var) => Crash.impossible "evalAtpat(LONGIDatpat)")

	 | LONGIDatpat(_, OP_OPT(LONGCON longcon, _)) =>
	     let
	       open CoreDynObject
	     in
	       case v
		 of CON0val con' =>
		      if con' = #2(Con.decompose longcon)
		      then emptyVE else raise FAIL

		  | CON1val _ =>
		      raise FAIL

		  | _ =>		(* Typechecker ensures unreachable. *)
		      Crash.impossible "evalAtpat(LONGCON)"
	     end

	 | LONGIDatpat(_, OP_OPT(LONGEXCON longexcon, _)) =>
	     let
	       open CoreDynObject
	       val name = lookup_LongExcon(E, longexcon)
	     in
	       case v
		 of EXNAME0val name' =>
		      if name = name' then emptyVE else raise FAIL

		  | EXNAME1val _ =>
		      raise FAIL

		  | _ =>		(* Typechecking ensures unreachable. *)
		      Crash.impossible "evalAtpat(LONGEXCON)"
	     end

	 | RECORDatpat(_, None) =>
	     C.emptyVE

	 | RECORDatpat(_, Some patrow) =>
	     evalPatrow(E, v, patrow)
	     
	 | PARatpat(_,pat) => evalPat(E, v, pat)


   (****** pattern rows - Definition v3 page 54 ******)

    and evalPatrow(E, v, patrow) =
      case patrow
	of DOTDOTDOT _ => C.emptyVE

	 | PATROW(_, lab, pat, patrow_opt) =>
	     let
	       open CoreDynObject
	     in
	       case v
	         of RECORDval env =>
		      (case SortedFinMap.lookup env lab
			 of Some v' =>
			      C.VE_plus_VE(
				evalPat(E, v', pat),
				case patrow_opt
				  of Some patrow' => evalPatrow(E, v, patrow')
				   | None => C.emptyVE
			      )

			  | None =>
			      Crash.impossible "evalPatrow(lookup)"
		      )

		  | _ => Crash.impossible "evalPatrow"
	     end

   (****** patterns - Definition v3 pages 54,55 ******)

    and evalPat(E, v, pat) =
      case pat
	of ATPATpat(_, atpat) => evalAtpat(E, v, atpat)

         | CONSpat(_, OP_OPT(LONGCON longcon, _), atpat) =>
	     let
	       open CoreDynObject
	     in
	       case v
		 of CON1val(con', v') =>
		      if con' = #2(C.Con.decompose longcon)
		      then evalAtpat(E, v', atpat)
		      else raise FAIL

		  | CON0val _ =>
		      raise FAIL

		  | ADDRval addr =>	(* Guess who forgot about "ref"? *)
		      evalAtpat(E, Store.retrieve addr, atpat)

		  | _ =>		(* Typechecker ensures unreachable. *)
		      Crash.impossible("evalPat(LONGCON, \""
				       ^ PP.oneLiner layoutVal v
				       ^ "\")"
			  	      )
	     end

         | CONSpat(_, OP_OPT(LONGEXCON longexcon, _), atpat) =>
	     let
	       open CoreDynObject
	       val name = lookup_LongExcon(E, longexcon)
	     in
	       case v
		 of EXNAME1val(name', v') =>
		      if name = name' then evalAtpat(E, v', atpat)
				      else raise FAIL

		  | EXNAME0val _ =>
		      raise FAIL

		  | _ =>		(* Typechecker ensures unreachable. *)
		      Crash.impossible "evalPat(LONGEXCON)"
	     end

         | CONSpat(_, OP_OPT(LONGVAR longvar, _), atpat) =>

(***Normal code: *)

	     Crash.impossible "EvalDec.evalPat(CONSpat(VAR))"

(***KEVIN's modification:
	     let
	       open CoreDynObject

	       val v' =
		 (case lookup_LongVar(E, longvar)
		   of BASval b => Apply.APPLY(b, v)
		     
		    | CLOSUREval closure =>
			let
			  val (match, E', VE) = unClosure closure
			  val E'' = E_plus_E(E', VE_in_E(Rec VE))
			in
			  evalMatch(E'', v, match)
			end

		    | CON0val con => CON1val(con, v)
		    | EXNAME0val name => EXNAME1val(name, v)
		    | _ => Crash.impossible "evalPat(CONSpat)"
		 )

		 handle exn as C.EXCEPTION pack =>
		   raise FAIL
	     in
	       evalAtpat(E, v', atpat)
	     end
 ***)

         | TYPEDpat(_, pat, ty) => evalPat(E, v, pat)

	 | LAYEREDpat(_, OP_OPT(var, _), _, pat) =>
	     C.VE_plus_VE(C.singleVE(var, v), evalPat(E, v, pat))

	 | UNRES_INFIXpat _ =>
	     Crash.impossible "evalPat(UNRES_INFIX)"

   (* Uncaught packets get propagated using UNCAUGHT. We pass the packets out
      rather than pretty-printing them or something because we might have to
      pass the packet through various "use" contexts and have it caught
      lower down. *)

    type Pack = C.Pack

    fun pr_Pack p =
      case p
	of C.EXNAME0pack name =>
	     C.ExName.pr_ExName name
	 | C.EXNAME1pack(name, v) =>
	     C.ExName.pr_ExName name ^ "(" ^ PP.oneLiner C.layoutVal v ^ ")"

    exception UNCAUGHT of Pack

    val RE_RAISE = RAISE_pack		(* How to re-raise a packet from
					   the interpreter look. *)

    fun eval(E, dec) =
      evalDec(E, dec)
      handle C.EXCEPTION pack => raise UNCAUGHT pack

    type StringTree = C.StringTree
    val layoutEnv = C.layoutEnv
  end;
