/* v1.00 : Taskbar program installation */
SIGNAL ON FAILURE
SIGNAL ON ERROR

CALL RxFuncAdd "SysLoadFuncs","REXXUTIL","SysLoadFuncs"
CALL SysLoadFuncs

PARSE ARG destDir

IF destDir="" | destDir="/?" | TRANSLATE(destDir)="HELP" | TRANSLATE(destDir)="/H" THEN
    DO
        SAY "Usage: INSTALL destDir"
        SAY ""
        SAY "       destDir     = Directory location where you want"
        SAY "                     the Taskbar program installed."
        EXIT
    END

destDir = CreateDestDir(destDir)
currentDir = DIRECTORY()
if TRANSLATE(currentDir)\=TRANSLATE(destDir) THEN CALL CopyProgramFiles destDir
configChanged = 0
CALL UpdateLibpath destDir
CALL MakeObjects destDir
SAY "Program installation complete."
EXIT

/***** SUBROUTINES *****/

/*****************************************************************************
 * CreateDestDir    - procedure to create the destination directory.
 *                    Returns the fully qualified pathname of the
 *                    destination directory. Terminates the install script
 *                    if an error occurs.
 *****************************************************************************/
CreateDestDir: PROCEDURE
    PARSE ARG destDir

    CALL SETLOCAL
    target = DIRECTORY(destDir) /* check whether dest dir exists */
    IF target="" THEN /* dest dir doesn't exist so create it */
        DO
            IF 0\=SysMkDir(destDir) THEN
                DO
                    SAY "Error creating directory '"||destDir||"'"
                    EXIT
                END
        END
    destDir = DIRECTORY(destDir) /* get fully qualified pathname of dest dir */
    CALL ENDLOCAL
    RETURN destDir
/*****************************************************************************/

/*****************************************************************************
 * CopyProgramFiles - procedure to copy the program files to the
 *                    specified directory. Terminates the install
 *                    script if an error occurs.
 *****************************************************************************/
CopyProgramFiles: PROCEDURE
    PARSE ARG destDir

    filelist.0 = 3
    filelist.1 = "TaskBar.EXE"
    filelist.2 = "TskBarHk.DLL"
    filelist.3 = "MakeObjs.CMD"

    SAY "Copying program files to "||destDir||" ..."
    DO n=1 TO filelist.0
        '@COPY '||filelist.n||' '||destDir
        IF rc\=0 THEN
            DO
                SAY "Error copying "||filelist.n
                EXIT
            END
    END

    RETURN
/*****************************************************************************/

/*****************************************************************************
 * UpdateLibpath    - procedure to revise, if necessary, the Config.Sys
 *                    LIBPATH statement. Terminates the install script
 *                    if an error occurs.
 *****************************************************************************/
UpdateLibpath: PROCEDURE EXPOSE configChanged
    PARSE ARG destDir

    SAY "Checking/revising Config.Sys ..."

    '@BOOTDRIV'
    bootDrive = SUBSTR("ABCDEFGHIJKLMNOPQRSTUVWXYZ",rc,1)
    configSys = bootDrive||":\Config.Sys"

    IF STREAM(configSys,"C","QUERY EXISTS")="" THEN
        DO
            SAY "Error: Unable to find '"||configSys||"'."
            EXIT
        END

    tempfile = SysTempFileName(bootDrive||":\XX????.TMP")
    libpathFound = 0

    DO WHILE LINES(configSys) > 0
        line = LINEIN(configSys)
        sline = TRANSLATE(STRIP(line))
        PARSE VAR sline first rest
        first = TRANSLATE(STRIP(first))
        rest = TRANSLATE(STRIP(rest))
        IF LEFT(first,8) = "LIBPATH=" THEN
            DO
                libpathFound=1
                line = UpdateLine(line,destDir)
            END
        CALL LINEOUT tempfile,line
    END
    CALL STREAM configSys,'C','CLOSE'
    CALL STREAM tempfile,'C','CLOSE'

    IF libpathFound=0 THEN
        DO
            SAY "LIBPATH statement not found in Config.Sys."
            CALL SysFileDelete tempfile
            EXIT
        END

    IF configChanged = 1 THEN
        DO
            backupFile = SysTempFileName( bootDrive||":\Config.???" )
            IF backupFile = "" THEN
                DO
                    SAY "Unable to backup existing Config.Sys."
                    SAY "The updated Config.Sys is in file "||tempfile||"."
                    EXIT
                END
            '@COPY '||configSys||' '||backupFile
            IF rc\=0 THEN
                DO
                    SAY "Unable to backup existing Config.Sys."
                    SAY "The updated Config.Sys is in file "||tempfile||"."
                    EXIT
                END
            '@COPY '||tempfile||' '||configSys
            IF rc\=0 THEN
                DO
                    SAY "Unable to write the updated Config.Sys."
                    SAY "A backup copy of the original Config.Sys is in file "||backupFile||"."
                    SAY "The updated Config.Sys is in file "||tempfile||"."
                    EXIT
                END

            SAY "Your Config.Sys has been updated. Reboot before running Taskbar."
        END
    ELSE
        DO
            SAY "Your Config.Sys required no modification."
        END

    CALL SysFileDelete tempfile
    RETURN
/*****************************************************************************/

/*****************************************************************************
 * UpdateLine   - procedure to add the destination directory to the
 *                specified path list.
 *****************************************************************************/
UpdateLine: PROCEDURE EXPOSE configChanged
    PARSE ARG line destDir

    PARSE VAR line first"="rest
    rest = STRIP(rest,"t")
    restUpr = TRANSLATE(rest)
    pgmDir = TRANSLATE(destDir)
    pathFound = 0

    DO WHILE restUpr <> ""
        PARSE VAR restUpr thisPath";"restUpr
        IF thisPath=pgmDir | thisPath="." THEN
            DO
                pathFound = 1
                LEAVE
            END
    END

    IF pathFound=0 THEN
        DO
            rest = strip(rest,'t',';')||";"||destDir||";"
            configChanged = 1
            line = first"="rest
        END

    RETURN line
/*****************************************************************************/

/*****************************************************************************
 * MakeObjects  - procedure to create the necessary Workplace Shell objects.
 *                Terminates the install script if an error occurs.
 *****************************************************************************/
MakeObjects: PROCEDURE
    PARSE ARG destDir

    SAY "Creating/updating Taskbar program object."
    objectSetup = "OBJECTID=<Taskbar_RFY>;EXENAME="||destDir||"\Taskbar.EXE;STARTUPDIR="||destDir
    IF 0=SysCreateObject( "WPProgram","Taskbar","<WP_DESKTOP>",objectSetup,"update" ) THEN
        DO
            SAY "Unable to create program object."
            EXIT
        END

    SAY "Creating/updating shadow in startup folder."
    objectSetup = "OBJECTID=<Taskbar_RFY_Shadow>;SHADOWID=<Taskbar_RFY>"
    IF 0=SysCreateObject( "WPShadow","Taskbar","<WP_START>",objectSetup,"update" ) THEN
        DO
            SAY "Unable to create shadow object."
            EXIT
        END
    RETURN
/*****************************************************************************/

/*****************************************************************************/
 * Standard REXX program error handler
 *****************************************************************************/
ERROR:
    SAY "Error" rc "in system call."
    SAY "Line number =" sigl
    SAY "Instruction = "||SOURCELINE(sigl)
    EXIT

FAILURE:
    SAY "System failure detected (error code ="rc")"
    SAY "Line number =" sigl
    SAY "Instruction = "||SOURCELINE(sigl)
    EXIT
/*****************************************************************************/
