;;;
;;; Postgres/CommonLISP interface
;;;
;;; Copyright (c) 1986 Regents of the University of California
;;; 
;;; Permission to use, copy, modify, and distribute this software and its
;;; documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notice appear in all copies and
;;; that both that copyright notice and this permission notice appear in
;;; supporting documentation, and that the name of the University of
;;; California not be used in advertising or publicity pertaining to
;;; distribution of the software without specific, written prior
;;; permission.  The University of California makes no representations
;;; about the suitability of this software for any purpose.  It is
;;; provided "as is" without express or implied warranty.
;;; 
;;; $Author: picasso $
;;; $Source: RCS/dumpdata.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 90/07/23 13:17:58 $
;;;

(in-package 'libpq :use '(lisp excl))

(defvar ID-VAL "T")
(defvar BYTELEN 8)
(defvar BITMAP (make-array 100 :element-type '(unsigned-byte 8)))

(defvar G-VBLOCK (getvblock))
(defvar G-VBLOCK-PTR 0)

;;;
;;; Dump the data in a portal buffer
;;;
(defun dump-data (pname)
  "Dumps the data into the libpq portal buffer \fBpname\fP"
  ;; if the portal pname is not in the portal hash table, put it in
  (if (not (PQparray pname t))
      (portal-setup pname))
  (let* ((d-id (subseq ID-VAL 0 1))	;identifier
	 (pblock (pqparray pname))	;portal block
	 (pblock-ptr 2)			;portal block pointer
	 (main-block pblock)		;the main portal block for this portal
	 first-pblock			;the first portal block for this group
	 first-pblock-ptr		;the first portal block pointer
	 tblock				;type info block
	 iblock				;tuple index block
	 (ntype-info 0)			;the number of tuple groups
	 (ttuple 0)			;the number of total tuples
	 (ntuple 0)			;the number of tuples in a group
	 (nfields 0))			;the number of fields (attributes) in a tuple group
    (loop
     (cond ((string= d-id "T") 
	    (when (plusp ntype-info) 
		  (setf (aref first-pblock first-pblock-ptr) ntuple)
		  (setq ttuple (+ ttuple ntuple)))
	    (setq ntuple 0)
	    (setq ntype-info (1+ ntype-info))
	    (setq nfields (Cgetpint 2)) 
	    (when (= pblock-ptr (1- PBsize))
		  (setf (aref pblock pblock-ptr) (getpblock))
		  (setq pblock (aref pblock pblock-ptr))
		  (setq pblock-ptr 0))
	    (setq first-pblock pblock)
	    (setq first-pblock-ptr pblock-ptr)
	    (setq pblock-ptr (1+ pblock-ptr))
	    (when (= pblock-ptr (1- PBsize))
		  (setf (aref pblock pblock-ptr) (getpblock))
		  (setq pblock (aref pblock pblock-ptr))
		  (setq pblock-ptr 0))
	    (setf (aref pblock pblock-ptr) nfields)
	    (setq pblock-ptr (1+ pblock-ptr))
	    (setq tblock (gettblock))
	    (when (= pblock-ptr (1- PBsize))
		  (setf (aref pblock pblock-ptr) (getpblock))
		  (setq pblock (aref pblock pblock-ptr))
		  (setq pblock-ptr 0))
	    (setf (aref pblock pblock-ptr) tblock)
	    (setq pblock-ptr (1+ pblock-ptr))
	    (dump-type tblock nfields))
	   ((string= d-id "D") 
	    (setq ntuple (1+ ntuple)) 
	    (setq iblock (getiblock))
	    (when (= pblock-ptr (1- PBsize))
		  (setf (aref pblock pblock-ptr) (getpblock))
		  (setq pblock (aref pblock pblock-ptr))
		  (setq pblock-ptr 0))
	    (setf (aref pblock pblock-ptr) iblock)
	    (setq pblock-ptr (1+ pblock-ptr))
	    (dump-tuple iblock nfields))
	   ((string= d-id "A")
	    (d-end ntuple)
	    (setq ntuple 0)
	    (d-alerter))
	   ((string= d-id "C") 
	    (setf (aref first-pblock first-pblock-ptr) ntuple)
	    (setq ttuple (+ ttuple ntuple))
	    (setf (aref main-block 0) ttuple)
	    (setf (aref main-block 1) ntype-info)
	    (d-finish) 
	    (return t))
	   (t 
	    (error "cannot interpret the identifier from backend")))
     (if (eql (Cgetpid d-id 0 1) -1)
	 (error "end of transmission from backend")))))

;;;
;;; Dump the type information
;;;
(defun dump-type(tblock nfields)
  "Dumps the type information"
  (let ((tblock-ptr 0)
	(fname (make-string 16))
	(fnamelen 0))
    ;; read in all the type information
    (dotimes (fcnt nfields)
	     ;; if the current tblock is full
	     (when (> tblock-ptr (- TBsize 4))
		   ;; get another tblock and initialize it
		   (setf (aref tblock (1- TBsize)) (gettblock))
		   (setq tblock (aref tblock (1- TBsize)))
		   (setq tblock-ptr 0))
	     ;; read in field name
	     (setq fnamelen (Cgetpstr fname 16))
	     (setf (aref tblock tblock-ptr) (subseq fname 0 fnamelen))
	     ;; read adtid
	     (setf (aref tblock (incf tblock-ptr)) (Cgetpint 4))
	     ;; read adtsize
	     (setf (aref tblock (incf tblock-ptr)) (Cgetpint 2))
	     ;; increment type block pointer
	     (incf tblock-ptr))))

;;;
;;; Dump a tuple
;;;
(defun dump-tuple(iblock nfields)
  "Display a tuple. Read in the bitmap first and store it in BITMAP"
  ;; compute the number of fields
  (let ((nwords (ceiling (/ nfields BYTELEN))))
    ;; read in the tuple bitmap
    (Cgetpbitmap BITMAP 0 nwords)
    ;; allocate more locals
    (let* ((iblock-ptr 0)
	   (vblock G-VBLOCK)
	   (vblock-ptr G-VBLOCK-PTR)
	   ;(vblock (getvblock))
	   ;(vblock-ptr 0)
	   (value-list nil)
	   (fcnt 1)
	   (bcnt 0)
	   (bitcnt 7)
	   (bmap (aref BITMAP bcnt)))
      ;; loop over fields
      (loop
       ;; test if field count is > number of fields
       (cond ((> fcnt nfields)
	      (setq G-VBLOCK vblock)
	      (setq G-VBLOCK-PTR vblock-ptr)
	      (return))
	     ((> iblock-ptr (- IBsize 4))
	      (setf (aref iblock (1- IBsize)) (getiblock))
	      (setq iblock (aref iblock (1- IBsize)))
	      (setq iblock-ptr 0)))
       ;; test if bit is set
       (if (not (logbitp bitcnt bmap))
	   ;; set index block to nil
	   (setf (aref iblock iblock-ptr) nil)
	 ;; get the value length (first four bytes are for length)
	 (let ((vlen (- (Cgetpint 4) 4)))
	   ;; set index block value length
	   (setf (aref iblock (+ iblock-ptr 2)) vlen)
	   ;; test if value fits w/in value block
	   (cond ((<= vlen (- VBsize vblock-ptr))
		  ;; fits -- stick it in
		  (setf (aref iblock iblock-ptr) vblock)
		  (setf (aref iblock (1+ iblock-ptr)) vblock-ptr)
		  (Cgetpchar vblock vblock-ptr vlen)
		  (incf vblock-ptr vlen))
		 ((<= vlen VBsize)
		  ;; too big -- get another value block
		  (setq vblock (getvblock))
		  ;; reset value block pointer
		  (setq vblock-ptr 0)
		  (setf (aref iblock iblock-ptr) vblock)
		  (setf (aref iblock (1+ iblock-ptr)) vblock-ptr)
		  (Cgetpchar vblock vblock-ptr vlen)
		  (incf vblock-ptr vlen))
		 (t 
		  ;; very big -- get a number of value blocks, 
		  ;; and make them into a list.  The start point
		  ;; in the index block will be filled with a -1.
		  (setf (aref iblock (1+ iblock-ptr)) -1)
		  ;; loop until all the data are read in
		  (loop
		   (when (<= vlen VBsize)
			 (setq vblock (getvblock))
			 (Cgetpchar vblock 0 vlen)
			 (setq vblock-ptr vlen)
			 (setq value-list (cons vblock value-list))
			 (setf (aref iblock iblock-ptr) (reverse value-list))
			 (return))
		   (setq vblock (getvblock))
		   (setq value-list (cons vblock value-list))
		   (Cgetpchar vblock 0 VBsize)
		   (setq vlen (- vlen VBsize)))))))
       (incf iblock-ptr 3)
       (incf fcnt)
       (decf bitcnt)
       (when (minusp bitcnt)
	     (incf bcnt)
	     (setq bmap (aref BITMAP bcnt))
	     (setq bitcnt 7))))))

;;;
;;; End transmission for one command
;;;
(defun d-finish()
  "End transmission for one command"
  (let ((command (make-string 80)))
    (Cgetpint 4)
    (Cgetpstr command 80)))
