/*
 * select.c
 *
 * x-kernel v3.1	12/10/90
 *
 * Copyright (C) 1990  Larry L. Peterson and Norman C. Hutchinson
 */

#include "xkernel.h"
#include "ip.h"
#include "vchan.h"
#include "select.h"
#include "select_internal.h"


#define sameipaddr(A,B)  (((A)->a==(B)->a) && ((A)->b == (B)->b) && ((A)->c == (B)->c) && ((A)->d == (B)->d))


Map map_create();
static XObj real_open();
static int err_push();
static void phdr();
static void psaddr();
static void boost_concurrency();

int traceselectp=0;


select_instantiateprotl(self)
XObj self;
{
  self->state = malloc(sizeof(PSTATE));

  return(0);
}


select_init(self)
XObj self;
{
  Part 		part[3];
  XObj		IP;
  PSTATE 	*pstate;
    
  TRACE0(selectp, 1, "SELECT init");

  pstate = (PSTATE *)self->state;
  pstate->myCHANaddr.prot = SELECT_PROT_ID;
  IP = x_getprotlbyname("ip");
  x_controlprotl(IP, GETMYADDR, (char *)&pstate->myCHANaddr.host,
		 sizeof(IPhost));
  pstate->passive_map = map_create(100, sizeof(PassiveID));
  pstate->active_map = map_create(100, sizeof(ActiveID));

  init_partlist(part, 2, CHANaddr);
  set_part(part, 0, pstate->myCHANaddr);

  if (x_openenable(self, self->down[0], part) < 0) {
    TRACE0(selectp,0,"select_init: could not openenable lower protocol");
    return(-1);
  }
  return(0);
}


static XObj select_open(self,hlp, p)
XObj self;
XObj   hlp;
Part    *p;             /* p[0]=RPC address of server */
{
  XObj   	s;
  PSTATE 	*pstate;
  SELECT_STATE 	*state;
  Part 		part[3];
  CHANaddr 	remoteCHANaddr;
  RPCaddr 	remoteRPCaddr;
  ActiveID	key;

  TRACE0(selectp, 3, "SELECT open");

  pstate = (PSTATE *)self->state;
  s = (XObj) NULL;
  if (!p) {
    x_errno = BAD_ADDR;
    TRACE0(selectp, 1, "select_open: bad participant list");
    return(ERR_XOBJ);
  }
 
  /* Open a vchan session.  Part of the key for select sessions is the
   * lower level session, so we can't determine whether the select session
   * we want to open already exists until we open the lower session
   */
  remoteRPCaddr = get_part(p, 0, RPCaddr);
  remoteCHANaddr.host = remoteRPCaddr.host;
  remoteCHANaddr.prot = pstate->myCHANaddr.prot;
  init_partlist(part, 2, CHANaddr);
  set_part(part, 0, pstate->myCHANaddr);
  set_part(part, 1, remoteCHANaddr);
  key.under_s = x_open(self, self->down[0], part);
  if (key.under_s == ERR_XOBJ) {
      TRACE0(selectp, 1, "select_open: could not open under_session");
      return ERR_XOBJ;
  }

  /* Check to see if an appropriate vchan session already exists */
  key.command = remoteRPCaddr.command;
  s = (XObj)map_resolve(pstate->active_map, (char *)&key);
  if (s != ERR_XOBJ) {
      TRACE1(selectp, 3, "Active session %x already exists", s);
      s->rcnt++;
      /* Undo the open */
      x_close(key.under_s);  
      boost_concurrency(s, part);
      return s;
  }

  /* session did not exist -- get an uninitialized select session */
  TRACE0(selectp, 3, "No active session existed.  Creating new one.");
  s = real_open(self, hlp, p);
  if (s == ERR_XOBJ) {
      TRACE0(selectp,0,"select_open: real_open failed");
      return ERR_XOBJ;
  }
  state = (SELECT_STATE *)s->state;
  state->down_s = key.under_s;
  boost_concurrency(s, part);
  s->binding = (Bind)map_bind(pstate->active_map, (char *) &key, s);
  return s;
}



/* real_open creates a select session, but doesn't assign most of the state
 * variables.  State assignment should be done in the calling routine (open
 * or demux).
 */

static XObj
real_open(self, hlp, p)
  XObj   self;  		
  XObj   hlp;
  Part    *p;             /* p[0]=RPC address of server */
{
  XObj   s;
  SELECT_STATE *state;
  SELECT_HDR *hdr;
  RPCaddr server;

  TRACE0(selectp, 3, "real_open");

  s = (XObj) NULL;
  if (!p) {
    x_errno = BAD_ADDR;
    TRACE0(selectp,9,"error: 1");
    return(ERR_XOBJ);
  }
  server = get_part(p, 0, RPCaddr);

  state = (SELECT_STATE *) malloc(sizeof(SELECT_STATE));
  bzero(state,sizeof(SELECT_STATE));
  hdr = &state->hdr;

  state->server = server;

  IFTRACE(selectp,3) {
    printf("server:\n");
    psaddr(state->server);
  }

  hdr->server = server;
  hdr->status = SEL_OK;

 
  /* create session and bind to address */
  s = x_createsession(hlp, self,1);
  s->binding = 0;
  s->state = (char *) state;
  s->rcnt = 1;


  TRACE1(selectp, 3, "real_open returns %x", s);
  return(s);
}


select_openenable(self,hlp, p)
XObj   self;
XObj   hlp;
Part    *p;
			/* p[0] RPCaddress of server */
{
  PassiveID key;
  PSTATE *pstate;
  
  TRACE0(selectp, 3, "SELECT open enable");
  pstate = (PSTATE *)self->state;
  key = get_part(p, 0, RPCaddr);

  /* bind server to high level protocol */
  TRACE2(selectp, 3, "Binding key to openenable using key: host: %x  cmd: %d",
	 key.host, key.command);
  if ((map_bind(pstate->passive_map, (char *) &key, (int)hlp) == ERR_BIND)) {
      x_errno = ALREADY_OPEN;
      TRACE0(selectp,1,"select_openenable: could not bind hlp");
      return(-1);
  }

  return(0);
}


select_close(s)
    XObj   s;
{
    SELECT_STATE 	*state;
    int		n;
    
    TRACE1(selectp, 3, "select_close of session %x", s);

    assert(x_is_session(s));
    
    /* decrement refcount */  
    s->rcnt = s->rcnt - 1;
    
    if (s->rcnt > 0) {
	TRACE1(selectp,3,"select_close: reference count %d, not closing\n",
	       s->rcnt);
	return(0);
    } else {
	/* free select state */
	state = (SELECT_STATE *) s->state;
	if (state->down_s) {
	    /* Release extra channels which the vchan session acquired when
	     * this select session opened.
	     */
	    if (EXTRA_CHANNELS > 0) {
		n = EXTRA_CHANNELS;
		x_controlsessn(state->down_s, VCHAN_DECCONCURRENCY,
			       (char *)&n, sizeof(int));
	    }
	    x_close(state->down_s);
	}
	x_destroysession(s);
    }
    return 0;
}


select_push(s, msg, rmsg_ptr)
XObj   s;
Msg     msg;
Msg     *rmsg_ptr;
{
  Msg   	reply;
  SELECT_STATE  *state;
  SELECT_HDR 	*reply_hdr;
  SELECT_HDR 	*hdr_ptr;
 

  TRACE0(selectp, 3, "in select_push");

/* TMP */
  if (! rmsg_ptr) {
      TRACE0(selectp, 4, "select_push: incoming null msg pointer");
  }
/* End TMP */
  
  assert(msg.stack->ref.ref > 0);
  state = (SELECT_STATE *) s->state;
  state->hdr.status = SEL_OK;

  TRACE1(selectp, 4, "select_push: state = %d",state);
  IFTRACE(selectp,4) {
    phdr(&state->hdr);
  }
  TRACE1(selectp, 4, "select_push: outgoing msg length (no select hdr): %d",
	 msg_len(msg));

  hdr_ptr = (SELECT_HDR *)msg_push(msg,SELECTHLEN);
  *hdr_ptr = state->hdr;

  msg_clear(reply);
  TRACE1(selectp, 4, "select_push: down_s = %d",state->down_s);
  /* push message to lls */
  if (x_push(state->down_s, msg,&reply) >= 0) { 
    if (!msg_isnull(reply)) {
      reply_hdr = (SELECT_HDR *) msg_top(reply,SELECTHLEN);
      TRACE0(selectp, 4, "select_push: back from push with reply");
    } else { 
      TRACE0(selectp, 4, "select_push: back from push no reply");
     return(0);
    }
  } else {
    TRACE0(selectp, 4, "select_push: back from push error ");
    return(-1);
  }

  assert(reply.stack->ref.ref > 0);
  
  msg_pop(reply,SELECTHLEN);
  if (rmsg_ptr) {
      *rmsg_ptr = reply;
      TRACE1(selectp, 4, "select_push: return msg length (no select hdr): %d",
	     msg_len(*rmsg_ptr));
  } else {
      TRACE0(selectp, 4,
	     "select_push: null return messgage pointer, reply ignored");
  }
  return(0);
}


select_demux(self,s, dg)
  XObj	self;
  XObj  s;
  Msg	dg;
{
  SELECT_HDR 	*hdr;
  PassiveID 	passive_key;
  ActiveID 	active_key;
  SELECT_STATE 	*state;
  Part    	part[2];
  XObj   	select_s;
  XObj		hlp;
  PSTATE 	*pstate;


  TRACE0(selectp,3,"select_demux: called\n");
  assert(dg.stack->ref.ref > 0);

  pstate = (PSTATE *)self->state;
  hdr = (SELECT_HDR *) msg_top(dg,SELECTHLEN);
  
  IFTRACE(selectp,4) {
    phdr(hdr);
  }
 
  active_key.command = hdr->server.command;
  active_key.under_s = s;
  
  select_s = (XObj) map_resolve(pstate->active_map, (char *) &active_key);
  
  if (select_s != ERR_XOBJ) {
      TRACE1(selectp,5,"select_demux: active sessn = %x\n",select_s);
      x_pop(select_s, s, dg);
      return 0;
  }
  
  /* no active session exists -- check for openenable */
  
  passive_key = hdr->server;
  
  TRACE2(selectp, 5, "Checking for openenable using host: %x cmd: %d",
	 passive_key.host, passive_key.command);
  hlp = (XObj) map_resolve(pstate->passive_map, (char *) &passive_key);
  if (hlp == ERR_XOBJ) {
      TRACE0(selectp,4,"select_demux: no openenable done ");
      err_push(s,hdr);
      return(0);
  }
  init_partlist(part, 1, RPCaddr);
  set_part(part, 0, hdr->server);
  if ((select_s = real_open(self, hlp, part)) == ERR_XOBJ) {
      TRACE0(selectp,4,"select_demux: could not open new session ");
      err_push(s,hdr);
      return(0);
  }
  TRACE0(selectp, 4, "select_demux creates new server session");
  state = (SELECT_STATE *) select_s->state;
  state->down_s = s;
  select_s->binding = map_bind(pstate->active_map, (char *)&active_key,
			       select_s);
  x_pop(select_s,s,dg);
  return(0);
}


select_pop(s,lls, dg)
XObj   s;
XObj   lls;
Msg     dg;
{
  TRACE0(selectp, 3, "SELECT_pop");

  /* x_pop is only done for call messages received by server */

  msg_pop(dg,SELECTHLEN);
  TRACE1(selectp, 4, "select_pop: incoming msg length (no select hdr): %d",
	 msg_len(dg));
  return(x_demux(s,dg));
}


select_controlprotl(self,opcode, buf, len)
XObj	self;
int     opcode;
char    *buf;
int     len;
{
  switch (opcode) {
    default:
        x_errno = INVALID_OPCODE;
        return(-1);
   } 
}
  

select_controlsessn(s, opcode, buf, len)
XObj   s;
int     opcode;
char    *buf;
int     len;
{
    TRACE2(selectp, 3, "in select_control with session=%x, code = %d\n",
	   s, opcode); 

    assert(x_is_session(s));

    x_errno = INVALID_OPCODE;
    return(-1);
}


static void boost_concurrency(s, p)
    XObj	s;
    Part	*p;
{
    int			buf[2];
    SELECT_STATE	*state;

    assert(x_is_session(s));
    state = (SELECT_STATE *)s->state;
    if (EXTRA_CHANNELS > 0) {
	/* Get the vchan session to open some extra channel sessions.
	 * These will be (logically) removed when this session closes.
         */
	buf[0] = (int)p;
	buf[1] = EXTRA_CHANNELS;
	x_controlsessn(state->down_s, VCHAN_INCCONCURRENCY, buf,
		       2 * sizeof(int));
    }
}


/* push error message directly to lower-level session */
static int err_push(s, hdr)
XObj   s;
SELECT_HDR *hdr;
{
  Msg   msg;
  SELECT_HDR copy_hdr;

  TRACE0(selectp, 4, "in err_push");

  copy_hdr = *hdr;
  copy_hdr.status = SEL_FAIL;

  msg_make_allstack(msg,128,&copy_hdr,SELECTHLEN);

  return x_push(s, msg, 0);
}


static void phdr(hdr)
SELECT_HDR *hdr;
{
  printf("SELECT header:\n");
  psaddr(hdr->server);
  if (hdr->status == SEL_OK)
    printf("status = SEL_OK\n");
  else if (hdr->status == SEL_FAIL)  
    printf("status = SEL_FAIL\n");
  else 
    printf("status =INVALID \n");
}


static void psaddr(addr)
RPCaddr addr;
{
  printf("command: %d\n",addr.command);
  printf("host: %s\n",inet_ntoa(addr.host));
}



select_getproc(p,type)
XObj p;
XObjType type;
{
  if (type == Protocol) {
    p->instantiateprotl = select_instantiateprotl;
    p->init = select_init;
    p->close = noop;
    p->push = noop;
    p->pop = noop;
    p->control = select_controlprotl;
  } else {
    p->push = select_push;
    p->pop = select_pop;
    p->instantiateprotl = noop;
    p->init = noop;
    p->control = select_controlsessn;
    p->close = select_close;
  }
  p->open = (Pfi) select_open;
  p->openenable = select_openenable;
  p->opendone = noop;
  p->closedone = noop;
  p->opendisable = noop;
  p->demux = select_demux;
  p->getproc = select_getproc;
}

