/*
 * tcp_subr.c
 *
 * Derived from:
 *
 * Copyright (c) 1982, 1986 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of California at Berkeley. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 *
 *	@(#)tcp_subr.c	7.13 (Berkeley) 12/7/87
 *
 * Modified for x-kernel v3.1	12/10/90
 * Modifications Copyright (C) 1990  Larry L. Peterson and Norman C. Hutchinson
 */

#include "xkernel.h"
#include "ip.h"
#include "icmp.h"
#include "tcp_internal.h"
#include "tcp_fsm.h"
#include "tcp_seq.h"
#include "tcp_timer.h"
#include "tcp_var.h"
#include "tcpip.h"

int	tcp_ttl = TCP_TTL;

/*
 * Create template to be used to send tcp packets on a connection.
 * Call after host entry created, allocates an mbuf and fills
 * in a skeletal tcp/ip header, minimizing the amount of work
 * necessary when the connection is used.
 */
struct tcpiphdr *
tcp_template(tp)
	struct tcpcb *tp;
{
	register struct inpcb *inp = tp->t_inpcb;
	register struct tcpiphdr *n;

	if ((n = tp->t_template) == 0) {
		n = (struct tcpiphdr *)malloc(sizeof *n);
	}
	n->ti_next = n->ti_prev = 0;
	n->ti_x1 = 0;
	n->ti_pr = tcp_protocolnum;
	n->ti_len = htons(sizeof (struct tcpiphdr) - sizeof (struct ipovly));
	n->ti_src = inp->inp_laddr;
	n->ti_dst = inp->inp_raddr;
	n->ti_sport = inp->inp_lport;
	n->ti_dport = inp->inp_rport;
	n->ti_seq = 0;
	n->ti_ack = 0;
	n->ti_x2 = 0;
	n->ti_off = 5;
	n->ti_flags = 0;
	n->ti_win = 0;
	n->ti_sum = 0;
	n->ti_urp = 0;
	return (n);
}

/*
 * Send a single message to the TCP at address specified by
 * the given TCP/IP header.  If flags==0, then we make a copy
 * of the tcpiphdr at ti and send directly to the addressed host.
 * This is used to force keep alive messages out using the TCP
 * template for a connection tp->t_template.  If flags are given
 * then we send a message back to the TCP which originated the
 * segment ti, and discard the mbuf containing it and any other
 * attached mbufs.
 *
 * In any case the ack and sequence number of the transmitted
 * segment are as specified by the parameters.
 */
tcp_respond(tp, ti, m0, ack, seq, flags)
	struct tcpcb *tp;
	register struct tcpiphdr *ti;
	tcp_seq ack, seq;
	int flags;
	Msg *m0;
{
	int win = 0, tlen;
	struct tcpiphdr *nti;
	Msg m;
	XObj s = NULL;
	extern int tcp_recvspace;

	if (tp) {
		s = tp->t_inpcb->inp_session;
		win = tcp_recvspace;
		s = s->down[0];
	}
#ifdef TCP_COMPAT_42
	tlen = flags == 0;
#else
	tlen = 0;
#endif
	msg_make_allstack(m, 64 + sizeof (struct tcpiphdr)+tlen, 0, 0);
	nti = (struct tcpiphdr *) msg_push(m, sizeof(struct tcpiphdr)+tlen);
	*nti = *ti;
	ti = nti;
	if (flags == 0) {
		flags = TH_ACK;
	} else {
		msg_free(*m0);
#define xchg(a,b,type) { type t; t=a; a=b; b=t; }
		xchg(ti->ti_dst.s_addr, ti->ti_src.s_addr, u_long);
		xchg(ti->ti_dport, ti->ti_sport, u_short);
#undef xchg
	}
	ti->ti_next = ti->ti_prev = 0;
	ti->ti_x1 = 0;
	ti->ti_len = htons((u_short)(sizeof (struct tcphdr) + tlen));
	ti->ti_seq = htonl(seq);
	ti->ti_ack = htonl(ack);
	ti->ti_x2 = 0;
	ti->ti_off = sizeof (struct tcphdr) >> 2;
	ti->ti_flags = flags;
	ti->ti_win = htons((u_short)win);
	ti->ti_urp = 0;
	ti->ti_sum = in_cksum(m, sizeof (struct tcpiphdr) + tlen);
	msg_pop(m, sizeof(struct ipovly));
	if (s) {
	  x_push(s, m, (Msg *)0);
	} else {
	  Part p[3];
	  extern XObj TCP;
#ifndef 0
	  IPaddr here, there;
	  here.host = *(IPhost *)&ti->ti_src;
	  here.protocolnum = tcp_protocolnum;
	  there.host = *(IPhost *)&ti->ti_dst;
	  there.protocolnum = tcp_protocolnum;
	  p[0].address = (char *)&here;
	  p[0].length = sizeof here;
	  p[1].address = (char *)&there;
	  p[1].length = sizeof there;
#else
	  p[0].address = (char *)&tcp_protocolnum;
	  p[0].length = sizeof tcp_protocolnum;
	  p[1].address = (char *)&ti->ti_dst;
	  p[1].length = sizeof ti->ti_dst;
#endif
	  p[2].address = (char *)0;
	  p[2].length = 0;
	  s = x_open(TCP, TCP->down[0], p);
	  x_push(s, m, (Msg *) 0);
	  x_close(s);
	}
}

/*
 * Create a new TCP control block, making an
 * empty reassembly queue and hooking it to the argument
 * protocol control block.  Assumes that the ip control block and associated
 * tcp session already exist.
 */
struct tcpcb *
tcp_newtcpcb(inp)
	struct inpcb *inp;
{
	register struct tcpcb *tp;

	tp = (struct tcpcb *)malloc(sizeof *tp);
	bzero((char*)tp, sizeof *tp);
	tp->seg_next = tp->seg_prev = (struct reass *)tp;
	tp->t_maxseg = TCP_MSS;
	tp->t_flags = 0;		/* sends options! */
	tp->t_inpcb = inp;
	/*
	 * Init srtt to TCPTV_SRTTBASE (0), so we can tell that we have no
	 * rtt estimate.  Set rttvar so that srtt + 2 * rttvar gives
	 * reasonable initial retransmit time.
	 */
	tp->t_srtt = TCPTV_SRTTBASE;
	tp->t_rttvar = TCPTV_SRTTDFLT << 2;
	TCPT_RANGESET(tp->t_rxtcur, 
	    ((TCPTV_SRTTBASE >> 2) + (TCPTV_SRTTDFLT << 2)) >> 1,
	    TCPTV_MIN, TCPTV_REXMTMAX);
	tp->snd_cwnd = sbspace(sototcpst(inp->inp_session)->snd);
	tp->snd_ssthresh = 65535;		/* XXX */
	inp->inp_ppcb = (caddr_t)tp;
	return (tp);
}

/*
 * Drop a TCP connection, reporting
 * the specified error.  If connection is synchronized,
 * then send a RST to peer.
 */
struct tcpcb *
tcp_drop(tp, errno)
	register struct tcpcb *tp;
	int errno;
{
	XObj so = tp->t_inpcb->inp_session;

	TRACE2(tcpp, 2, "tcp_drop: tcpcb %X s %X", tp, so);
	if (TCPS_HAVERCVDSYN(tp->t_state)) {
	  tp->t_state = TCPS_CLOSED;
	  (void) tcp_output(tp);
	  tcpstat.tcps_drops++;
	} else {
	  tcpstat.tcps_conndrops++;
	}
	socantrcvmore(so);
#if 0
	return (tcp_destroy(tp));
#else
	return 0;
#endif
}

/*
 * Close a TCP control block:
 *	discard all space held by the tcp
 *	discard internet protocol block
 *	wake up any sleepers
 */
struct tcpcb *
tcp_destroy(tp)
	register struct tcpcb *tp;
{
	register struct reass *this, *next;
	struct inpcb *inp;
	XObj so;
	extern Map             tcp_map;

	TRACE1(tcpp, 1, "tcp_destroy: tcpcb %X", tp);
	inp = tp->t_inpcb;
	TRACE1(tcpp, 3, "tcp_destroy: inpcb %X", inp);
	so = inp->inp_session;
	this = tp->seg_next;
	while (this != (struct reass *)tp) {
		next = this->next;
		msg_free(this->m);
		remque(this);
		free((char *)this);
		this = next;
	}
	if (tp->t_template) free((char *)tp->t_template);
	free((char *)tp);
	inp->inp_ppcb = 0;
	/*
	 * This used to be a soisdisconnected, but really needs to delete
	 * all of the state.  
	 */
	delete_tcpstate(sototcpst(so));
	sototcpst(so) = 0;
	map_unbindbinding(tcp_map, so->binding);
	so->rcnt --;
	x_destroyxobj(so);

	in_pcbdetach(inp);
	tcpstat.tcps_closed++;
	return ((struct tcpcb *)0);
}

tcp_drain()
{

}

/*
 * Notify a tcp user of an asynchronous error;
 * just wake up so that he can collect error status.
 */
tcp_notify(inp)
	register struct inpcb *inp;
{

	sorwakeup(inp->inp_session);
	sowwakeup(inp->inp_session);
}

#if BSD<43
/* XXX fake routine */
tcp_abort(inp)
	struct inpcb *inp;
{
	return;
}
#endif

/*
 * When a source quench is received, close congestion window
 * to one segment.  We will gradually open it again as we proceed.
 */
tcp_quench(inp)
	struct inpcb *inp;
{
	struct tcpcb *tp = intotcpcb(inp);

	if (tp)
		tp->snd_cwnd = tp->t_maxseg;
}
