/*
 * tcp_x.c
 *
 * Derived from:
 *
 * Copyright (c) 1982, 1986 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of California at Berkeley. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 *
 * Modified for x-kernel v3.1	12/10/90
 * Modifications Copyright (C) 1990  Larry L. Peterson and Norman C. Hutchinson
 */

#include "xkernel.h"
#include "ip.h"
#include "tcp_internal.h"
#include "tcp_fsm.h"
#include "tcp_seq.h"
#include "tcp_timer.h"
#include "tcp_var.h"
#include "tcpip.h"


char *prurequests[] = {
	"ATTACH",	"DETACH",	"BIND",		"LISTEN",
	"CONNECT",	"ACCEPT",	"DISCONNECT",	"SHUTDOWN",
	"RCVD",		"SEND",		"ABORT",	"CONTROL",
	"SENSE",	"RCVOOB",	"SENDOOB",	"SOCKADDR",
	"PEERADDR",	"CONNECT2",	"FASTTIMO",	"SLOWTIMO",
	"PROTORCV",	"PROTOSEND",
};

Map             tcp_map;
u_char          tcp_protocolnum = 6;
XObj            TCP;
#define         IP (TCP->down[0])

static IPhost myipaddr;
static IPaddr	myaddr;  /* IPaddr is <host, protocolnum> pair */

/*************************************************************************/

tcp_instantiateprotl(self)
XObj    self;

{
PSTATE  *pstate;

  TRACE0(tcpp, 1, "TCP instantiateprotl");
  assert(x_is_protocol(self));

/* create pstate and save it */
  pstate = (PSTATE *)malloc(sizeof(PSTATE));
  self->state = (char *)pstate;
}

/*************************************************************************/

tcp_init(self)
XObj            self;
{
  extern int tcp_slowtimo(), tcp_fasttimo();
  Part		part[2];

  TCP = self;
  TRACE0(tcpp, 1, "TCP init");
  myaddr.protocolnum = tcp_protocolnum;
  myaddr.host = myipaddr;
  init_partlist(part, 1, IPaddr);
  set_part(part, 0, myaddr);
  tcb.inp_next = tcb.inp_prev = &tcb;
  tcp_iss = 1;
  x_openenable(TCP, IP, part);
  x_controlprotl(IP, GETMYADDR, (char *)&myipaddr, sizeof myipaddr);
  event_register(tcp_fasttimo, 0, 200, EV_REPEAT | EV_CREATEPROCESS);
  event_register(tcp_slowtimo, 0, 500, EV_REPEAT | EV_CREATEPROCESS);
  tcp_map = map_create(101,8);
  return (0);
}

/*************************************************************************/
struct tcpstate *new_tcpstate()
{
  struct tcpstate *tcpstate;
  tcpstate = (struct tcpstate *) malloc(sizeof(struct tcpstate));
  bzero((char *)tcpstate, sizeof(struct tcpstate));
  tcpstate->waiting = (caddr_t)malloc(sizeof(Semaphore));
  InitSemaphore((Semaphore *)tcpstate->waiting, 0);
  tcpstate->lock = (caddr_t)malloc(sizeof(Semaphore));
  InitSemaphore((Semaphore *)tcpstate->lock, 1);
  tcpstate->closed = 0;
  tcpstate->snd = (struct sb *)malloc(sizeof(struct sb));
  tcpstate->rcv = (struct sb *)malloc(sizeof(struct sb));
  return tcpstate;
}

/*************************************************************************/
void delete_tcpstate(tcpstate)
struct tcpstate *tcpstate;
{
  VAll((Semaphore *)tcpstate->waiting);
  free((char *)tcpstate->waiting);
  free((char *)tcpstate);

  /* From RICE 07/03/90 */
  VAll((Semaphore *)tcpstate->lock);
  free((char *)tcpstate->lock);
  free((char *)tcpstate->rcv);
  free((char *)tcpstate->snd);
  free((char *)tcpstate);
}

/*************************************************************************/
XObj	tcp_establishopen(hlp, raddr, rport, lport)
XObj hlp;
IPhost raddr;
unsigned short rport, lport;
{
  Part            part[3];
  XObj		 new;
  struct tcpstate *tcpst;
  struct inpcb *inp;
  TCP_EXID        ex_id;
  IPaddr	r_ipaddr;

  TRACE1(tcpp, 3, "tcp_establish: on %X", hlp);
  assert(x_is_protocol(hlp));

  ex_id.localport = lport;
  ex_id.remoteport = rport;
  ex_id.remoteaddr = raddr;
  if ((new = (Sessn) map_resolve(tcp_map, (char *) &ex_id)) != ERR_SESSN) {
    TRACE3(tcpp, 2, "tcp_establish: (%d->%X.%d) already open", ex_id.localport,
      ex_id.remoteaddr, ex_id.remoteport);
    x_errno = ALREADY_OPEN;
    return 0;
  }

  new = x_createsession(hlp, TCP, 1);
  TRACE3(tcpp, 3, "Mapping %d->%X.%d", lport, raddr, rport);
  new->binding = map_bind(tcp_map, (char *) &ex_id, (int) new);
  if ((int)new->binding == -1) {
    TRACE3(tcpp, 1, "tcp_establish: Bind of %d->%X.%d failed", 
      lport, raddr, rport);
    return 0;
  }

  init_partlist(part, 2, IPaddr);
  set_part(part, 0, myaddr);
  r_ipaddr.protocolnum = myaddr.protocolnum;
  r_ipaddr.host = raddr;
  set_part(part, 1, r_ipaddr);
  TRACE0(tcpp, 4, "tcp_open: opening IP");
  if ((new->down[0] = x_open(TCP, IP, part)) == ERR_SESSN) {
    TRACE1(tcpp, 2, "tcp_open: cannot open IP session (x_errno=%)", x_errno);
    x_destroysession(new);
    return 0;
  }

  tcpst = new_tcpstate();
  new->state = (char *)tcpst;
  TRACE0(tcpp, 4, "tcp_open: attaching...");
  tcp_attach(new);

  inp = sotoinpcb(new);
  tcpst->inpcb = inp;
  tcpst->tcpcb = (struct tcpcb *)inp->inp_ppcb;
  *(IPhost *)&inp->inp_laddr = myipaddr;
  inp->inp_lport = ex_id.localport;
  *(IPhost *)&inp->inp_raddr = ex_id.remoteaddr;
  inp->inp_rport = ex_id.remoteport;

  return new;
}

/*************************************************************************/
XObj            tcp_open(self, hlp, p)
    XObj            self, hlp;
    Part           *p;
{
  XObj            s;
  TCPaddr tcpaddr0, tcpaddr1;

  TRACE0(tcpp, 3, "TCP open");
  if (!p) {
    x_errno = BAD_ADDR;
    s = ERR_SESSN;
    return s;
  }
/* get_part(...) expands to (*(TCPaddr *)(p)[0].address) cliff 070590 */
  tcpaddr0 = get_part(p, 0, TCPaddr);
  tcpaddr1 = get_part(p, 1, TCPaddr);

  s = tcp_establishopen(hlp, tcpaddr1.host, tcpaddr1.port, tcpaddr0.port);
  if (!s) {
    s = ERR_SESSN;
  } else {
    TRACE0(tcpp, 4, "tcp_open: connecting...");
    tcp_usrreq(s, PRU_CONNECT, 0, 0);

    TRACE0(tcpp, 4, "tcp_open: waiting...");
    P((Semaphore *)sototcpst(s)->waiting);
    if (sototcpcb(s)->t_state != TCPS_ESTABLISHED) {
      TRACE3(tcpp, 4, "tcp_open: open (%d->%X.%d) connect failed",
	tcpaddr0.port,  tcpaddr1.host, tcpaddr1.port);
      tcp_destroy(sototcpcb(s));
      s = ERR_SESSN;
    }
  }
  TRACE1(tcpp, 1, "Return from tcp_open, session = %x", s);
  if (s == ERR_SESSN) TRACE1(tcpp, 1, "  x_errno = %d", x_errno);
  return s;
}

/*************************************************************************/
tcp_openenable(self, hlp, p)
XObj		self, hlp;
Part           *p;
{
  TCP_EXID        ex_id;
  TCPaddr         addr;

  addr = get_part(p, 0, TCPaddr);
  ex_id.localport = addr.port;
  ex_id.remoteport = (u_short) PADDING;
  *(unsigned long *)&ex_id.remoteaddr = PADDING;
  TRACE3(tcpp, 3, "Mapped %d->%X.%d", ex_id.localport, ex_id.remoteaddr, ex_id.remoteport);
  if (map_bind(tcp_map, (char *) &ex_id, (int) hlp) == ERR_BIND) {
    x_errno = ALREADY_OPEN;
    return -1;
  }
  return 0;
}

/*************************************************************************/
/*
 * User issued close, and wish to trail through shutdown states:
 * if never received SYN, just forget it.  If got a SYN from peer,
 * but haven't sent FIN, then go to FIN_WAIT_1 state to send peer a FIN.
 * If already got a FIN from peer, then almost done; go to LAST_ACK
 * state.  In all other cases, have already sent FIN to peer (e.g.
 * after PRU_SHUTDOWN), and just have to play tedious game waiting
 * for peer to send FIN or not respond to keep-alives, etc.
 * We can let the user exit from the close as soon as the FIN is acked.
 */
tcp_close(so)
    XObj            so;
{
  register struct tcpcb *tp;
  register struct tcpstate *tcpst;

  TRACE0(tcpp, 3, "in tcp close");

  tcpst = sototcpst(so);
  tcpst->closed |= 1;

  tp = sototcpcb(so);
  tp = tcp_usrclosed(tp);
  if (tp) tcp_output(tp);

  if (tcpst->closed == 1) {
    /*
     * This was a user close before the network close
     */
    TRACE0(tcpp, 4, "tcp_close: waiting...");
    P((Semaphore*)tcpst->waiting);
    TRACE0(tcpp, 4, "tcp_close: done");
  }
}

/*************************************************************************/
tcp_push(so, msg, rmsgptr)
XObj so;
Msg msg, *rmsgptr;
{
  int error;
  int space;
  register struct tcpstate *tcpst = sototcpst(so);
  TRACE2(tcpp, 3, "tcp_push: session %X, msg %d bytes", so, msg_len(msg));

  if (so->rcnt != 1) {
    TRACE1(tcpp, 2, "tcp_push: session ref count = %d", so->rcnt);
    msg_free(msg);
    return -1;
  }
#define MALLOCBOGUS(X) ((int)(X) < KERNEL_MALLOC || (int)(X) > KERNEL_MALLOC_LIMIT)
  if (MALLOCBOGUS(tcpst)) {
    TRACE1(tcpp, 2, "tcp_push: bogus tcpst = %X", tcpst);
    msg_free(msg);
    return -1;
  }

  if (tcpst->closed) {
    TRACE1(tcpp, 2, "tcp_push: session already closed %d", tcpst->closed);
    msg_free(msg);
    return -1;
  }
  
#undef CHUNKSIZE (sbhiwat(tcpst->snd) / 2)
#define CHUNKSIZE 1024

  while (! msg_isnull(msg)) {
    Msg pushmsg, rest;
    TRACE2(tcpp, 6, "tcp_push: msg = <%d, %d>", msg_len(msg),
	   msg.stack->ref.ref);
/*    msg_display(msg, 7); */
    if (msg_len(msg) <= CHUNKSIZE) {
      pushmsg = msg;
      msg = NULL_MSG;
    } else {
      msg_break(msg, rest, CHUNKSIZE, 0);
      TRACE4(tcpp, 6, "tcp_push: after break msg = <%d, %d> rest = <%d, %d>",
	     msg_len(msg), msg.stack->ref.ref,
	     msg_len(rest), rest.stack->ref.ref);
/*    msg_display(msg, 7);
      msg_display(rest, 7); */
      pushmsg = msg;
      msg = rest;
    }
    while ((space = sbspace(tcpst->snd)) < msg_len(pushmsg)) {
      TRACE1(tcpp, 2, "tcp_push: waiting for space (%d available)", space);
      P((Semaphore*)tcpst->waiting);
      if (tcpst->closed) {
	TRACE1(tcpp, 2, "tcp_push: session already closed %d", tcpst->closed);
	msg_free(pushmsg);
	if (!msg_isnull(msg)) msg_free(msg);
	return -1;
      }
    }
    sbappend(tcpst->snd, pushmsg);
    error = tcp_output(sototcpcb(so));
    if (error) {
      TRACE1(tcpp, 2, "tcp_push failed with code %d", error);
      if (!msg_isnull(msg)) msg_free(msg);
      return -1;
    }
  }
  return 0;
}

/*************************************************************************/
tcp_controlsessn(s, opcode, buf, len)
    Sessn           s;
    char           *buf;
    int             opcode;
    int             len;
{
  struct tcpstate *ts;
  TCPaddr         *taddr;
  struct tcpcb   *tp;

  ts = (struct tcpstate *) s->state;
  tp = ts->tcpcb;
  switch (opcode) {
   case GETMYADDR:
     checkLen(len, sizeof(TCPaddr));
     taddr = (TCPaddr *)buf;
     if (x_controlsessn(s->down[0], GETMYADDR, (char *) &taddr->host, IPADLEN) < 0)
       return -1;
     taddr->port = tp->t_template->ti_sport;
     return sizeof(TCPaddr);
   case GETPEERADDR:
     checkLen(len, sizeof(TCPaddr));
     taddr = (TCPaddr *)buf;
     if (x_controlsessn(s->down[0], GETPEERADDR, (char *) &taddr->host, IPADLEN) < 0) 
       return -1;
     taddr->port = tp->t_template->ti_dport;
     return sizeof(TCPaddr);
   case TCP_PUSH:
     tp->t_force = 1;
     tcp_output(tp);
     tp->t_force = 0;
     return 0;
   case TCP_GETSTATEINFO:
     *(int *) buf = tp->t_state;
     return (sizeof(int));
   default:
     x_errno = INVALID_OPCODE;
     return (-1);
  }
}

/*************************************************************************/
tcp_controlprotl(self, opcode, buf, len)
    XObj	    self;
    int             opcode, len;
    char           *buf;
{
  extern void tcp_dumpstats();
  switch (opcode) {
   case TCP_DUMPSTATEINFO:
      tcp_dumpstats();
      return 0;
    default:
      x_errno = INVALID_OPCODE;
      return (-1);
  }
}

/*************************************************************************/
tcp_getproc(p,type)
  XObj p;
  XObjType type;
{
  extern int tcp_input();
  if (type == Protocol) {
    p->instantiateprotl = noop;
    p->init = tcp_init;
    p->close = noop;
    p->push = noop;
    p->pop = noop;
    p->control = tcp_controlprotl;
  } else {
    p->push = tcp_push;
    p->pop = noop;
    p->instantiateprotl = noop;
    p->init = noop;
    p->close = tcp_close;
    p->control = tcp_controlsessn;
  }
  p->open = (Pfi) tcp_open;
  p->openenable = tcp_openenable;
  p->opendone = noop;
  p->closedone = noop;
  p->opendisable = noop;
  p->demux = tcp_input;
  p->getproc = tcp_getproc;
}

/*************************************************************************/
sorwakeup(so)
XObj so;
{
  TRACE1(tcpp, 3, "tcp: sorwakeup on %X", so);
}
    
/*************************************************************************/
sowwakeup(so)
XObj so;
{
  TRACE1(tcpp, 3, "tcp: sowwakeup on %X", so);
  V((Semaphore *)sototcpst(so)->waiting);
}

/*************************************************************************/
/*ARGSUSED*/
soreserve(so, send, recv)
XObj so;
int send, recv;
{
  struct tcpstate *tcpst = sototcpst(so);
  sbinit(tcpst->snd);
  sbinit(tcpst->rcv);
  return 0;
}

/*************************************************************************/
socantsendmore(so)
XObj so;
{
  TRACE1(tcpp, 3, "tcp:  socantsendmore on %X", so);
}

/*************************************************************************/
soisdisconnected(so)
XObj so;
{
  TRACE1(tcpp, 3, "tcp:  soisdisconnected on %X", so);
  if (sototcpst(so)->closed) {
    /* It was already closed, either by the network or the user */
  } else {
    /* We used to: x_closedone(so); */
    x_callclosedone(so);
  }
  VAll((Semaphore *)sototcpst(so)->waiting);
}

/*************************************************************************/
soabort(so)
XObj so;
{
}

/*************************************************************************/
socantrcvmore(so)
XObj so;
{
  TRACE1(tcpp, 3, "tcp:  socantrcvmore on %X", so);

  sototcpst(so)->closed |= 2;
  if (sototcpst(so)->closed & 1) {
    /* do nothing, the user already knows that it is closed */
  } else {
    x_callclosedone(so);
  }
}

/*************************************************************************/
soisconnected(so)
XObj so;
{
  Semaphore *s = (Semaphore *)sototcpst(so)->waiting;
  TRACE1(tcpp, 3, "tcp:  soisconnected on %X", so);
  if (s->count < 0) {
    TRACE1(tcpp, 4, "tcp:  waking up opener on %X", so);
    V(s);
  } else {
    /*
     * I think that this is the time for a open done
     */
    Part part[3];
    TCPaddr here, there;
    struct inpcb *inp;

/*===================  from ../protocols/tcp/tcp_internal.h
struct tcpstate {
  struct inpcb *inpcb;
  struct tcpcb *tcpcb;
  caddr_t       waiting;
  caddr_t       lock;
  struct sb    *snd;
  int closed;
};

#define sotoinpcb(S) (((struct tcpstate *)((S)->state))->inpcb)

===========================================================*/

    inp = sotoinpcb(so);
    here.host = *(IPhost *)&inp->inp_laddr;
    here.port = inp->inp_lport;
    there.host = *(IPhost *)&inp->inp_raddr;
    there.port = inp->inp_rport;
    TRACE4(tcpp, 2, "Calling opendone local %X.%d remote %X.%d",
      here.host, here.port, there.host, there.port);
    part[0].address = (char *)&here;
    part[0].length = sizeof here;
    part[1].address = (char *)&there;
    part[1].length = sizeof there;
    part[2].address = 0;
    part[2].length = 0;
    TRACE1(tcpp, 4, "tcp: should do open done on %X", so);
    x_callopendone(so->up, so, part);
  }
}

/*************************************************************************/
soisconnecting(so)
XObj so;
{
  TRACE1(tcpp, 3, "tcp:  soisconnecting on %X", so);
}

/*************************************************************************/
soisdisconnecting(so)
XObj so;
{
  TRACE1(tcpp, 3, "tcp:  soisdisconnecting on %X", so);
}

/*************************************************************************/
XObj sonewconn(so, src, sport, dport)
XObj so;
struct in_addr src;
unsigned short sport, dport;
{
  XObj new;
  assert(x_is_protocol(so));
  TRACE1(tcpp, 3, "sonewconn on %X", so);
  new = tcp_establishopen(so, src, sport, dport);
  return new;
}

/*************************************************************************/
void tcp_dumpstats()
{
  printf("tcps_badsum %d\n", tcpstat.tcps_badsum);
  printf("tcps_badoff %d\n", tcpstat.tcps_badoff);
  printf("tcps_hdrops %d\n", tcpstat.tcps_hdrops);
  printf("tcps_badsegs %d\n", tcpstat.tcps_badsegs);
  printf("tcps_unack %d\n", tcpstat.tcps_unack);
  printf("connections initiated %d\n", tcpstat.tcps_connattempt);
  printf("connections accepted %d\n", tcpstat.tcps_accepts);
  printf("connections established %d\n", tcpstat.tcps_connects);
  printf("connections dropped %d\n", tcpstat.tcps_drops);
  printf("embryonic connections dropped %d\n", tcpstat.tcps_conndrops);
  printf("conn. closed (includes drops) %d\n", tcpstat.tcps_closed);
  printf("segs where we tried to get rtt %d\n", tcpstat.tcps_segstimed);
  printf("times we succeeded %d\n", tcpstat.tcps_rttupdated);
  printf("delayed acks sent %d\n", tcpstat.tcps_delack);
  printf("conn. dropped in rxmt timeout %d\n", tcpstat.tcps_timeoutdrop);
  printf("retransmit timeouts %d\n", tcpstat.tcps_rexmttimeo);
  printf("persist timeouts %d\n", tcpstat.tcps_persisttimeo);
  printf("keepalive timeouts %d\n", tcpstat.tcps_keeptimeo);
  printf("keepalive probes sent %d\n", tcpstat.tcps_keepprobe);
  printf("connections dropped in keepalive %d\n", tcpstat.tcps_keepdrops);
  printf("total packets sent %d\n", tcpstat.tcps_sndtotal);
  printf("data packets sent %d\n", tcpstat.tcps_sndpack);
  printf("data bytes sent %d\n", tcpstat.tcps_sndbyte);
  printf("data packets retransmitted %d\n", tcpstat.tcps_sndrexmitpack);
  printf("data bytes retransmitted %d\n", tcpstat.tcps_sndrexmitbyte);
  printf("ack-only packets sent %d\n", tcpstat.tcps_sndacks);
  printf("window probes sent %d\n", tcpstat.tcps_sndprobe);
  printf("packets sent with URG only %d\n", tcpstat.tcps_sndurg);
  printf("window update-only packets sent %d\n", tcpstat.tcps_sndwinup);
  printf("control (SYN|FIN|RST) packets sent %d\n", tcpstat.tcps_sndctrl);
  printf("total packets received %d\n", tcpstat.tcps_rcvtotal);
  printf("packets received in sequence %d\n", tcpstat.tcps_rcvpack);
  printf("bytes received in sequence %d\n", tcpstat.tcps_rcvbyte);
  printf("packets received with ccksum errs %d\n", tcpstat.tcps_rcvbadsum);
  printf("packets received with bad offset %d\n", tcpstat.tcps_rcvbadoff);
  printf("packets received too short %d\n", tcpstat.tcps_rcvshort);
  printf("duplicate-only packets received %d\n", tcpstat.tcps_rcvduppack);
  printf("duplicate-only bytes received %d\n", tcpstat.tcps_rcvdupbyte);
  printf("packets with some duplicate data %d\n", tcpstat.tcps_rcvpartduppack);
  printf("dup. bytes in part-dup. packets %d\n", tcpstat.tcps_rcvpartdupbyte);
  printf("out-of-order packets received %d\n", tcpstat.tcps_rcvoopack);
  printf("out-of-order bytes received %d\n", tcpstat.tcps_rcvoobyte);
  printf("packets with data after window %d\n", tcpstat.tcps_rcvpackafterwin);
  printf("bytes rcvd after window %d\n", tcpstat.tcps_rcvbyteafterwin);
  printf("packets rcvd after \"close\" %d\n", tcpstat.tcps_rcvafterclose);
  printf("rcvd window probe packets %d\n", tcpstat.tcps_rcvwinprobe);
  printf("rcvd duplicate acks %d\n", tcpstat.tcps_rcvdupack);
  printf("rcvd acks for unsent data %d\n", tcpstat.tcps_rcvacktoomuch);
  printf("rcvd ack packets %d\n", tcpstat.tcps_rcvackpack);
  printf("bytes acked by rcvd acks %d\n", tcpstat.tcps_rcvackbyte);
  printf("rcvd window update packets %d\n", tcpstat.tcps_rcvwinupd);


  tcpstat.tcps_badsum = 0;
  tcpstat.tcps_badoff = 0;
  tcpstat.tcps_hdrops = 0;
  tcpstat.tcps_badsegs = 0;
  tcpstat.tcps_unack = 0;
  tcpstat.tcps_connattempt = 0;
  tcpstat.tcps_accepts = 0;
  tcpstat.tcps_connects = 0;
  tcpstat.tcps_drops = 0;
  tcpstat.tcps_conndrops = 0;
  tcpstat.tcps_closed = 0;
  tcpstat.tcps_segstimed = 0;
  tcpstat.tcps_rttupdated = 0;
  tcpstat.tcps_delack = 0;
  tcpstat.tcps_timeoutdrop = 0;
  tcpstat.tcps_rexmttimeo = 0;
  tcpstat.tcps_persisttimeo = 0;
  tcpstat.tcps_keeptimeo = 0;
  tcpstat.tcps_keepprobe = 0;
  tcpstat.tcps_keepdrops = 0;
  tcpstat.tcps_sndtotal = 0;
  tcpstat.tcps_sndpack = 0;
  tcpstat.tcps_sndbyte = 0;
  tcpstat.tcps_sndrexmitpack = 0;
  tcpstat.tcps_sndrexmitbyte = 0;
  tcpstat.tcps_sndacks = 0;
  tcpstat.tcps_sndprobe = 0;
  tcpstat.tcps_sndurg = 0;
  tcpstat.tcps_sndwinup = 0;
  tcpstat.tcps_sndctrl = 0;
  tcpstat.tcps_rcvtotal = 0;
  tcpstat.tcps_rcvpack = 0;
  tcpstat.tcps_rcvbyte = 0;
  tcpstat.tcps_rcvbadsum = 0;
  tcpstat.tcps_rcvbadoff = 0;
  tcpstat.tcps_rcvshort = 0;
  tcpstat.tcps_rcvduppack = 0;
  tcpstat.tcps_rcvdupbyte = 0;
  tcpstat.tcps_rcvpartduppack = 0;
  tcpstat.tcps_rcvpartdupbyte = 0;
  tcpstat.tcps_rcvoopack = 0;
  tcpstat.tcps_rcvoobyte = 0;
  tcpstat.tcps_rcvpackafterwin = 0;
  tcpstat.tcps_rcvbyteafterwin = 0;
  tcpstat.tcps_rcvafterclose = 0;
  tcpstat.tcps_rcvwinprobe = 0;
  tcpstat.tcps_rcvdupack = 0;
  tcpstat.tcps_rcvacktoomuch = 0;
  tcpstat.tcps_rcvackpack = 0;
  tcpstat.tcps_rcvackbyte = 0;
  tcpstat.tcps_rcvwinupd = 0;
}
