/*     
 * ether.c
 *
 * x-kernel v3.2
 *
 * Copyright (c) 1991  Arizona Board of Regents
 *
 *
 * $Revision: 1.23 $
 * $Date: 1992/02/12 01:17:41 $
 */

#include <sys/file.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netdb.h>
#include <syslog.h>
#include <errno.h>
#include <signal.h>

#include <sys/types.h>
#include <netinet/in.h>

#include <mach.h> 
#include <mach/message.h> 
#include <mach_privileged_ports.h> 
#include <cthreads.h> 
#include <device/net_status.h> 

#include "assert.h"
#include "upi.h"
#include "xk_debug.h"
#include "xkernel.h"
#include "eth.h"
#include "eth_i.h"

struct net_status ether_net_status;
#define ETHER_MAX_PACKET_SIZE 1600

/*
 * Set default filter priority. >100 means noone else gets the packet.
 */
#define FILTER_PRIORITY_DEFAULT 20

/*
 * Defining TEST_TYPE_FILTER will enable code which filters packets on
 * type before creating processes.  The types in the filter can only
 * be configured once at boot time.
 */
/* #define TEST_TYPE_FILTER */


/* global data of ethernet protocol */

int	traceethdrv = 0;	/* ethdrv == ethernet driver */


ETHhost	BCAST = BCAST_ETH_AD;
ETHhost	myetheraddr;

static	bool	initialized = FALSE;
static  int	ethFilterPriority = FILTER_PRIORITY_DEFAULT;
static  char	errBuf[100];
static  Map	typeMap;

void	init_ether();
void 	etherInit();
int 	ether2demux();
void 	eth_demux_stub(char *);

/* preallocated buffers */
#define MAX_ETH_BLOCKS 32

 /*
 * Packet filter declarations.
 */
struct mach_hdrs {
  mach_msg_header_t msg_hdr;
  mach_msg_type_t header_type;
  char header[NET_HDW_HDR_MAX];
  mach_msg_type_t packet_type;
  struct packet_header packet_header;
};

/*
 * Global data of ethernet protocol.
 */

static mach_port_t interface_port;
static mach_port_t filter_port;


int ether2demux()
{
  kern_return_t kr;
  struct buf {
    struct mach_hdrs hdr;
    char data[1];
  } *buf;
	  int stack_size = (sizeof(struct mach_hdrs) +
		    ether_net_status.max_packet_size + 3) & ~0x3;
  cthread_set_name(cthread_self(),"ether_thread");

  while (1) {

    buf = (struct buf *)xMalloc(stack_size);
    
    buf->hdr.msg_hdr.msgh_size = stack_size;
    buf->hdr.msg_hdr.msgh_local_port = filter_port;

    kr = mach_msg_receive(&buf->hdr.msg_hdr);
    if (kr != RCV_SUCCESS) {
      printf("ether.ether2demux.mach_msg_receive() failed: %d\n",kr);
      continue;
    }

    xTrace4(ethdrv,TR_FUNCTIONAL_TRACE,
	    "ether in pkt: msg bits=%d size=%d kind=%d id=%d",
	    buf->hdr.msg_hdr.msgh_bits,
	    buf->hdr.msg_hdr.msgh_size,
	    buf->hdr.msg_hdr.msgh_kind,
	    buf->hdr.msg_hdr.msgh_id);
    xTrace2(ethdrv,TR_FUNCTIONAL_TRACE,
	    "ether in pkt: pkt length=%d type=%d",
	    buf->hdr.packet_header.length,
	    buf->hdr.packet_header.type);
    xTrace3(ethdrv,TR_FUNCTIONAL_TRACE,
	    "ether in pkt: dest 0x%04x 0x%04x 0x%04x",
	    ((ETHhdr *)buf->hdr.header)->dst.high,
	    ((ETHhdr *)buf->hdr.header)->dst.mid,
	    ((ETHhdr *)buf->hdr.header)->dst.low);
    xTrace3(ethdrv,TR_FUNCTIONAL_TRACE,
	    "ether in pkt: src 0x%04x 0x%04x 0x%04x",
	    ((ETHhdr *)buf->hdr.header)->src.high,
	    ((ETHhdr *)buf->hdr.header)->src.mid,
	    ((ETHhdr *)buf->hdr.header)->src.low);
    xTrace1(ethdrv,TR_FUNCTIONAL_TRACE,
	    "ether in pkt: type 0x%04x",
	    ((ETHhdr *)buf->hdr.header)->type);
    xTrace3(ethdrv,TR_FUNCTIONAL_TRACE,
	    "ether in pkt: hdr 0x%02x 0x%02x 0x%02x",
	    buf->hdr.header[0],buf->hdr.header[1],buf->hdr.header[2]);
    xTrace3(ethdrv,TR_FULL_TRACE,
	    "ether in pkt: hdr 0x%02x 0x%02x 0x%02x",
	    buf->hdr.header[3],buf->hdr.header[4],buf->hdr.header[5]);
    xTrace3(ethdrv,TR_FULL_TRACE,
	    "ether in pkt: hdr 0x%02x 0x%02x 0x%02x",
	    buf->hdr.header[6],buf->hdr.header[7],buf->hdr.header[8]);
    xTrace3(ethdrv,TR_FULL_TRACE,
	    "ether in pkt: hdr 0x%02x 0x%02x 0x%02x",
	    buf->hdr.header[9],buf->hdr.header[10],buf->hdr.header[11]);
    xTrace2(ethdrv,TR_FULL_TRACE,
	    "ether in pkt: hdr 0x%02x 0x%02x",
	    buf->hdr.header[12],buf->hdr.header[13]);
    xTrace4(ethdrv,TR_FUNCTIONAL_TRACE,
	    "ether in pkt: data 0x%08x 0x%08x 0x%08x 0x%08x",
	    ((int *)buf->data)[0],((int *)buf->data)[1],
	    ((int *)buf->data)[2],((int *)buf->data)[3]);
    xTrace4(ethdrv,TR_FUNCTIONAL_TRACE,
	    "ether in pkt: data 0x%08x 0x%08x 0x%08x 0x%08x",
	    ((int *)buf->data)[4],((int *)buf->data)[5],
	    ((int *)buf->data)[6],((int *)buf->data)[7]);
    xTrace4(ethdrv,TR_FUNCTIONAL_TRACE,
	    "ether in pkt: data 0x%08x 0x%08x 0x%08x 0x%08x",
	    ((int *)buf->data)[8],((int *)buf->data)[9],
	    ((int *)buf->data)[10],((int *)buf->data)[11]);
    xTrace4(ethdrv,TR_FUNCTIONAL_TRACE,
	    "ether in pkt: data 0x%08x 0x%08x 0x%08x 0x%08x",
	    ((int *)buf->data)[12],((int *)buf->data)[13],
	    ((int *)buf->data)[14],((int *)buf->data)[15]);

#ifdef TEST_TYPE_FILTER
    /*
     * Make sure the type is one that we accept
     */
    {
	ETHtype	type = ntohs(((ETHhdr *)buf->hdr.header)->type);

	if ( mapResolve(typeMap, &type) != -1 ) {
	    xTrace1(ethdrv, TR_EVENTS,
		    "eth type %x blocked by type filter", type);
	    continue;
	} else {
	    xTrace1(ethdrv, TR_MORE_EVENTS,
		   "eth type %x passes the type filter", type);
	}
    }
#endif

/*    kr = xInvokeShepherd(XK_SHEP_NONBLOCKING,
			 eth_demux_stub,
			 (char *)buf);
*/

/* Disallowed, use sledgehammer instead. */
    xTrace0(ethdrv,TR_EVENTS,"ether2demux: starting process for incoming packet."); 
/* use STD_PRIO + 1 to indicate an expendable process, i.e., we might drop the
   packet if we run out of resources
*/
    CreateProcess(eth_demux_stub,STD_PRIO+1,1,(char *)buf);  
      /* Sledgehammer process */
    
  }
}


/*
 * readHex -- read a number from 's' which may be either hex (prefixed
 * with an x as in x34b2) or decimal.
 *
 * non-zero on success, zero on failure
 */
static int
readHex( s, n )
    char *s;
    int *n;
{
    if ( *s == 'x' ) {
	return ( sscanf(s + 1, "%x", n) == 1 );
    } else {
	return ( sscanf(s, "%d", n) == 1 );
    }
}


static void
processRomFile()
{
    int i;

    for ( i=0; rom[i][0]; i++ ) {
	if ( ! strcmp(rom[i][0], "eth") ) {
	    if ( rom[i][1] ) {
		if ( ! strcmp(rom[i][1], "priority") ) {
		    if ( rom[i][2] &&
			 sscanf(rom[i][2], "%d", &ethFilterPriority) == 1 ) {
			continue;
		    }

		} else if ( ! strcmp(rom[i][1], "block") ) {
#ifdef TEST_TYPE_FILTER
		    int		n;
		    ETHtype	type;

		    if ( rom[i][2] && readHex(rom[i][2], &n) ) {
			type = n;
			mapBind(typeMap, (char *)&type, 0);
			xTrace1(ethdrv, TR_GROSS_EVENTS,
				"eth driver blocking type %x",
				type);
			continue;
		    }
#else
		    xTrace1(ethdrv, TR_GROSS_EVENTS,
			    "eth driver ignoring type filter on line %d",
			    i+1);
		    continue;
#endif
		}
	    xError(sprintf(errBuf, "ETH format error in line %d of rom file",
			   i + 1));
	    }
	}
    }
}

void
etherInit()
{
  kern_return_t kr;
  unsigned int if_stat_count;
  char if_addr[16];
  unsigned int if_addr_count;
  filter_t filter[40];
  int filter_index = 0;
  cthread_t th;
  
#ifdef TEST_TYPE_FILTER
  typeMap = mapCreate(11, sizeof(ETHtype));
#endif
  processRomFile();
  kr = device_open(mach_device_server_port(),0,
		   "se0",&interface_port); 
  if (kr != D_SUCCESS) {
    printf("ether.device_open() failed: %d\n",kr);
    exit(1);
  }
  xTrace0(ethdrv,TR_MAJOR_EVENTS,"ether.device_open() completed");
  
  if_stat_count = NET_STATUS_COUNT;
  kr = device_get_status(interface_port,NET_STATUS,
			 &ether_net_status,&if_stat_count);
  if (kr != D_SUCCESS) {
    (void) device_close(interface_port);
    printf(
	   "ether.device_get_status(NET_STATUS) failed: %d\n",kr);
    exit(1);
  }
  xTrace0(ethdrv,TR_MAJOR_EVENTS,"ether.device_get_status() returned D_SUCCESS");
  
  if (ether_net_status.header_format != HDR_ETHERNET) {
    printf("ether: unsupported device header format: %d\n",
	   ether_net_status.header_format);
    exit(1);
  }
  if (ether_net_status.max_packet_size > ETHER_MAX_PACKET_SIZE) {
    printf(
	   "ether: invalid device max packet size: %d (max %d)\n",
	   ether_net_status.max_packet_size,ETHER_MAX_PACKET_SIZE);
    exit(1);
  }
  if (ether_net_status.header_size != sizeof(ETHhdr)) {
    printf("ether: invalid device header size: %d\n",
	   ether_net_status.header_size);
    exit(1);
  }
  if (ether_net_status.address_size != sizeof(ETHhost)) {
    printf("ether: invalid device address size: %d\n",
	   ether_net_status.address_size);
    exit(1); 
  }
  xTrace0(ethdrv,TR_MAJOR_EVENTS,"ether.device_get_status(NET_STATUS) completed");
  if (ether_net_status.mapped_size) {
    printf("error: Ethernet mapped\n");
    exit(1);
  }
  
  if_addr_count = sizeof(if_addr)/sizeof(int);
  kr = device_get_status(interface_port,NET_ADDRESS,
			 (dev_status_t)if_addr,&if_addr_count);
  if (kr != D_SUCCESS) {
    (void) device_close(interface_port);
    printf(
	   "ether.device_get_status(NET_ADDRESS) failed: %d\n",kr);
    exit(1); 
  }
  
  xTrace1(ethdrv,TR_GROSS_EVENTS,"ether: interface address (1) %s",
	  ethHostStr((ETHhost *)if_addr));

  { /* htohl seems to be broken on this platform, I'll do it myself */
    char t0,t1;
    t0 = if_addr[0];
    t1 = if_addr[1];
    if_addr[0] = if_addr[3];
    if_addr[1] = if_addr[2];
    if_addr[3] = t0;
    if_addr[2] = t1;
    t0 = if_addr[4];
    t1 = if_addr[5];
    if_addr[4] = if_addr[7];
    if_addr[5] = if_addr[6];
    if_addr[7] = t0;
    if_addr[6] = t1;
  } 
  bcopy((char *)if_addr, (char *)&myetheraddr, sizeof(ETHhost));
  xTrace1(ethdrv,TR_GROSS_EVENTS,"ether: interface address %s", ethHostStr(&myetheraddr));
  xTrace3(ethdrv,TR_GROSS_EVENTS,"ether: interface address 0x%04x 0x%04x 0x%04x",
	  myetheraddr.high,myetheraddr.mid,myetheraddr.low);

  kr = mach_port_allocate(mach_task_self(),
			  MACH_PORT_RIGHT_RECEIVE, &filter_port);
  if (kr != KERN_SUCCESS) {
    (void) device_close(interface_port);
    printf("ether: cannot allocate filter port: %d\n",kr);
    exit(1);
  }
  xTrace0(ethdrv,TR_MAJOR_EVENTS,"ether.mach_port_allocate() completed");
  
  (void) mach_port_set_qlimit(mach_task_self(), filter_port,
			      MACH_PORT_QLIMIT_MAX);
  xTrace0(ethdrv,TR_MAJOR_EVENTS,"ether.mach_port_set_qlimit() completed");


  xTrace1(ethdrv, TR_GROSS_EVENTS, "Ethernet driver using packet filter priority %d",
	  ethFilterPriority);
  filter[filter_index++] = NETF_PUSHLIT;
  filter[filter_index++] = (filter_t) TRUE;
  kr = device_set_filter(interface_port,
			 filter_port,
			 MACH_MSG_TYPE_MAKE_SEND,
			 ethFilterPriority,
			 filter,
			 filter_index);
  if (kr != KERN_SUCCESS) {
    (void) device_close(interface_port);
    printf("ether.device_set_filter() failed: %d\n",kr);
    exit(1);
  }
  xTrace0(ethdrv,TR_MAJOR_EVENTS,"ether.device_set_filter completed");

/* Since ether2demux does not use any xkernel routines, this should be 
   allowed to remain a cthread, in spite of the sledgehammer concurrency
   control otherwise enforced in the xkernel. */

  th = cthread_fork(ether2demux,0); /* Creating receive thread */
  thread_priority(th,STD_PRIO-1,FALSE);
  cthread_detach(th);
}

extern mutex_t sledgehammer_concurrency_control;


int
write_ether(buffer,len)
    char *buffer;
    int len;
{
	kern_return_t	kr;

	xTrace0(ethdrv, TR_MAJOR_EVENTS, "write_ether enter");
	xTrace1(ethdrv, TR_MAJOR_EVENTS, "write_ether sending %d bytes", len);

	kr = device_write_request(interface_port,PORT_NULL,0,0,buffer,len);
	if (kr != KERN_SUCCESS) {
		printf("ether.device_write_request() failed: %d\n",kr);
	}

	xTrace0(ethdrv, TR_MAJOR_EVENTS, "write_ether exit");
	return 0;
}


/*
 * Parameter is useless for us.  We determine our own address rather
 * than being told about it.
 */
void
ethCtlrInit( ETHhost host )
{
    if ( ! initialized ) {
	etherInit();
    }
    initialized = TRUE;
}


void
getLocalEthHost( host )
    ETHhost *host;
{
    ethCtlrInit(*host);
    bcopy((char *)&myetheraddr, (char *)host, sizeof(ETHhost));
}




static void
ethMsgStore( hdr, netHdr, len, arg )
    VOID *hdr;
    char *netHdr;
    long len;
    VOID *arg;
{
    xTrace3(ethdrv, TR_EVENTS, "ethHdrStore:  src: %s  dst %s  type %x",
	    ethHostStr(&((ETHhdr *)hdr)->src),
	    ethHostStr(&((ETHhdr *)hdr)->dst), ((ETHhdr *)hdr)->type);
    xIfTrace(ethdrv, TR_FUNCTIONAL_TRACE) {
	int i;
	
	for (i=0; i < sizeof(ETHhdr); i++) {
	    printf("%x ", *(((u_char *)hdr) + i));
	}
	printf("\n");
    }
    bcopy(hdr, netHdr, sizeof(ETHhdr));
}



static long
ethMsgLoad( hdr, netHdr, len, arg )
    VOID *hdr;
    char *netHdr;
    long len;
    VOID *arg;
{
    xAssert(len == sizeof(ETHhdr));
    bcopy(netHdr, (char *)hdr, sizeof(ETHhdr));
    ((ETHhdr *)hdr)->type = ntohs(((ETHhdr *)netHdr)->type);
    xTrace3(ethdrv, TR_EVENTS, "ethHdrLoad:  src: %s  dst %s  type %x",
	    ethHostStr(&((ETHhdr *)hdr)->src),
	    ethHostStr(&((ETHhdr *)hdr)->dst), ((ETHhdr *)hdr)->type);
    return sizeof(ETHhdr);
}



void
ethCtlrXmit( msg, dst, type )
    Msg *msg;
    ETHhost *dst;
    ETHtype type;
{
    ETHhdr	hdr;

    xTrace1(ethdrv, TR_EVENTS, "ethCtlrXmit destination %s", ethHostStr(dst));
    hdr.src = myetheraddr;
    hdr.dst = *dst;
    hdr.type = htons(type);
    msgPush(msg, ethMsgStore, &hdr, sizeof(ETHhdr), NULL);
    push2writeether(msg);
}



void
eth_demux_stub( buffer )
    char *buffer;
{
    struct buf {
	struct mach_hdrs hdr;
	char data[1];
    } 	*buf;
    char 	*nbuf;
    int 	len;
    Msg 	msg;
    ETHhdr	ethHdr;
    
    xTrace0(ethdrv, TR_MAJOR_EVENTS, "eth_demux_stub");
    /* This ugly stuff is to get the ethernet header where it is
       supposed to be. Mach messages from the driver looks strange... */
    
    buf = (struct buf *)buffer;
    /*
     * len is size of ethernet packet not including header
     */
    len = buf->hdr.packet_header.length - sizeof(struct packet_header);
    xTrace1(ethdrv, TR_MAJOR_EVENTS, "eth_demux_stub: len == %d", len);
    xAssert(len >= 0);
    /*
     * nbuf points to start of data
     */
    nbuf = (char *)buf + sizeof(struct mach_hdrs);
    msgConstructBuffer(&msg, nbuf, len);
    /*
     * grab ethernet header from buffer
     */
    bcopy((char *)buf->hdr.header, (char *)&ethHdr, sizeof(ETHhdr));
    xFree(buffer);
    eth_demux(msg, ntohs(ethHdr.type), ethHdr.src, ethHdr.dst);
}

int
SetPromiscuous()
{
    /*
     * There's probably a way to do this ...
     */
    return -1;
}


static bool
msg2Buf( msgPtr, len, bufPtr )
    char *msgPtr;
    long len;
    VOID *bufPtr;
{
  bcopy(msgPtr, *(char **)bufPtr, len);
  *(char **)bufPtr += len;
  return TRUE;
}


int
push2writeether( packet )
    Msg *packet;
{
  int	len;
  char	buffer[ETHER_MAX_PACKET_SIZE];
  char	*bufPtr = buffer;
  
  xTrace0(ethdrv, TR_MAJOR_EVENTS, "push2writeether");

  if ((len = msgLen(packet)) > ETHER_MAX_PACKET_SIZE) {
    return -1;
  }

  /* Place message contents in buffer */
  msgForEach(packet, msg2Buf, &bufPtr);
/*
printf("%x %x %x %x %x %x    %x %x ",buffer[0],buffer[1],buffer[2],
       buffer[3],buffer[4],buffer[5],buffer[6],buffer[7]);
printf("%x %x %x %x    %x %x    %s\n",buffer[8],buffer[9],buffer[10],
       buffer[11],buffer[12],buffer[13],&buffer[14]);
*/
  write_ether(buffer, len);
  return 0;
}

