/* 
 * upi.c
 *
 * x-kernel v3.2
 *
 * Copyright (c) 1991  Arizona Board of Regents
 *
 *
 * $Revision: 1.38 $
 * $Date: 1992/02/05 21:34:11 $
 */


#ifndef XKMACHKERNEL
#include <varargs.h>
#else
#include <sys/varargs.h>
#endif XKMACHKERNEL
#include "upi.h"
#include "xk_debug.h"
#include "process.h"
#include "assert.h"
#include "prottbl.h"
#include "platform.h"
#ifndef XKMACHKERNEL
#include "x_stdio.h"
#include "x_libc.h"
#endif XKMACHKERNEL

#ifndef NDEBUG
static char *controlops[] = {
  "getmyhost",
  "getmyhostcount",
  "getpeerhost",
  "getpeerhostcount",
  "getmaxpacket",
  "getoptpacket",
  "getmyproto",
  "getpeerproto",
  "resolve",
  "rresolve",
  "freeresources",
  "getparticipants"
};

#  define CONTROLMSG(n) ((n) < sizeof controlops / sizeof(char *) ? \
	controlops[n] : "unknown")
#else
#  define CONTROLMSG(n) ""
#endif

#ifdef __STDC__

extern char *	realloc( char *, unsigned );

static XObj xCreateXObj(int downc, XObj *downv);
static xkern_return_t xDestroyXObj(XObj s);

#else

extern char *	realloc();

static XObj xCreateXObj();
static xkern_return_t xDestroyXObj();

#endif __STDC__

#include "upi_inline.h"



/*************************************************
 * Uniform Protocol Interface Operations
 *************************************************/


XObj
xOpen(hlpRcv, hlpType, llp, participants)
    XObj hlpRcv, hlpType, llp;
    Part *participants;
{
    XObj s;
    
    xAssert(xIsProtocol(llp));
    xAssert(xIsProtocol(hlpType));
    xAssert(xIsProtocol(hlpRcv));
    xTrace3(protocol, 1, "Calling open[%s] by (%s,%s)",
	    llp->name, hlpRcv->name, hlpType->name);
    s = (XObj)(*(llp->open))(llp, hlpRcv, hlpType, participants);
    if (xIsSession(s)) {
	s->rcnt++;
    }
    xTrace5(protocol, TR_MAJOR_EVENTS,
	    "Open[%s] by (%s,%s) returns %x (rcnt == %d)",
	    llp->name, hlpRcv->name, hlpType->name, s,
	    xIsSession(s) ? s->rcnt : 0);
    return s;
}


xkern_return_t
xOpenEnable( hlpRcv, hlpType, llp, p )
    XObj hlpRcv, hlpType, llp;
    Part *p;
{
    xAssert(xIsProtocol(llp));
    xAssert(xIsProtocol(hlpType));
    xAssert(xIsProtocol(hlpRcv));
    xTrace3(protocol, TR_MAJOR_EVENTS, "Calling openenable[%s] by (%s,%s)",
	    llp->name, hlpRcv->name, hlpType->name);
    return (*(llp->openenable))(llp, hlpRcv, hlpType, p);
}


xkern_return_t
xOpenDone(s, hlpRcv)
    XObj s, hlpRcv;
{
    register Pfk od;
    
    xAssert(xIsSession(s));
    if (!(od = s->up->opendone)) return XK_SUCCESS;
    xTrace2(protocol, TR_MAJOR_EVENTS, "Calling opendone[%s] by %s",
	    s->up->name, s->myprotl->name);
    return (*od)(s->up, s, hlpRcv);
}


xkern_return_t
xCloseDone(s)
    XObj s;
{
  register Pfk cd;

  xAssert(xIsSession(s));
  if (!(cd = s->up->closedone)) return XK_SUCCESS;
  xTrace2(protocol, TR_MAJOR_EVENTS, "Calling closedone[%s] by %s", s->up->name,
	 s->myprotl->name);
  return (*cd)(s);
}


xkern_return_t
xOpenDisable( hlpRcv, hlpType, llp, participants )
    XObj hlpRcv, hlpType, llp;
    Part *participants;
{
    xAssert(xIsProtocol(llp));
    xAssert(xIsProtocol(hlpRcv));
    xAssert(xIsProtocol(hlpType));
    xTrace3(protocol, TR_MAJOR_EVENTS, "Calling opendisable[%s] by (%s,%s)",
	    llp->name, hlpRcv->name, hlpType->name);
    return (*(llp->opendisable))(llp, hlpRcv, hlpType, participants);
}


xkern_return_t
xClose(s)
    XObj s;
{
  xTrace0(protocol, TR_MAJOR_EVENTS, "xClose: entered");
  /*
   * DEC_REF_COUNT_UNCOND comes from upi_inline.h
   */
  DEC_REF_COUNT_UNCOND(s, "xClose");
  xTrace0(protocol, TR_FULL_TRACE, "xClose returning");
  return XK_SUCCESS;
}


xkern_return_t
xDuplicate(s)
    XObj s;
{
  xTrace1(protocol, TR_MAJOR_EVENTS, "calling duplicate[%s]", s->myprotl->name);
  return (*s->duplicate)(s);
}


static xkern_return_t
defaultDuplicate(s)
    XObj s;
{
    s->rcnt++;
    return XK_SUCCESS;
}


/*
 * xDemux, xCallDemux, xPush and xCall are defined as macros in upi.h when
 * optimized
 */

#if !defined(NDEBUG)

xkern_return_t
xCall(s, msg, replyMsg)
    XObj s;
    Msg *msg;
    Msg *replyMsg;
{
    xkern_return_t retVal;
    
    xAssert(xIsSession(s));
    xTrace3(protocol, TR_MAJOR_EVENTS, "Calling call[%s] by %s, %d bytes", s->myprotl->name,
	    s->up->name, msgLen(msg));
    xIfTrace(protocol, TR_FUNCTIONAL_TRACE) {
	xTrace0(protocol, TR_ALWAYS, "       Message:");
	msgShow(msg);
    }
    retVal = (*s->call)(s, msg, replyMsg);
    xTrace3(protocol, TR_MAJOR_EVENTS, "call[%s] returns %d bytes in reply to %s",
	    s->myprotl->name, msgLen(replyMsg), s->up->name);
    return retVal;
}


xmsg_handle_t
xPush(s, msg)
    XObj s;
    Msg *msg;
{
    int retVal;
    
    xAssert(xIsSession(s));
    xTrace3(protocol, TR_MAJOR_EVENTS, "Calling push[%s] by %s, %d bytes", s->myprotl->name,
	    s->up->name, msgLen(msg));
    xIfTrace(protocol, TR_FUNCTIONAL_TRACE) {
	xTrace0(protocol, TR_ALWAYS, "       Message:");
	msgShow(msg);
    }
    retVal = (*s->push)(s, msg);
    xTrace3(protocol, TR_MAJOR_EVENTS, "push[%s] by %s returns %d", s->myprotl->name,
	    s->up->name, retVal);
    return retVal;
}


xkern_return_t
xDemux(s, msg)
    XObj s;
    Msg *msg;
{
  register Pfk demux;

  xAssert(xIsSession(s));
  xTrace3(protocol, TR_MAJOR_EVENTS, "Calling demux[%s] by %s, %d bytes", s->up->name,
	 s->myprotl->name, msgLen(msg));
  xIfTrace(protocol, TR_FUNCTIONAL_TRACE) {
      xTrace0(protocol, TR_ALWAYS, "       Message:");
      msgShow(msg);
  }
  if (!(demux = s->up->demux)) return XK_SUCCESS;
  return (*demux)(s->up, s, msg);
}


xkern_return_t
xCallDemux(s, msg, replyMsg)
    XObj s;
    Msg *msg;
    Msg *replyMsg;
{
  register Pfk calldemux;
  xkern_return_t	retVal;

  xAssert(xIsSession(s));
  xTrace3(protocol, TR_MAJOR_EVENTS, "Calling calldemux[%s] by %s, %d bytes", s->up->name,
	 s->myprotl->name, msgLen(msg));
  xIfTrace(protocol, TR_FUNCTIONAL_TRACE) {
    xTrace0(protocol, TR_ALWAYS, "       Message:");
    msgShow(msg);
  }
  if (!(calldemux = s->up->calldemux)) return XK_SUCCESS;
  retVal = (*calldemux)(s->up, s, msg, replyMsg);
  xTrace2(protocol, TR_MAJOR_EVENTS, "calldemux[%s] returns %d bytes",
	  s->up->name, msgLen(replyMsg));
  return retVal;
}



#endif !NDEBUG



int
xControl(s, opcode, buf, len)
    XObj s;
    int opcode;
    char *buf;
    int len;
{
  register Pfi c;

  int res;
  if (! (c = s->control)) {
    return 0;
  }
  xTrace3(protocol, TR_MAJOR_EVENTS, "Calling control[%s] op %s (%d)",
	 s->myprotl->name, CONTROLMSG(opcode), opcode);
  res = c(s, opcode, buf, len);
  xTrace4(protocol, TR_MAJOR_EVENTS, "Control[%s] op %s (%d) returns %d",
	s->myprotl->name, CONTROLMSG(opcode), opcode, res);
  return res;
}




/*************************************************
 * Create and Destroy XObj's Sessions and Protocols 
 *************************************************/

static int
noop()
{
    return 0;
}


XObj
xCreateSessn(f, hlp, llp, downc, downv)
    Pfv f;
    XObj hlp;
    XObj llp;
    int downc;
    XObj *downv;
{
  XObj s;

  xTrace2(protocol, TR_MAJOR_EVENTS, "xCreateSession:[%s] by [%s]", llp->name, hlp->name);
  if ((s = xCreateXObj(downc, downv)) == ERR_XOBJ) {
    xTrace0(protocol, TR_ERRORS, "CreateSessn failed at CreateXObj");
    return s;
  }
  s->type = Session;
  s->rcnt = 0;
  s->name = llp->name;
  s->instance = llp->instance;
  s->myprotl = llp;
  s->id = 0;			/* only relevant to protocols */
  s->up = hlp;
  llp->rcnt++;
  hlp->rcnt++;
  if (f) (*f)(s);
  return s;
}


XObj
xCreateProtl(f, name, downc, downv)
    Pfv f;
    char *name;
    int downc;
    XObj *downv;
{
  XObj s;
  int  i;
  XObj *dv;
  int  id;

  xTrace1(protocol, TR_MAJOR_EVENTS, "xCreateProtocol:[%s]", name);
  if ( (id = protTblGetId(name)) == -1 ) {
    xTrace1(protocol, TR_ERRORS, "CreateProtl could not find protocol id for %s", name);
      return ERR_XOBJ;
  }
  if ((s = xCreateXObj(downc, downv)) == ERR_XOBJ) {
    xTrace0(protocol, TR_ERRORS, "CreateProtl failed at CreateXObj");
    return ERR_XOBJ;
  }
  s->type = Protocol;
  s->rcnt = 1;
  s->name = name;
  /* 
   * Should automatically determine an instance number for each protocol
   */
  s->instance = 0;
  s->myprotl = s;
  s->id = id;
  s->up = ERR_XOBJ;		/* only relevant to sessions */
  dv = s->down;
  for (i=0; i < downc; i++) {
    downv[i]->rcnt++;
  }
  if (f) (*f)(s);
  return s;
}
		     

static XObj
xCreateXObj(downc, downv)
    int downc;
    XObj *downv;
{
  XObj s;
  XObj *dv;
  int  i;

  s = (struct xobj *) xMalloc(sizeof(struct xobj));
  if (! s) {
    xTrace0(protocol, TR_ERRORS, "xCreateObj malloc failure(1)");
    return ERR_XOBJ;
  }
  bzero((char *)s, sizeof(struct xobj));
  s->numdown = downc;
  if (downc > STD_DOWN) {
    s->downlistsz = (((downc - STD_DOWN) / STD_DOWN) + 1) * STD_DOWN;
    s->downlist = (XObj *) xMalloc(s->downlistsz * sizeof(XObj));
    if (! s->downlist) {
      xFree((char *)s);
      xTrace0(protocol, TR_ERRORS, "xCreateObj malloc failure(2)");
      return ERR_XOBJ;
    }
  } else {
    s->downlistsz = 0;
  }
  dv = s->down;
  for (i=0; i < downc; i++, dv++) {
    if (i == STD_DOWN) {
      dv = s->downlist;
    }
    *dv = downv[i];
  }
  s->idle = FALSE;
  s->open = (Pfo)noop;
  s->close = (Pfk)noop;
  s->closedone = (Pfk)noop;
  s->openenable = (Pfk)noop;
  s->opendisable = (Pfk)noop;
  s->opendone = (Pfk)noop;
  s->demux = (Pfk)noop;
  s->calldemux = (Pfk)noop;
  s->callpop = (Pfk)noop;
  s->pop = (Pfk)noop;
  s->push = (Pfh)noop;
  s->call = (Pfk)noop;
  s->control = noop;
  s->duplicate = defaultDuplicate;
  return (XObj)s;
}


static xkern_return_t
xDestroyXObj(s)
    XObj s;
{
    xkern_return_t err = XK_SUCCESS;
    
    if (s->state) {
	err = xFree((char *)s->state) ? XK_FAILURE : XK_SUCCESS;
    }
    return xFree((char *)s) ? XK_FAILURE : err;
}


xkern_return_t
xDestroy(s)
    XObj s;
{
  if ((s->type == Session) && ((--(s->myprotl->rcnt)) < 1)) { 
    xClose(s->myprotl);
  } 
  return xDestroyXObj(s);
}



/*************************************************
 * Utility Routines
 *************************************************/


int
xIsSession(s)
    XObj s;
{
    return s && ( s != ERR_XOBJ ) && s->type == Session;
}


int
xIsProtocol(s)
    XObj s;
{
    return s && ( s != ERR_XOBJ ) && s->type == Protocol;
}


XObj
xGetProtlByName(name)
    char *name;
{
  register XObj p;
  register int i;

  for (i = 0; (p = protl_tab[i]); i++) {
    if (*p->name == *name && !strcmp(p->name, name)) return p;
  }
  return ERR_XOBJ;
}


xkern_return_t
xSetDown(s, i, obj)
    XObj s;
    int i;
    XObj obj;
{
  XObj *newdl;
  int  newsz;

  if (i < STD_DOWN) {
    s->down[i] = obj;
  } else {
    i -= STD_DOWN;
    if (i >= s->downlistsz) {
      newsz = ((i / STD_DOWN) + 1) * STD_DOWN;
      newdl = (XObj *)realloc((char *)s->downlist, newsz * sizeof(XObj));
      if (! newdl) {
	return XK_FAILURE;
      }
      s->downlist = newdl;
      s->downlistsz = newsz;
    }
    s->downlist[i] = obj;
  }
  if (i + 1 > s->numdown) {
    s->numdown = i + 1;
  }
  return XK_SUCCESS;
}


XObj
xGetDown(s, i)
    XObj s;
    int i;
{
  if (i + 1 > s->numdown) {
    return ERR_XOBJ;
  }
  return (i < STD_DOWN) ? s->down[i] : s->downlist[i - STD_DOWN];
}



/*************************************************
 * Miscellaneous Routines
 *************************************************/


void
xPrintXObj(p)
    XObj p;
{
  int i;
  
  if (!p) printf("\nXOBJ NULL \n");
  printf("\nXOBJ %x %s \n", p, p->name);
  printf("type = %s\n", p->type == Protocol ? "Protocol" : "Session");
  printf("myprotocol = %x %s\n", p->myprotl, p->myprotl->name);
  printf("id = %d instance = %d\n", p->id, p->instance);
  printf("rcnt = %d \n",p->rcnt);
  printf("up = %x ",p->up);
  if (p->up) {
    printf("up->type = %s ", p->type == Protocol ? "Protocol" : "Session");
    printf("up->name = %s\n",p->up->name);
  } else {
    printf("\n");
  }
  printf("numdown = %d ",p->numdown);
  for (i=0;i<p->numdown;i++) {
    if (p->down[i]) {
      printf(" %d = %s ",i,p->down[i]->name);
    }
  }
  printf("\n");
}

