/* 
 * arp.c
 *
 * x-kernel v3.2
 *
 * Copyright (c) 1991  Arizona Board of Regents
 *
 *
 * $Revision: 1.55 $
 * $Date: 1992/02/07 04:34:02 $
 */


#include "xkernel.h"
#include "eth.h"
#include "ip.h"
#include "arp.h"
#include "arp_i.h"
#include "arp_table.h"

int tracearpp;

#ifndef NDEBUG

static char *codenames[] = {
  "impossible",
  "req",
  "rply",
  "rreq",
  "rrply"
};

#endif


/* global data of arp protocol */

static ETHhost	bcastEthHost = BCAST_ETH_AD;
static ARPprotlstate pstate;
static XObj 	ARP=0;


#ifdef __STDC__

static int	arpControlProtl( XObj, int, char *, int );
static xkern_return_t
  		arpDemux( XObj, XObj, Msg * );
static void	arpHdrStore(void *, char *, long, void *);
static long	arpHdrLoad(void *, char *, long, void *);
static void 	arpInterfaceInit( XObj, XObj , Interface * );
static Interface *isMyIp( IPhost * );
static void 	sendArpReply( Interface *, XObj, arppak * );
#ifdef ANSWER_RARPS 
static void 	sendRarpReply( XObj, arppak * );
#endif

#else

static int	arpControlProtl();
static xkern_return_t arpDemux();
static void	arpHdrStore();
static long	arpHdrLoad();
static void 	arpInterfaceInit();
static Interface *isMyIp();
static void 	sendArpReply();
#ifdef ANSWER_RARPS 
static void 	sendRarpReply();
#endif

#endif __STDC__


static void
arp_print(s, m)
    char *s;
    arppak *m;
{
    xTrace3(arpp, TR_ALWAYS, "%s arp %s (%d) message:", s,
	    codenames[m->arp_op], m->arp_op);
    xTrace2(arpp, TR_ALWAYS, "  source %s @ %s",
	    ethHostStr(&m->arp_sha), ipHostStr(&m->arp_spa));
    xTrace2(arpp, TR_ALWAYS, "  target %s @ %s",
	    ethHostStr(&m->arp_tha), ipHostStr(&m->arp_tpa));
}


static void
arpFuncs(self)
    XObj self;
{
    self->control = arpControlProtl;
    self->demux = arpDemux;
}


void
arp_init(self)
    XObj self;
{
    int  	i;
    
    xTrace0(arpp, 1, "ARP init");
    arpTableInit();
    arpPlatformInit(self);
    ARP = self;
    arpFuncs(self);
    self->state = (VOID *)&pstate;
    pstate.numif = ARP->numdown;
    /* 
     * We create a rarp protocol for each arp protocol, mainly to be
     * able to find the protocol number of rarp relative to the llp
     */
    if ( (pstate.rarp = xCreateProtl(arpFuncs, "rarp", 0, 0)) == ERR_XOBJ ) {
	xError("ARP could not create RARP protocol");
	return;
    }
    pstate.rarp->state = self->state;
    for ( i = 0; i < self->numdown; i++ ) {
	xSetDown(pstate.rarp, i, xGetDown(self, i));
    }
    for ( i = 0; i < pstate.numif; i++ ) {
	arpInterfaceInit(self, xGetDown(self, i), &pstate.ifs[i]);
    }
}


static void
arpInterfaceInit(self, llp, ifs)
    XObj self;
    XObj llp;
    Interface *ifs;
{
    ETHhost	localEthHost;
    PSTATE	*ps;
    Part	part;

    ps = (PSTATE *)self->state;
    bzero((char *)ifs, sizeof(Interface));
    xControl(llp, GETMYHOST, (char *)&ifs->pub.ethaddr, sizeof(ETHhost));
    localEthHost = ifs->pub.ethaddr;
    /*
     * Openenable lower protocol for arps and rarps
     */
    partInit(&part, 1);
    partPush(part, ANY_HOST);
    xOpenEnable(self, self, llp, &part);
    partPush(part, ANY_HOST);
    xOpenEnable(self, ps->rarp, llp, &part);
    /*
     * Open broadcast sessions for this interface
     */
    partInit(&part, 1);
    partPush(part, &bcastEthHost);
    if ( (ifs->priv.rarp_s = xOpen(self, ps->rarp, llp, &part)) == ERR_XOBJ) {
	xTrace1(arpp, 0, "ARP could not open rarp session for llp %s",
		llp->name);
	Kabort("ARP init failure (rarp)");
    }
    partInit(&part, 1);
    partPush(part, &bcastEthHost);
    if ((ifs->priv.arp_s = xOpen(self, self, llp, &part)) == ERR_XOBJ) {
	xTrace1(arpp, 0, "ARP could not open arp session for llp %s",
		llp->name);
	Kabort("ARP init failure (arp)");
    }
    /*
     * Create a default header for requests sent on this interface
     */
    ifs->priv.hdr.arp_hrd=1;
    ifs->priv.hdr.arp_prot=ARP_PROT;
    ifs->priv.hdr.arp_hlen=6;
    ifs->priv.hdr.arp_plen=4;
    ifs->priv.hdr.arp_sha = ifs->pub.ethaddr;

    ifs->pub.ifprotl = llp;
    /*
     * Get my IP address for this interface
     */
    xTrace1(arpp, 3, "My eaddr = %s", ethHostStr(&ifs->pub.ethaddr));
    while ( arpRevLookup(&ifs->pub.ipaddr, &ifs->pub.ethaddr) != 0 ) {
	xTrace1(arpp, 0,
		"ARP: Could not get my ip address for interface %s",
		llp->name);
	/*
	 * Most protocols aren't going to be very useful if we can't
	 * find out our own IP address.  Keep trying.
	 */
	Delay( INIT_RARP_DELAY );
    }
    arpLock( &ifs->pub.ipaddr );
    ifs->priv.hdr.arp_spa = ifs->pub.ipaddr;
}


/*
 * sendArpReply -- send an ARP reply to the sender of 'srcHdr'
 * with my ip and physical addresses on the interface 'ifs'
 */
static void
sendArpReply(ifs, lls, srcHdr)
    Interface *ifs;
    XObj lls;
    struct arppak *srcHdr;
{
    arppak	reply;
    Msg		repMsg;

    reply = ifs->priv.hdr;
    reply.arp_tha = srcHdr->arp_sha;
    reply.arp_tpa = srcHdr->arp_spa;
    reply.arp_op = ARP_RPLY;
    msgConstructEmpty(&repMsg);
    msgPush(&repMsg, arpHdrStore, &reply, ARP_HLEN, NULL);
    xTrace2(arpp, 3, "replying with arp message with op %d (%s)", 
	    reply.arp_op, codenames[reply.arp_op]);
    xPush(lls, &repMsg);
    msgDestroy(&repMsg);
}


#ifdef ANSWER_RARPS 

/*
 * sendRarpReply -- send a RARP reply to the sender of 'srcHdr'
 * if we know the answer
 */
static void
sendRarpReply(lls, srcHdr)
    XObj lls;
    struct arppak *srcHdr;
{
    struct arppak	reply;
    Msg			repMsg;
    IPhost		ipHost;
    Interface		*ifs;
    
    if ( arpRevLookupTable( &ipHost, &srcHdr->arp_tha ) != 0 ) {
	/*
	 * We don't have this value in our table
	 */
	xTrace1(arpp, 3, "Don't know address of %s, can't send RARP reply",
		ethHostStr(&srcHdr->arp_tha));
	return;
    }
    if ( (ifs = arpIpToInterface(&ipHost)) == 0 ) {
	/*
	 * This shouldn't happen
	 */
	xTrace1(arpp, 0, "arp could not get interface to send to %s",
		ipHostStr(&ipHost));
	return;
    }
    reply = ifs->priv.hdr;
    reply.arp_op = ARP_RRPLY;
    reply.arp_tha = srcHdr->arp_sha;
    reply.arp_tpa = ipHost;
    msgConstructEmpty(&repMsg);
    msgPush(&repMsg, arpHdrStore, &reply, ARP_HLEN, NULL);
    xTrace1(arpp, 3, "replying with arp message with op %d", reply.arp_op);
    xPush(lls, &repMsg);
    msgDestroy(&repMsg);
}

#endif ANSWER_RARPS


static xkern_return_t
arpDemux(self, s, msg)
    XObj self;
    XObj s;
    Msg *msg;
{
    struct arppak Amsg;
    
    xAssert(xIsSession(s));
    xAssert(xIsProtocol(self));
    
    msgPop(msg, arpHdrLoad, (VOID *)&Amsg, ARP_HLEN, NULL);
    xIfTrace(arpp, 3) arp_print("received", &Amsg);
    switch(Amsg.arp_op) {
      case ARP_REQ:
	{
	    Interface	*ifs;

	    if ((ifs = isMyIp(&Amsg.arp_tpa)) != 0) {
		arpSaveBinding(&Amsg.arp_spa, &Amsg.arp_sha);    
		sendArpReply(ifs, s, &Amsg);
	    }
	}
	break;
	
      case ARP_RPLY:
	arpSaveBinding(&Amsg.arp_spa, &Amsg.arp_sha);
	break;
	
      case ARP_RREQ:
#ifdef ANSWER_RARPS 
	sendRarpReply(s, &Amsg);
#endif
	break;
	
      case ARP_RRPLY:
	arpSaveBinding(&Amsg.arp_spa, &Amsg.arp_sha);
	arpSaveBinding(&Amsg.arp_tpa, &Amsg.arp_tha);
	break;
	
      default:
	{/*do nothing*/}
	break;
    }
    return XK_SUCCESS;
}


static void
arpTimeout(arg)
    VOID *arg;
{
    arpSendRequest( (ArpWait *)arg );
}

/*
 * arpSendRequest -- this is called both for initial requests and
 * as the timeout event
 */ 
void
arpSendRequest(w)
    ArpWait *w;
{
    Msg		msg;
    
    xTrace1(arpp, 3, "Arp timeout, state = %x", w);
    xAssert(w->type == ARP_ARP || w->type == ARP_RARP);
    if (*w->status == ARP_RSLVD) {
	xTrace0(arpp, 5, "Request already resolved, timeout exiting");
	return;
    }
    if (w->type==ARP_ARP && w->tries++ > ARP_RTRY) {
	xTrace0(arpp, 1, "arp timeout: giving up");
	arpSaveBinding( &w->u.reqMsg.arp_tpa, 0 );
	return;
    } else if (w->type==ARP_RARP && w->tries++ > ARP_RRTRY) {
	xTrace0(arpp, 1, "arp timeout: giving up");
	arpSaveBinding( 0, &w->u.remPhysHost );
	return;
    }
    xTrace0(arpp, 3, "arp timeout: trying again");
    /*
     *  start a new event
     */
    w->event = evSchedule( arpTimeout, w, ARP_TIME * 1000 );
    msgConstructEmpty(&msg);
    if ( w->type == ARP_RARP ) {
	/*
	 * Send a RARP request over each interface
	 */
	arppak		hdr;
	Interface	*ifs;
	int		i;

	for ( i=0; i < ARP->numdown; i++ ) {
	    ifs = &pstate.ifs[i];
	    hdr = ifs->priv.hdr;
	    hdr.arp_op = ARP_RREQ;
	    hdr.arp_tha = w->u.remPhysHost;
	    msgPush(&msg, arpHdrStore, &hdr, ARP_HLEN, NULL);
	    xIfTrace(arpp, 3) arp_print("sending", &hdr);
	    xPush(ifs->priv.rarp_s, &msg);
	    msgTruncate(&msg, 0);
	}
    } else {
	/*
	 * Send an ARP request -- we already know which interface
	 */
	msgPush(&msg, arpHdrStore, &w->u.reqMsg, ARP_HLEN, NULL);
	xIfTrace(arpp, 3) arp_print("sending", &w->u.reqMsg);
	xPush(w->ifs->priv.arp_s, &msg);
    }
    msgDestroy(&msg);
}


static int
arpControlProtl(self, op, buf, len)
    XObj self;
    int op;
    char *buf;
    int len;
{
    int reply;
    int i;
    IPhost	tmpIp;
    ETHhost	tmpEth;
    
    xAssert(xIsProtocol(self));
    switch (op) {
	
	/* 
	 * RESOLVE and RRESOLVE have to do all the bcopy's because
	 * they can't assume proper alignment of the buffer.
	 */
      case RESOLVE:
	{
	    checkLen(len, sizeof(ETHhost));
	    bcopy((char *)buf, (char *)&tmpIp, sizeof(IPhost));
	    if ( arpLookup(&tmpIp, &tmpEth) == 0 ) {
		reply = sizeof(ETHhost);
		bcopy((char *)&tmpEth, (char *)buf, sizeof(ETHhost));
	    } else {
		reply = -1;
	    }
	}
	break;
	
      case RRESOLVE:
	{
	    checkLen(len, sizeof(ETHhost));
	    bcopy((char *)buf, (char *)&tmpEth, sizeof(ETHhost));

	    if ( arpRevLookup(&tmpIp, &tmpEth) == 0 ) {
		reply = sizeof(IPhost);
		bcopy((char *)&tmpIp, (char *)buf, sizeof(IPhost));
	    } else {
		reply = -1;
	    }
	}
	break;
	
      case ARP_INSTALL:
	{
	    struct foo {
		IPhost ipaddr;
		ETHhost  eaddr;
	    } *ptr;
	    checkLen(len, sizeof *ptr);
	    ptr = (struct foo *) buf;
	    arpSaveBinding(&ptr->ipaddr, &ptr->eaddr);
	    reply = 0;
	}
	break;
	
      case ARP_GETIPINTERFACES:
	checkLen(len, ARP_MAXINTERFACES * sizeof(Interfaceid));
	for( i = 0; i < pstate.numif; i++ ) {
	    bcopy( (char *)&pstate.ifs[i].pub, buf, sizeof(Interfaceid) );
	    buf = buf + sizeof(Interfaceid);
	}
	reply = pstate.numif;
	break;
	
      case ARP_GETIPADDRS:
	checkLen(len, ARP_MAXINTERFACES * sizeof(IPhost));
	for( i = 0; i < pstate.numif; i++)
	  *(IPhost *)(buf + i*sizeof(IPhost)) = pstate.ifs[i].pub.ipaddr;
	reply = pstate.numif;
	break;
	
      default:
	reply = -1;
    }
    xTrace2(arpp, 3, "Arp control %s returns %d", 
	    op == (int)RESOLVE ? "resolve":
	    op == (int)ARP_GETIPINTERFACES ? "getipinterfaces" :
	    op == (int)ARP_GETIPADDRS ? "getipaddrs" :
	    op == (int)RRESOLVE ? "rresolve" :
	    "unknown", reply);
    return(reply);
}


/*
 * isMyIp -- return the interfaceid pointer to the interface for which
 * this is one of my ip addresses, null if it is not
 */
static Interface *
isMyIp(h)
    IPhost *h; 			
{
    int	i;

    for (i=0; i<pstate.numif; i++) {
	if (! bcmp((char *)h, (char *)&(pstate.ifs[i].pub.ipaddr), 
		   sizeof(IPhost))) {
	    return(&pstate.ifs[i]);
	}
    }
    return(0);
}




static void
arpHdrStore(hdr, netHdr, len, arg)
    VOID *hdr;
    char *netHdr;
    long int len;
    VOID *arg;
{
    /*
     * Need a temporary header structure to avoid alignment problems
     */
    arppak tmpHdr;
    
    xTrace0(arpp, 5, "Entering arpHdrStore");
    xAssert( len == sizeof(arppak) );
    bcopy( hdr, (char *)&tmpHdr, sizeof(arppak) );
    tmpHdr.arp_hrd = htons(tmpHdr.arp_hrd);
    tmpHdr.arp_prot = htons(tmpHdr.arp_prot);
    tmpHdr.arp_op = htons(tmpHdr.arp_op);
    bcopy( (char *)&tmpHdr, netHdr, sizeof(arppak) );
    xTrace0(arpp, 7, "leaving arpHdrStore");
}


static long
arpHdrLoad(hdr, netHdr, len, arg)
    VOID *hdr;
    char *netHdr;
    long int len;
    VOID *arg;
{
    xAssert( len == sizeof(arppak) );

    xTrace0(arpp, 5, "Entering arpHdrLoad");
    bcopy( netHdr, hdr, sizeof(arppak) );
    ((arppak *)hdr)->arp_hrd = ntohs(((arppak *)hdr)->arp_hrd);
    ((arppak *)hdr)->arp_prot = ntohs(((arppak *)hdr)->arp_prot);
    ((arppak *)hdr)->arp_op = ntohs(((arppak *)hdr)->arp_op);
    xTrace0(arpp, 7, "leaving arpHdrLoad");
    return len;
}


void
newWait(w)
    ArpWait *w;
{
    bzero((char *)w, sizeof(ArpWait));
    semInit(&w->s, 0);
}


void
newArpWait(w, h, ifs, status)
    ArpWait *w;
    IPhost *h;
    Interface *ifs;
    ArpStatus *status;
{
    newWait(w);
    w->u.reqMsg = ifs->priv.hdr;
    w->u.reqMsg.arp_tpa = *h;
    w->u.reqMsg.arp_op = ARP_REQ;
    w->status = status;
    w->ifs = ifs;
    w->type = ARP_ARP;
}


void
newRarpWait(w, h, status)
    ArpWait *w;
    ETHhost *h;
    ArpStatus *status;
{
    newWait(w);
    w->u.remPhysHost = *h;
    w->status = status;
    w->type = ARP_RARP;
}


/*
 * arpIpToInterface -- return the structure for the interface through which
 * the host can be reached, or NULL if the host can not be directly reached
 * through ARP's interfaces.
 */
Interface *
arpIpToInterface(h)
    IPhost *h;
{
    int i;

    for( i = 0; i < pstate.numif; i++ ) {
	if (IP_NETEQ(*h, pstate.ifs[i].pub.ipaddr)) {
	    return &pstate.ifs[i];
	}
    }
    return 0;
}

