/* RCS      -- $Header: /u2/dvadura/src/generic/dmake/src/RCS/make.c,v 1.4 90/11/29 21:51:17 dvadura Exp $
-- SYNOPSIS -- perform the update of all outdated targets.
-- 
-- DESCRIPTION
--	This is where we traverse the make graph looking for targets that
--	are out of date, and we try to infer how to make them if we can.
--	The usual make macros are understood, as well as some new ones:
--
--		$$	- expands to $
--		$@      - full target name
--		$*      - target name with no suffix, same as $(@:db)
--			  or, the value of % in % meta rule recipes
--		$?      - list of out of date prerequisites
--		$<      - all prerequisites associated with rules line
--		$&	- all prerequisites associated with target
--		$>      - library name for target (if any)
--		$^	- out of date prerequisites taken from value of $<
--		{{	- expands to {
--		}}	- expands to }
--		\#	- expands to #
-- 
-- AUTHOR
--      Dennis Vadura, dvadura@watdragon.uwaterloo.ca
--      CS DEPT, University of Waterloo, Waterloo, Ont., Canada
--
-- COPYRIGHT
--      Copyright (c) 1990 by Dennis Vadura.  All rights reserved.
-- 
--      This program is free software; you can redistribute it and/or
--      modify it under the terms of the GNU General Public License
--      (version 1), as published by the Free Software Foundation, and
--      found in the file 'LICENSE' included with this distribution.
-- 
--      This program is distributed in the hope that it will be useful,
--      but WITHOUT ANY WARRANTY; without even the implied warrant of
--      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--      GNU General Public License for more details.
-- 
--      You should have received a copy of the GNU General Public License
--      along with this program;  if not, write to the Free Software
--      Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
--
-- LOG
--     $Log:	make.c,v $
 * Revision 1.4  90/11/29  21:51:17  dvadura
 * Ignore targets with no recipe if Augmake set.
 * 
 * Revision 1.3  90/11/07  14:06:44  dvadura
 * Enahncec Augmake compatibility to assume NULL recipe for targets that
 * have prerequisites.
 * 
 * Revision 1.2  90/10/30  23:02:41  dvadura
 * Add Push_dir test and call immediately after inferring a recipe for a
 * target.  This catches and handles properly the case when a target cell
 * infers a recipe from a %-meta rule that has a .SETDIR attribute.
 * 
 * Revision 1.1  90/10/06  12:03:59  dvadura
 * dmake Release, Version 3.6
 * 
*/

#include "extern.h"
#include "alloc.h"
#include "db.h"

static	int	_exec_commands ANSI((CELLPTR, HOWPTR));
static	void	_drop_mac ANSI((HASHPTR));
static	void	_print_cmnd ANSI((char*, int, int));
static	void	_set_recipe ANSI((char*, int));
static	void	_set_tmd ANSI(());
static	void	_append_file ANSI((STRINGPTR, FILE*, char*, int));

#define RP_GPPROLOG	0
#define RP_RECIPE	1
#define RP_GPEPILOG	2
#define NUM_RECIPES	3

static STRINGPTR _recipes[ NUM_RECIPES ];
static CELLPTR	 Root;		/* root cell for make graph */



int
Make_targets()/*
================
   Actually go and make the targets on the target list */
{
   LINKPTR	    lp;
   int		    done = 0;

   DB_ENTER( "Make_targets" );

   /* Make sure the graph gets blown up for .SETDIR targets.
    * Only required for architectures that support parallel  
    * makes. */
   Explode_prq( Fringe_hd, NIL(CELL), FALSE );

   _set_recipe( ".GROUPPROLOG", RP_GPPROLOG );
   _set_recipe( ".GROUPEPILOG", RP_GPEPILOG );

   while( !done ) {
      done = F_MADE;
      for( lp = Fringe_hd; lp != NIL(LINK); lp = lp->cl_next ) {
	 int rval = 0;

	 Root = lp->cl_prq;
	 if( (rval |= Make(Root, Root->CE_HOW, NIL(CELL))) == -1 )
	    DB_RETURN(1);
	 else
	    done &= (Root->ce_flag & F_MADE);

	 if( !rval && !done ) Wait_for_child( FALSE, -1 );
      }
   }

   for( lp = Fringe_hd; lp != NIL(LINK); lp = lp->cl_next ) {
      Root = lp->cl_prq;
      if( !(Root->ce_attr & A_UPDATED) )
	 printf( "`%s' is up to date\n", Root->CE_NAME );
   }

   DB_RETURN( 0 );
}



int
Make( cp, how, setdirroot )/*
=============================  Make a specified target */
CELLPTR cp;
HOWPTR  how;
CELLPTR setdirroot;
{
   register LINKPTR dp;
   register CELLPTR tcp;
   char		    *name, *lib;
   HASHPTR	    m_at    = NIL(HASH);
   char             *all    = NIL(char);
   char             *inf    = NIL(char);
   char             *outall = NIL(char);
   char             *imm    = NIL(char);
   int              rval    = 0;
   int		    push    = 0;
   time_t           otime   = (time_t) 1L;
   time_t	    ttime   = (time_t) 1L;

   DB_ENTER( "Make" );
   DB_PRINT( "mem", ("%s:-> mem %ld", cp->CE_NAME, (long) coreleft()) );

   /* NOTE:  The only time we are called with a NIL how pointer is for
    *	     cells that are to be statted only or for cells that are destined
    *	     to have a set of rules inferred for them. */

   if( how == NIL(HOW) ) {
      TALLOC( cp->CE_HOW, 1, HOW );
      how = cp->CE_HOW;
   }

   /* Check to see if we have made the node already.  If so then don't do
    * it again, except if the cell's ce_setdir field is set to something other
    * than the value of setdirroot.  If they differ then, and we have made it
    * already, then make it again and set the cell's stat bit to off so that
    * we do it again. */

   if( how->hw_flag & F_VISITED ) {
      /* we may return if we made it already from the same setdir location,
       * or if it is not a library member whose lib field is non NULL.  (if
       * it is such a member then we have a line of the form:
       *	lib1 lib2 .LIBRARY : member_list...
       * and we have to make sure all members are up to date in both libs. */

      if( cp->ce_setdir == setdirroot &&
	  !((cp->ce_attr & A_LIBRARYM) && (cp->ce_lib != NIL(char))) )
	 DB_RETURN( 0 );

      /* We check to make sure that we are comming from a truly different
       * directory, ie. ".SETDIR=joe : a.c b.c d.c" are all assumed to come
       * from the same directory, even though setdirroot is different when
       * making dependents of each of these targets. */

      if( cp->ce_setdir != NIL(CELL) && setdirroot != NIL(CELL) &&
          cp->ce_setdir->ce_dir == setdirroot->ce_dir )
         DB_RETURN( 0 );

      cp->ce_flag  &= ~(F_STAT|F_VISITED|F_MADE);
      how->hw_flag &= ~(F_VISITED|F_MADE);
   }

   if( how->hw_next != NIL(HOW))
      if( (rval = Make(cp, how->hw_next, setdirroot)) == -1 ||
	  !(how->hw_next->hw_flag & F_MADE) )
	 goto stop_making_it;

   /* If we are supposed to change directories for this target then do so.
    * If we do change dir, then modify the setdirroot variable to reflect
    * that fact for all of the prerequisites that we will be making. */

   cp->ce_setdir = setdirroot;

   if( cp->ce_attr & A_SETDIR ) {
      /* Change directory only if the previous .SETDIR is a different
      /* directory from the current one.  ie. all cells with the same .SETDIR
      /* attribute are assumed to come from the same directory. */

      if( (setdirroot == NIL(CELL) || setdirroot->ce_dir != cp->ce_dir) &&
          (push = Push_dir(cp, (((cp->ce_attr|Glob_attr)&A_IGNORE) != 0))) )
	 setdirroot = cp;
   }

   DB_PRINT( "mem", ("%s:-A mem %ld", cp->CE_NAME, (long) coreleft()) );
   if( (how->hw_recipe == NIL(STRING) && !(how->hw_flag & F_INFER)) ) {
      HOWPTR thp;
      
      if( Verbose )
	 printf( "%s:  Infering prerequisite(s) and recipe for [%s]\n", Pname,
	 	 cp->CE_NAME );

      thp = how->hw_next;
      (void) Infer_recipe( cp, how, NIL(DFASET), setdirroot );

      /* Change directory only if the previous .SETDIR is a different
      /* directory from the current one.  ie. all cells with the same .SETDIR
      /* attribute are assumed to come from the same directory. */
      if( cp->ce_attr & A_SETDIR )
	 if( (setdirroot == NIL(CELL) || setdirroot->ce_dir != cp->ce_dir) &&
	     (push += Push_dir(cp, (((cp->ce_attr|Glob_attr)&A_IGNORE) != 0))) )
	    setdirroot = cp;

      /* If we inferred a new set of prerequisites then make
       * them before the current list represented by the current how pointer */
      if( thp != how->hw_next )
	 if( (rval = Make(cp, how->hw_next, cp->ce_setdir)) == -1 ||
	     !(how->hw_next->hw_flag & F_MADE) )
	    goto stop_making_it;
   }

   DB_PRINT( "mem", ("%s:-A mem %ld", cp->CE_NAME, (long) coreleft()) );
   /* If we have not yet statted the target then do so. */
   if( !(cp->ce_flag & F_STAT) ) {
      time_t itime = cp->ce_time;
      Stat_target( cp, TRUE );

      if( cp->ce_time == (time_t)0L ) {
         if( cp->ce_flag & F_INFER )
	    cp->ce_time = itime;
      }
      else
         cp->ce_attr |= A_PRECIOUS;  /* File exists so don't remove it later. */

      if( Verbose )
	 printf("%s:  Time stamp of [%s] is %ld\n",Pname,cp->CE_NAME,
	        cp->ce_time);
   }

   DB_PRINT( "make", ("(%s, %ld, 0x%08x, 0x%04x)", cp->CE_NAME,
			cp->ce_time, cp->ce_attr, cp->ce_flag) );

   if( !(cp->ce_flag & F_TARGET) && (cp->ce_time == (time_t) 0L) )
      if( Makemkf ) {
	 rval = -1;
	 goto stop_making_it;
      }
      else if(how->hw_prq != NIL(LINK)||(Augmake && (cp->ce_flag&F_EXPLICIT)))
	 /* Assume an empty recipe for a target that we have run inference on
	  * but do not have a set of rules for but for which we have inferred
	  * a list of prerequisites. */
	 cp->ce_flag |= F_RULES;
      else
	 Fatal( "`%s' not found, and can't be made", cp->CE_NAME );

   DB_PRINT( "mem", ("%s:-A mem %ld", cp->CE_NAME, (long) coreleft()) );

   /* set value of $* if we have not infered a recipe, in this case $* is
    * the same as $(@:db), this allows us to be compatible with BSD make */
   if( how->hw_per == NIL(char) ) how->hw_per = "$(@:db)";

   for( dp = how->hw_prq; dp != NIL(LINK); dp = dp->cl_next ) {
      int seq;

      /* Make the prerequisite, note that if the current target has the
       * .LIBRARY attribute set we pass on to the prerequisite the .LIBRARYM
       * attribute and pass on the name of the current target as the library
       * name, and we take it away when we are done.  */

      tcp = dp->cl_prq;
      seq = (((cp->ce_attr | how->hw_attr | Glob_attr) & A_SEQ) != 0);

      if( tcp->ce_flag & F_VISITED )
	 if( tcp->ce_flag & F_MADE ) {
	    if( tcp->ce_time > ttime ) ttime = tcp->ce_time;
	    continue;
	 }
	 else
	    goto stop_making_it;

      if( strchr(tcp->CE_NAME, '$') ) {
	 /* Make this prerequisite link point at the real prerequisite we
	  * are after, ie figure out what the dynamic one is and point at it. */

	 m_at = Def_macro( "@", cp->ce_fname, M_MULTI );
	 name = Expand( tcp->CE_NAME );

	 tcp = dp->cl_prq = Def_cell( name, setdirroot );
	 FREE( name );
      }

      if( cp->ce_attr & A_LIBRARY ) {
         tcp->ce_attr |= A_LIBRARYM;
	 tcp->ce_lib   = cp->ce_fname;
      }

      if( (tcp->ce_flag & (F_INFER|F_STAT))==F_INFER && cp->ce_time > ttime )
	 tcp->ce_time = cp->ce_time;

      /* Propagate parents F_REMOVE and F_INFER attribute to children.
       * Make certain you do this AFTER propagating the time, since the
       * time propagation test above uses the F_INFER flag to decide if
       * it should do so. */
      tcp->ce_flag |= cp->ce_flag & (F_REMOVE|F_INFER);

      rval |= Make(tcp, tcp->CE_HOW, setdirroot);

      if( cp->ce_attr & A_LIBRARY )
         tcp->ce_attr ^= A_LIBRARYM;

      if( rval == -1 || (seq && (rval == 1)) )
	 goto stop_making_it;

      if( tcp->ce_time > ttime ) ttime = tcp->ce_time;
   }

   /* Do the loop again.  We are most definitely going to make the current
    * how cell now.  NOTE:  doing this loop here also results in a reduction
    * in peak memory usage by the algorithm. */

   for( dp = how->hw_prq; dp != NIL(LINK); dp = dp->cl_next ) {
      char *dir;
      int  tgflg;
      tcp  = dp->cl_prq;
      name = tcp->ce_fname;

      /* make certain that all prerequisites are made prior to advancing. */
      if( !(tcp->ce_flag & F_MADE) ) goto stop_making_it;

      /* If the target is a library, then check to make certain that a member
       * is newer than an object file sitting on disk.  If the disk version
       * is newer then set the time stamps so that the archived member is
       * replaced. */
      if( cp->ce_attr & A_LIBRARY )
	 if( tcp->ce_time < cp->ce_time ) {
	    time_t mtime = Do_stat( tcp->ce_fname, tcp->ce_lib, NIL(char *) );
	    if( mtime < tcp->ce_time ) tcp->ce_time = cp->ce_time+1L;
	 }

      if( tcp->ce_time > otime ) otime = tcp->ce_time;

      /* Note:  If the prerequisite was made using a SETDIR
       * 	directory then we will include the directory in the name
       *	of the prerequisite when we build the $?, $&, $^ and $< lists
       */

      if( (dir = tcp->ce_dir ) != NIL(char) )
	 name = Build_path( dir, name );

      all  = _strapp( all, name );
      if( tgflg = (dp->cl_flag & F_TARGET) ) inf = _strapp( inf, name );

      if((cp->ce_time<tcp->ce_time) || ((tcp->ce_flag & F_TARGET) && Force)) {
         outall = _strapp( outall, name );
         if( tgflg ) imm = _strapp( imm, name );
      }
   }

   DB_PRINT( "mem", ("%s:-C mem %ld", cp->CE_NAME, (long) coreleft()) );
   DB_PRINT( "make", ("I make '%s' if %ld > %ld", cp->CE_NAME, otime,
	      cp->ce_time) );

   /* mark the how cell as visited since we are making it for certain now. */
   how->hw_flag |= F_VISITED;
   if( Verbose ) printf( "%s:  >>>> Making [%s]\n", Pname, cp->CE_NAME );

   if( Check && (cp->ce_time < otime) ) {
      /* Only checking so stop as soon as we determine we will make something */
      rval = -1;
      goto stop_making_it;
   }

   if( (cp->ce_time < otime) || ((cp->ce_flag & F_TARGET) && Force) ) {
      if( Verbose )
	 printf( "%s:  Updating [%s], (%ld > %ld)\n", Pname,
		 cp->CE_NAME, otime, cp->ce_time );

      cp->ce_attr |= A_UPDATED;

      if( Touch ) {
	 name = cp->ce_fname;
	 lib  = cp->ce_lib;

	 if( !(Glob_attr & A_SILENT) || !Trace )
	    if( lib == NIL(char) )
	       printf("touch(%s)", name );
	    else if( cp->ce_attr & A_SYMBOL )
	       printf("touch(%s((%s)))", lib, name );
	    else
	       printf("touch(%s(%s))", lib, name );

	 if( !Trace )
	    if( Do_touch( name, lib,
		(cp->ce_attr & A_SYMBOL) ? &name : NIL(char *) ) != 0 )
	       printf( "  not touched - non-existant" );

	 printf( "\n" );
	 Update_time_stamp( cp, how );
      }
      else if( how->hw_recipe != NIL(STRING) ) {
	 HASHPTR m_q, m_b, m_g, m_l, m_bb, m_up;
	 TKSTR   tk;

	 m_at = Def_macro( "@", cp->ce_fname,M_MULTI );
	 m_g  = Def_macro( ">", cp->ce_lib,  M_MULTI );
	 m_q  = Def_macro( "?", outall,      M_MULTI );
	 m_b  = Def_macro( "<", inf,         M_MULTI );
	 m_l  = Def_macro( "&", all,         M_MULTI );
	 m_bb = Def_macro( "*", how->hw_per, M_MULTI );
	 m_up = Def_macro( "^", imm,         M_MULTI );

	 _recipes[ RP_RECIPE ] = how->hw_recipe;

	 if( !(how->hw_flag & F_SINGLE) )
	    rval = _exec_commands( cp, how );
	 else {
	    _drop_mac( m_q );
	    SET_TOKEN( &tk, outall );

	    Doing_bang = TRUE;
	    name = Get_token( &tk, "", FALSE );
	    do {
	       m_q->ht_value = name;

	       Wait_for_completion = TRUE;	/* Reset in _exec_commands */
	       rval = _exec_commands( cp, how );
	       Unlink_temp_files(how);
	    }
	    while( *(name = Get_token( &tk, "", FALSE )) != '\0' );
	    Doing_bang = FALSE;
	    Update_time_stamp( cp, how );

	    m_q->ht_value = NIL(char);
	 }

	 _drop_mac( m_g  );
	 _drop_mac( m_q  );
	 _drop_mac( m_b  );
	 _drop_mac( m_l  );
	 _drop_mac( m_bb );
	 _drop_mac( m_up );
      }
      else if( !(cp->ce_flag & F_RULES) &&
	       ((cp != Root) || !(cp->ce_flag & F_EXPLICIT)) )
	 Fatal( "Don't know how to make `%s'",cp->CE_NAME );
      else {
         /* Empty recipe, set the how flag as MADE and update the time stamp */
	 Update_time_stamp( cp, how );
      }
   }
   else {
      /* Make sure everyone gets remade if Force is set */
      if( !(cp->ce_flag & F_TARGET) && Force ) cp->ce_time = Do_time();

      /* Set how to MADE since it is done, it was not out of date */
      how->hw_flag |= F_MADE;
      if( cp->CE_HOW == how ) {
	 tcp = cp;
	 do {
	    tcp->ce_flag |= F_MADE;
	    tcp = tcp->ce_all;
	 }
	 while( tcp != NIL(CELL) && tcp != cp );
      }
   }

   if( cp->CE_HOW == how ) cp->ce_flag |= F_VISITED;

stop_making_it:
   if( m_at != NIL(HASH) ) _drop_mac( m_at );

   while( push-- )  Pop_dir(FALSE);

   if( inf    != NIL(char) ) FREE( inf    );
   if( all    != NIL(char) ) FREE( all    );
   if( imm    != NIL(char) ) FREE( imm    );
   if( outall != NIL(char) ) FREE( outall );

   DB_PRINT( "mem", ("%s:-< mem %ld", cp->CE_NAME, (long) coreleft()) );
   DB_RETURN( rval );
}



static void
_drop_mac( hp )/*
================ set a macro value to zero. */
HASHPTR hp;
{
   if( hp->ht_value != NIL(char) ) {
      FREE( hp->ht_value );
      hp->ht_value = NIL(char);
   }
}



static int
_exec_commands( cp, how )/*
===========================
  Execute the commands one at a time that are pointed to by the rules pointer
  of the target cp. If a group is indicated, then the hw_attr determines
  .IGNORE and .SILENT treatment for the group.
  
  The function returns 0, if the command is executed and has successfully
  returned, and returns 1 if the command is executing but has not yet
  returned (for parallel makes).
  
  The F_MADE bit in the how cell is guaranteed set when the command has
  successfully completed.  */
CELLPTR cp;
HOWPTR  how;
{
   static HASHPTR hsh = NIL(HASH);
   register STRINGPTR	rp;
   char			*cmnd;
   char			*groupfile;
   FILE    		*tmpfile;
   int			do_it;
   t_attr		attr;
   int			group;
   int			trace;
   int			rval  = 0;

   DB_ENTER( "_exec_commands" );

   Current_target = how;
   attr  = Glob_attr | cp->ce_attr;
   trace = Trace || !(attr & A_SILENT);
   group = how->hw_flag & F_GROUP;

   if( group ) {
      attr  |= how->hw_attr;
      trace  = Trace || !(attr & A_SILENT);

      if( !Trace ) tmpfile = Start_temp( Grp_suff, cp, how, &groupfile );
      if( trace )  fputs( "[\n", stdout );

      /* Emit group prolog */
      if( attr & A_PROLOG )
         _append_file( _recipes[RP_GPPROLOG], tmpfile, cp->CE_NAME, trace );
   }

   if( hsh == NIL(HASH) )
      hsh = Def_macro("USESHELL", NIL(char), M_MULTI|M_EXPANDED);

   /* Process commands in recipe. If in group, merely append to file.
    * Otherwise, run them.  */
   for( rp = _recipes[RP_RECIPE]; rp != NIL(STRING); rp=rp->st_next,FREE(cmnd)){
      t_attr a_attr = attr|how->hw_attr|rp->st_attr;
      int    shell  = ((a_attr & A_SHELL) != 0);

      Swap_on_exec  = ((a_attr & A_SWAP) != 0);	  /* Swapping for DOS only */
      hsh->ht_value = (group||shell)?"yes":"no";
      do_it = !Trace;

      if( !group && Trace && _strstr(rp->st_string,"$(MAKE)") ) {
	 Wait_for_completion |= Trace;
	 do_it = TRUE;
      }

      /* Expand needs these set to properly handle some GNU style functions */
      Recipe_cell = cp;
      Recipe_how  = how;
      cmnd = Expand( rp->st_string );

      if( group )
         Append_line( cmnd, TRUE, tmpfile, cp->CE_NAME, trace, 0 );
      else {
	 if( *_strspn(cmnd, " \t") != '\0' )
	    _print_cmnd(cmnd, !(do_it && ((rp->st_attr | attr) & A_SILENT)), 0);
	 else
	    do_it = FALSE;

	 rval=Do_cmnd(cmnd,FALSE,do_it,cp,how,((attr|rp->st_attr)&A_IGNORE)!=0,
		      shell, rp->st_next == NIL(STRING) );
      }
   }

   /* If it is a group then output the EPILOG if required and possibly
    * execute the command */
   if( group ) {
      if( attr & A_EPILOG )	/* emit epilog */
	 _append_file( _recipes[RP_GPEPILOG], tmpfile, cp->CE_NAME, trace );

      if( trace ) fputs("]\n", stdout);

      if( do_it = !Trace ) Close_temp( how, tmpfile );
      rval = Do_cmnd(groupfile, TRUE, do_it, cp, how, (attr & A_IGNORE)!=0,
		     TRUE, TRUE);
   }

   Wait_for_completion = FALSE;
   DB_RETURN( rval );
}


static void
_print_cmnd( cmnd, echo, map )/*
================================
   This routine is called to print out the command to stdout.  If echo is
   false the printing to stdout is supressed, but the new lines in the command
   are still deleted. */
char *cmnd;
int  echo;
int  map;
{
   register char *p;
   register char *n;

   DB_ENTER( "_print_cmnd" );

   if( echo ) {
      printf( "%s\n", cmnd  );
      fflush(stdout);
   }

   for( p=cmnd; (n = strchr(p, CONTINUATION_CHAR)) != NIL(char); )
      if(n[1] == '\n') {
	 DB_PRINT( "make", ("fixing [%s]", p) );
	 strcpy( n, n+2 );
	 p = n;
      }
      else {
	 if( map ) Map_esc( n );
	 p = n+1;
      }

   DB_VOID_RETURN;
}



/* These routines are used to maintain a stack of directories when making
 * the targets.  If a target cd's to the directory then it is assumed that
 * it will undo it when it is finished making itself. */

static STRINGPTR dir_stack = NIL(STRING);

int
Push_dir( cp, ignore )/*
==========================
   Change the current working directory to dir and save the current
   working directory on the stack so that we can come back.
   
   If ignore is TRUE then do not complain about _ch_dir if not possible.*/
CELLPTR cp;
int  ignore;
{
   STRINGPTR   new_dir;
   char        *dir;

   DB_ENTER( "Push_dir" );

   if( (dir = cp->ce_dir) == NIL(char) ) dir = Pwd;
   if( *dir == '\'' && dir[strlen(dir)-1] == '\'' ) {
      dir = _strdup(dir+1);
      dir[strlen(dir)-1]='\0';
   }
   else
      dir = Expand(dir);

   if( Set_dir(dir) ) {
      if( !ignore )
         Fatal( "Unable to change to directory `%s', target is [%s]",
	        dir, cp->CE_NAME );
      FREE(dir);
      DB_RETURN( 0 );
   }

   DB_PRINT( "dir", ("Push: [%s]", dir) );
   if( Verbose ) printf( "%s:  Changed to directory [%s]\n", Pname, dir  );

   FREE( dir );
   TALLOC( new_dir, 1, STRING );
   new_dir->st_next   = dir_stack;
   dir_stack          = new_dir;
   new_dir->st_string = _strdup( Pwd );

   Def_macro( "PWD", Get_current_dir(), M_MULTI | M_EXPANDED );
   _set_tmd();

   DB_RETURN( 1 );
}



void
Pop_dir(ignore)/*
=================
   Change the current working directory to the previous saved dir. */
int ignore;
{
   STRINGPTR old_dir;
   char      *dir;

   DB_ENTER( "Pop_dir" );

   if( dir_stack == NIL(STRING) )
      if( ignore ) {
         DB_VOID_RETURN;
      }
      else
	 Error( "Directory stack empty for return from .SETDIR" );

   if( Set_dir(dir = dir_stack->st_string) )
      Fatal( "Could not change to directory `%s'", dir );

   Def_macro( "PWD", dir, M_MULTI | M_EXPANDED );
   DB_PRINT( "dir", ("Pop: [%s]", dir) );
   if( Verbose ) printf( "%s:  Changed back to directory [%s]\n", Pname, dir);

   old_dir   = dir_stack;
   dir_stack = dir_stack->st_next;

   FREE( old_dir->st_string );
   FREE( old_dir );
   _set_tmd();

   DB_VOID_RETURN;
}



static void
_set_tmd()/*
============
   Set the TWD Macro */
{
   TKSTR md, pd;
   char  *m, *p;
   char  *tmd;
   int   is_sep;
   int   first = 1;

   SET_TOKEN( &md, Makedir );
   SET_TOKEN( &pd, Pwd );

   m = Get_token( &md, DirBrkStr, FALSE );
   (void) Get_token( &pd, DirBrkStr, FALSE );
   is_sep = (strchr(DirBrkStr, *m) != NIL(char));
   tmd = _strdup( "" );

   do {
      m = Get_token( &md, DirBrkStr, FALSE );
      p = Get_token( &pd, DirBrkStr, FALSE );

      if( !is_sep && strcmp(m, p) ) {	/* they differ */
	 char *tmp;
	 if( first ) {		/* They differ in the first component	*/
	    tmd = Makedir;	/* In this case use the full path	*/
	    break;
	 }

	 if( *p ) tmp = Build_path( "..", tmd );
	 if( *m ) tmp = Build_path( tmd, m );
	 FREE( tmd );
	 tmd = _strdup( tmp );
      }

      is_sep = 1-is_sep;
      first  = 0;
   } while (*m || *p);

   CLEAR_TOKEN( &md );
   CLEAR_TOKEN( &pd );

   Def_macro( "TMD", tmd, M_MULTI | M_EXPANDED );
   if( tmd != Makedir ) FREE( tmd );
}


static void
_set_recipe( target, ind )/*
============================
   Set up the _recipes static variable so that the slot passed in points
   at the rules corresponding to the target supplied. */
char *target;
int  ind;
{
   CELLPTR cp;
   HASHPTR hp;

   if( (hp = Get_name( target, Defs, FALSE, NIL(CELL) )) != NIL(HASH) ) {
      cp = hp->CP_OWNR;
      if( cp->CE_HOW != NIL(HOW) ) _recipes[ ind ] = cp->CE_RECIPE;
   }
   else
      _recipes[ ind ] = NIL(STRING);
}



void
Append_line( cmnd, newline, tmpfile, name, printit, map )
char *cmnd;
int  newline;
FILE *tmpfile;
char *name;
int  printit;
int  map;
{
   _print_cmnd( cmnd, printit, map );

   if( Trace ) return;

   fputs(cmnd, tmpfile);
   if( newline ) fputc('\n', tmpfile);
   fflush(tmpfile);

   if( ferror(tmpfile) )
      Fatal("Write error on temporary file, while processing `%s'", name);
}



static void
_append_file( rp, tmpfile, name, printit )
register STRINGPTR rp;
FILE 		   *tmpfile;
char 		   *name;
int 		   printit;
{
   char *cmnd;

   while( rp != NIL(STRING) ) {
      Append_line(cmnd = Expand(rp->st_string), TRUE, tmpfile, name, printit,0);
      FREE(cmnd);
      rp = rp->st_next;
   }
}
