#charset "us-ascii"

/*
 *   MegaTADS. Additions and modifications to the en_us.t file.
 *
 *   Copyright 2007, Krister Fundin.
 */

#include <adv3.h>
#include <en_us.h>
#include <strcomp.h>

#include "mega.h"
#include "mega_en_us.h"

/* ---------------------------------------------------------------------- */
/*
 *   Allow the ampersand in some places where AND is allowed (not after a
 *   comma though). Also, allow a question mark as sentence-ending
 *   punctutation and understand several periods, exclamation marks, etc.
 *   in a row.
 */

#ifdef MEGA_PUNCT_ON

modify grammar commandOnlyConjunction(nonSentenceEnding):
    'then'
    | ('and' | '&') 'then'
    | ',' 'then'
    | ',' 'and' 'then'
    | ';'
    :
;

modify grammar commandOrNounConjunction(main):
    ','
    | ('and' | '&')
    | ',' 'and'
    :
;

modify grammar nounConjunction(main):
    ','
    | ('and' | '&')
    | ',' 'and'
    :
;

modify grammar disambigOrdinalList(tail):
    ordinalWord->ord1_ ('and' | '&' | ',') ordinalWord->ord2_
    :
;

modify grammar disambigOrdinalList(head):
    ordinalWord->ord_ ('and' | '&' | ',') disambigOrdinalList->lst_
    :
;

modify grammar commandOnlyConjunction(sentenceEnding):
    punctuationList
    :
;

grammar punctuationList:
    singlePunctuation
    | punctuationList singlePunctuation
    : BasicProd
;

grammar singlePunctuation:
    '.'
    | '!'
    | '?'
    : BasicProd
;

#endif

/* ---------------------------------------------------------------------- */
/*
 *   Interjections. We define a single ConvIAction for handling all of
 *   these. Ideally, though, the bare interjections should be treated as a
 *   different type of command, so that we can disallow weird commands like
 *   TELL BOB TO HELLO.
 */

#ifdef MEGA_INTERJECT_ON

VerbRule(Interject)
    singleInterjection
    : InterjectAction

    verbPhrase = ('say/saying ' + itrjc_.interjectionPhrase)
;

VerbRule(SayInterjection)
    ('say' | 'answer') singleInterjection
    : SayInterjectionAction

    verbPhrase = ('say/saying ' + itrjc_.interjectionPhrase)
;

VerbRule(SayInterjectionTo)
    ('say' | 'answer') singleInterjection 'to' singleDobj
    | ('tell' | 'answer') singleDobj singleInterjection
    : SayInterjectionToAction

    verbPhrase = ('say/saying ' + itrjc_.interjectionPhrase + ' to (whom)')
;

Interjection(Hello)
    'hello' ('there' |)
    | 'hallo'
    | 'hi'
    | 'good' ('day' | 'evening' | 'morning')
    | 'how' 'do' 'you' 'do'
    | 'greetings'
    : BasicProd

    sayProp = &sayHello
    interjectionPhrase = 'hello'
;

Interjection(Goodbye)
    'goodbye' | 'good-bye' | 'good' 'bye' | 'bye'
    : BasicProd

    sayProp = &sayGoodbye
    interjectionPhrase = 'goodbye'
;

Interjection(Yes)
    'yes' | 'affirmative' | 'yeah' | 'yea' | 'yup' | 'sure'
    : BasicProd

    sayProp = &sayYes
    interjectionPhrase = 'yes'
;

Interjection(No)
    'no' | 'negative' | 'nay' | 'nope'
    : BasicProd

    sayProp = &sayNo
    interjectionPhrase = 'no'
;

removeVerb(Hello);
removeVerb(Goodbye);
removeVerb(Yes);
removeVerb(No);

#endif

/* ---------------------------------------------------------------------- */
/*
 *   Some new ways of saying ALL X.
 */

#ifdef MEGA_GRAMMAR_ON

grammar qualifiedPluralNounPhrase(every):
    everyPhrase indetSingularNounPhrase->np_
    : AllPluralProd
;

grammar qualifiedPluralNounPhrase(everyOne):
    everyPhrase 'one' 'of' detPluralNounPhrase->np_
    : AllPluralProd
;

grammar everyPhrase(main):
    'every' ('single' | 'last' |) | 'each' | 'each' 'and' 'every'
    : BasicProd
;

#endif

/* ---------------------------------------------------------------------- */
/*
 *   Understand THIS, THAT, THESE, THOSE and THEM as definite articles and
 *   pronouns. Also, add 'EM as slang for THEM.
 */

#ifdef MEGA_GRAMMAR_ON

modify grammar qualifiedSingularNounPhrase(definite):
    ('the'
     | 'this' ('here' |)
     | 'that' ('there' |)
     | 'the' ('one' | '1')
     |)
    indetSingularNounPhrase->np_
    :
;

modify grammar explicitDetPluralNounPhrase(definite):
    ('the' | 'these' | 'those' | themPhrase) indetPluralNounPhrase->np_
    :
;

modify grammar explicitDetPluralNounPhrase(definiteNumber):
    ('the' | 'these' | 'those' | themPhrase)
    numberPhrase->quant_ indetPluralNounPhrase->np_
    :
;

modify grammar explicitDetPluralOnlyNounPhrase(definite):
    ('the' | 'these' | 'those' | themPhrase) indetPluralOnlyNounPhrase->np_
    :
;

modify grammar explicitDetPluralOnlyNounPhrase(definiteNumber):
    ('the' | 'these' | 'those' | themPhrase)
    numberPhrase->quant_ indetPluralOnlyNounPhrase->np_
    :
;

modify grammar completeNounPhraseWithoutAll(it):
    'it' | 'this' | 'that'
    :
;

modify grammar completeNounPhraseWithoutAll(them):
    themPhrase | 'these' | 'those'
    :
;

modify grammar disambigPhrase(all):
    'all' ('of' themPhrase |) | 'everything'
    :
;

modify grammar disambigPhrase(any):
    'any' ('of' themPhrase |)
    :
;

modify grammar disambigPhrase(both):
    'both' ('of' themPhrase |)
    :
;

grammar themPhrase(main):
    'them' | '\'em' | 'em'
    : BasicProd
;

PreinitObject
    execute()
    {
        /*
         *   make the tokenizer recognize words that start with an
         *   apostrophe, but only if they are in the dictionary
         */
        cmdTokenizer.insertRule(['word with initial apostrophe',
                                 new RexPattern('<squote><alpha>*'),
                                 tokWord, nil, acceptApostWord],
                                 'word', nil);
    }
;

acceptApostWord(txt)
{
    /* look up the word, filtering out truncated results */
    return cmdDict.isWordDefined(
                       txt, { result: (result & StrCompTrunc) == 0 });
}

#endif

/* ---------------------------------------------------------------------- */
/*
 *   Understand THEREIN, THEREON, etc. These noun phrases are internally
 *   equivalent to ordinary pronouns, but have built-in prepositions in
 *   their grammar, so they must be explicitly included as alternatives in a
 *   verb rule where they are applicable.
 */

#ifdef MEGA_GRAMMAR_ON

grammar thereInPhrase(main):
    'therein'
    : ItProd
;

grammar thereOnPhrase(main):
    'thereon' | 'thereupon'
    : ItProd
;

grammar thereUnderPhrase(main):
    'thereunder'
    : ItProd
;

#endif

/* ---------------------------------------------------------------------- */
/*
 *   Extend the various commandPhrase productions so that we can add
 *   miscellaneous modifiers before or after a predicate. Also, add new
 *   synonyms for ASK/TELL ACTOR TO DO SOMETHING.
 *
 *   This part is not very elegant, and doesn't work fully (NPC orders are
 *   not recognized if they have an empty a beforeCommandPhrase), so it is
 *   turned off by default.
 */

#ifdef MEGA_COMMAND_ON

modify grammar commandPhrase(definiteConj):
    beforeCommandPhrase->before_
    predicate->cmd_
    afterCommandPhrase->after_
    (commandOnlyConjunction->conj_ * |)
    :

    getCommandSepIndex()
    {
        if (conj_ != nil)
            return conj_.firstTokenIndex;
        else
            return after_.lastTokenIndex + 1;
    }

    getNextCommandIndex()
    {
        if (conj_ != nil)
            return conj_.lastTokenIndex + 1;
        else if (after_.lastTokenIndex >= after_.firstTokenIndex)
            return after_.lastTokenIndex + 1;
        else
            return cmd_.lastTokenIndex + 1;
    }
;

modify grammar commandPhrase(ambiguousConj):
    beforeCommandPhrase->before1_
    predicate->cmd1_
    afterCommandPhrase->after1_
    commandOrNounConjunction->conj_
    beforeCommandPhrase->before2_
    predicate->cmd2_
    afterCommandPhrase->after2_
    :

    getNextCommandIndex()
    {
        if (before2_.lastTokenIndex >= before2_.firstTokenIndex)
            return before2_.firstTokenIndex;
        else
            return cmd2_.firstTokenIndex;
    }
;

modify grammar firstCommandPhrase(askTellActorTo):
    beforeCommandPhrase askTellPhrase singleNounOnly->actor_
    'to' commandPhrase->cmd_
    :
;

modify grammar actorBadCommandPhrase(main):
    singleNounOnly->actor_ ',' miscWordList
    | beforeCommandPhrase askTellPhrase singleNounOnly->actor_
        'to' miscWordList
    :
;

grammar askTellPhrase(main):
    'ask' | 'tell' | 'a' | 't'
    : BasicProd
;

grammar askTellPhrase(extended):
    'beg' | 'persuade' | 'convince' | 'command' | 'order' | 'instruct'
    : BasicProd
;

grammar beforeCommandPhrase(empty): () : BasicProd
;

grammar afterCommandPhrase(empty): () : BasicProd
;

#endif

/* ---------------------------------------------------------------------- */
/*
 *   Understand PLEASE DO SOMETHING; recognize and skip some common adverbs,
 *   plus some other things.
 */

#ifdef MEGA_COMMAND_ON

grammar beforeCommandPhrase(please):
    'please' | 'kindly'
    : BasicProd
;

grammar afterCommandPhrase(please):
    'please'
    : BasicProd
;

grammar afterCommandPhrase(misc):
    'anyway' | 'again' | 'instead'
    : BasicProd
;

grammar beforeCommandPhrase(adverb):
    adverbList
    : BasicProd
;

grammar afterCommandPhrase(adverb):
    adverbList
    : BasicProd
;

grammar adverbList(main):
    (adverbModifier |) singleAdverb (commandOrNounConjunction adverbList |)
    : BasicProd
;

grammar singleAdverb(main):
    'slowly' | 'quickly' | 'carefully'
    : BasicProd
;

grammar adverbModifier(main):
    'very' | 'more' | 'less'
    : BasicProd
;

#endif

/* ---------------------------------------------------------------------- */
/*
 *   Accept NOTHING and equivalent as a disambiguation response.
 */

#ifdef MEGA_GRAMMAR_ON

grammar disambigPhrase(nothing):
    nothingPhrase
    | 'neither' ('one' |)
    | 'none'
    : DisambigProd

    resolveNouns(resolver, results)
    {
        gLibMessages.abortQuestion();
        exit;
    }

    getResponseList() { return [self]; }
;

grammar nothingPhrase(main):
    'no' 'one'
    | 'nothing'
    | 'nobody'
    | 'never' 'mind'
    | 'cancel'
    : BasicProd
;

#endif

/* ---------------------------------------------------------------------- */
/*
 *   Move locational qualifiers to separate sub-productions and also make
 *   sure that we recognize UNDER and BEHIND in addition to IN and ON.
 */

#ifdef MEGA_LOCATIONAL_ON

modify grammar indetSingularNounPhrase(locational):
    nounPhrase->np_
    locationalPhrase->lp_
    completeNounPhraseWithoutAll->cont_
    :
;

modify grammar qualifiedSingularNounPhrase(theOneIn):
    'the' ('one' | '1')
    singularLocationalPhrase->lp_
    completeNounPhraseWithoutAll->cont_
    :
;

modify grammar qualifiedSingularNounPhrase(anyOneIn):
    ('anything' | 'one')
    singularLocationalPhrase->lp_
    completeNounPhraseWithoutAll->cont_
    :
;

modify grammar qualifiedPluralNounPhrase(theOnesIn):
    ('the' 'ones' pluralLocationalPhrase->lp_
     | ('everything' | 'all') singularLocationalPhrase->lp_)
    completeNounPhraseWithoutAll->cont_
    :
;

modify grammar indetPluralNounPhrase(locational):
    (pluralPhrase->np_ | adjPhrase->np_)
    locationalPhrase->lp_
    completeNounPhraseWithoutAll->cont_
    :
;

modify grammar indetPluralOnlyNounPhrase(locational):
    pluralPhrase->np_
    locationalPhrase->lp_
    completeNounPhraseWithoutAll->cont_
    :
;

class LocationalPhraseProd: BasicProd
    contType = (prep_.contType)
;

grammar locationalPhrase(main):
    (singularSubClause | pluralSubClause |)
    locationalPrepPhrase->prep_
    : LocationalPhraseProd
;

grammar singularLocationalPhrase(main):
    (singularSubClause |)
    locationalPrepPhrase->prep_
    : LocationalPhraseProd
;

grammar pluralLocationalPhrase(main):
    (pluralSubClause |)
    locationalPrepPhrase->prep_
    : LocationalPhraseProd
;

grammar locationalPrepPhrase(in):
    inPhrase
    : BasicProd

    contType = inContType
;

grammar locationalPrepPhrase(on):
    onPhrase
    : BasicProd

    contType = onContType
;

grammar locationalPrepPhrase(under):
    underPhrase
    : BasicProd

    contType = underContType
;

grammar locationalPrepPhrase(behind):
    behindPhrase
    : BasicProd

    contType = behindContType
;

grammar locationalPrepPhrase(generic):
    'from' | 'at' | 'by'
    : BasicProd

    contType = genericContType
;

grammar singularSubClause(main):
    ('that' | 'which' | 'who') ('is' | 'was')
    | ('that' | 'who') tokApostropheS
    : BasicProd
;

grammar pluralSubClause(main):
    ('that' | 'which' | 'who') ('are' | 'were')
    : BasicProd
;

/*
 *   Define the ContainmentType class and the five containment types that we
 *   understand by default. (For consistency, the class should really be
 *   defined elsewhere and modified here to include the objInPrep property,
 *   but the language-independent part of the library doesn't really need
 *   to know about this class. We could use enums just as well, except that
 *   we need to generate error messages that include a preposition which
 *   corresponds to the containment type given by the player in a command.)
 */
class ContainmentType: object
    objInPrep = ''
;

inContType: ContainmentType 'in';
onContType: ContainmentType 'on';
underContType: ContainmentType 'under';
behindContType: ContainmentType 'behind';
genericContType: ContainmentType 'by';

#endif

/* ---------------------------------------------------------------------- */
/*
 *   Add some additional directional phrasings: TO THE NORTH, NORTHWARDS and
 *   NORTH-EAST with a hyphen.
 */

#ifdef MEGA_GRAMMAR_ON

class NestedDirectionProd: DirectionProd
    dir = (dirMatch.dir)
;

grammar toDirectionName(main):
    'to' 'the' compassDirectionName->dirMatch
    | plainDirectionName->dirMatch
    : NestedDirectionProd
;

grammar plainDirectionName(main):
    compassDirectionName->dirMatch | directionName->dirMatch
    : NestedDirectionProd
;

#define DefLangCompassDir(root, dirNames) \
    grammar compassDirectionName(root): dirNames: DirectionProd \
        dir = root##Direction

DefLangCompassDir(north, 'north' | 'n');
DefLangCompassDir(south, 'south' | 's');
DefLangCompassDir(east, 'east' | 'e');
DefLangCompassDir(west, 'west' | 'w');
DefLangCompassDir(northeast, 'northeast' | 'north-east' | 'ne');
DefLangCompassDir(northwest, 'northwest' | 'north-west' | 'nw');
DefLangCompassDir(southeast, 'southeast' | 'south-east' | 'se');
DefLangCompassDir(southwest, 'southwest' | 'south-west' | 'sw');

#define ReDefLangDir(root, dirNames) \
    modify grammar directionName(root): dirNames:

ReDefLangDir(north, 'northward' | 'northwards');
ReDefLangDir(south, 'southward' | 'southwards');
ReDefLangDir(east, 'eastward' | 'eastwards');
ReDefLangDir(west, 'westward' | 'westwards');

ReDefLangDir(up, 'up' | 'u' | 'upward' | 'upwards');
ReDefLangDir(down, 'down' | 'd' | 'downward' | 'downwards');

#endif

/* ---------------------------------------------------------------------- */
/*
 *   Grammar for new actions.
 */

#ifdef MEGA_SAYTOPIC_ON

VerbRule(SayLiteral)
    ('say' | 'answer') singleLiteral
    : SayLiteralAction

    verbPhrase = 'say/saying (what)'
;

VerbRule(SayLiteralTo)
    ('say' | 'answer') singleLiteral 'to' singleDobj
    : SayLiteralToAction

    askDobjResponseProd = toSingleNoun

    verbPhrase = 'say/saying (what) (to whom)'
;

#endif

#ifdef MEGA_USE_ON

VerbRule(Use)
    usePhrase singleDobj
    : UseAction

    verbPhrase = 'use/using (what)'
;

VerbRule(UseWith)
    usePhrase singleDobj ('on' | 'with') singleIobj
    : UseWithAction

    verbPhrase = 'use/using (what) (on what)'
;

#endif

/* ---------------------------------------------------------------------- */
/*
 *   Define a couple of new grammar productions for common verbs and
 *   prepositional phrases that are used in several places. Besides saving
 *   us a lot of typing, these sub-productions also make parsing a bit more
 *   efficient, since we can cut down on the number of alternatives that
 *   need to be considered in many places.
 */

#ifdef MEGA_GRAMMAR_ON

/*
 *   These ones are needed only by the grammar extension.
 */
grammar inventoryPhrase(main):
    ('take' |) 'inventory' | 'inv' | 'i'
    : BasicProd
;

grammar lookPhrase(main):
    'l' | (('take' | 'have') 'a' |) 'look'
    : BasicProd
;

grammar pushTravelPhrase(main):
    'push' | 'pull' | 'drag' | 'move'
    : BasicProd
;

grammar putPhrase(main):
    'put' | 'place' | 'set' | 'drop'
    : BasicProd
;

grammar throwPhrase(main):
    'throw' | 'toss' | 'hurl' | 'fling'
    : BasicProd
;

grammar turnPhrase(main):
    'turn' | 'twist' | 'rotate'
    : BasicProd
;

grammar withPhrase(main):
    'with' | ('by' |) 'using' | 'by' 'means' 'of'
    : BasicProd
;

#endif

#if defined(MEGA_GRAMMAR_ON) || defined(MEGA_LOCATIONAL_ON)

/*
 *   These ones are needed both by the grammar component and the locational
 *   phrasings component.
 */
grammar behindPhrase(main):
    'behind' | 'at' 'the' 'back' 'of'
    : BasicProd
;

grammar inPhrase(main):
    'in' | 'inside' | 'inside' 'of'
    : BasicProd
;

grammar onPhrase(main):
    'on' | 'upon' | ('on' | 'at' 'the') 'top' 'of'
    : BasicProd
;

grammar underPhrase(main):
    'under' | 'underneath' | 'below' | 'beneath'
    : BasicProd
;

#endif

#if defined(MEGA_GRAMMAR_ON) || defined(MEGA_USE_ON)

/*
 *   This one is needed both by the grammar component and the generic use
 *   action component.
 */
grammar usePhrase(main):
    'use' | 'utilize' | 'utilise' | 'make' 'use' 'of' | 'employ' | 'try'
    : BasicProd
;

#endif

/* ---------------------------------------------------------------------- */
/*
 *   Understand a couple of additional prepositions.
 */

#ifdef MEGA_GRAMMAR_ON

modify grammar completeNounPhrase(miscPrep):
    [badness 100]
    completeNounPhrase->np1_
    (behindPhrase | inPhrase | onPhrase | underPhrase | withPhrase
        | 'into' | 'in' 'to' | 'through' | 'thru' | 'for' | 'to' | 'onto'
        | 'on' 'to' | 'at' | 'against' | 'towards' | 'over' | 'across')
    completeNounPhrase->np2_
    :
;

#endif

/* ---------------------------------------------------------------------- */
/*
 *   Miscellaneous new synonyms for exisiting verb rules, in alphabetical
 *   order.
 */

#ifdef MEGA_GRAMMAR_ON

modify VerbRule(AttackWith)
    ('attack' | 'kill' | 'hit' | 'punch' | 'strike') singleDobj
    withPhrase singleIobj
    :
;

modify VerbRule(ClimbDown)
    ('climb' | 'go' | 'walk') 'down' singleDobj
    | 'descend' singleDobj
    :
;

modify VerbRule(ClimbDownWhat)
    [badness 200]
    'climb' 'down' | 'descend'
    :
;

modify VerbRule(ClimbUp)
    ('climb' | 'go' | 'walk') 'up' singleDobj
    | ('ascend' | 'scale') singleDobj
    :
;

modify VerbRule(ClimbUpWhat)
    [badness 200]
    'climb' 'up' | 'ascend'
    :
;

modify VerbRule(Dig)
    ('dig' ('in' |) | 'excavate') singleDobj
    :
;

modify VerbRule(DigWith)
    ('dig' ('in' |) | 'excavate') singleDobj withPhrase singleIobj
    | usePhrase singleIobj ('in' 'order' |) 'to' 'dig' ('in' |) singleDobj
    | usePhrase singleDobj 'for' 'digging' ('in' |) singleDobj
    :
;

modify VerbRule(Doff)
    ('doff' | 'take' 'off' | 'shed' | 'disrobe') dobjList
    | 'take' dobjList 'off'
    :
;

modify VerbRule(Drop)
    ('drop' | 'discard' | ('put' | 'set') 'down') dobjList
    | ('put' | 'set') dobjList 'down'
    :
;

modify VerbRule(EnterOn)
    'enter' singleLiteral
    (('on' | 'in' | 'into' | 'in' 'to' | withPhrase) singleDobj
        | thereOnDobj)
    :
;

modify VerbRule(Examine)
    ('examine'
        | 'x'
        | lookPhrase 'at'
        | 'inspect'
        | 'check'
        | 'see'
        | 'view'
        | 'watch'
        | 'observe'
        | 'describe')
    dobjList
    :
;

modify VerbRule(GetOutOf)
    (('get' | 'climb' |) 'out' 'of'
        | 'depart' 'from'
        | 'leave'
        | 'exit')
    singleDobj
    :
;

modify VerbRule(GiveTo)
    ('give' | 'offer' | 'hand' ('over' |)) dobjList 'to' singleIobj
    | 'hand' dobjList 'over' 'to' singleIobj
    :
;

modify VerbRule(GiveToType2)
    ('give' | 'offer' | 'hand') singleIobj dobjList
    :
;

modify VerbRule(Inventory)
    inventoryPhrase
    :
;

modify VerbRule(InventoryTall)
    inventoryPhrase 'tall'
    :
;

modify VerbRule(InventoryWide)
    inventoryPhrase 'wide'
    :
;

modify VerbRule(Jump)
    'jump' | 'leap' | 'hop' | 'skip'
    :
;

modify VerbRule(JumpOff)
    ('jump' | 'leap' | 'hop') 'off' singleDobj
    :
;

modify VerbRule(JumpOffI)
    ('jump' | 'leap' | 'hop') 'off'
    :
;

modify VerbRule(JumpOver)
    ('jump' | 'leap' | 'hop') ('over' | 'across') singleDobj
    | 'jump' singleDobj
    :
;

modify VerbRule(LockWith)
    'lock' singleDobj withPhrase singleIobj
    | usePhrase singleIobj ('in' 'order' |) 'to' 'lock' singleDobj
    | usePhrase singleDobj 'for' 'locking' singleDobj
    :
;

modify VerbRule(Look)
    lookPhrase ('around' |)
    :
;

modify VerbRule(LookBehind)
    lookPhrase behindPhrase dobjList
    :
;

modify VerbRule(LookIn)
    lookPhrase (inPhrase dobjList | thereInDobj)
    :
;

modify VerbRule(LookThrough)
    lookPhrase ('through' | 'thru' | 'out' ('of' |)) dobjList
    :
;

modify VerbRule(LookUnder)
    lookPhrase underPhrase (dobjList | thereUnderDobj)
    :
;

modify VerbRule(MoveWith)
    'move' singleDobj withPhrase singleIobj
    :
;

modify VerbRule(Open)
    'open' ('up' |) dobjList
    :
;

modify VerbRule(Out)
    'exit' | 'leave' | 'depart'
    :
;

modify VerbRule(PushTravelClimbDown)
    pushTravelPhrase singleDobj 'down' singleIobj
    :
;

modify VerbRule(PushTravelClimbUp)
    pushTravelPhrase singleDobj 'up' singleIobj
    :
;

modify VerbRule(PushTravelDir)
    pushTravelPhrase singleDobj toSingleDir
    :
;

modify VerbRule(PushTravelEnter)
    pushTravelPhrase singleDobj ('in' | 'into' | 'in' 'to') singleIobj
    :
;

modify VerbRule(PushTravelGetOutOf)
    pushTravelPhrase singleDobj 'out' ('of' |) singleIobj
    :
;

modify VerbRule(PushTravelThrough)
    pushTravelPhrase singleDobj ('through' | 'thru') singleIobj
    :
;

modify VerbRule(PutBehind)
    putPhrase dobjList behindPhrase singleIobj
    :
;

modify VerbRule(PutIn)
    putPhrase dobjList
    ((inPhrase | 'into' | 'in' 'to') singleIobj | thereInIobj)
    :
;

modify VerbRule(PutOn)
    putPhrase dobjList
    ((onPhrase | 'onto' | 'on' 'to') singleIobj | thereOnIobj)
    :
;

modify VerbRule(PutUnder)
    putPhrase dobjList (underPhrase singleIobj | thereUnderIobj)
    :
;

modify VerbRule(Quit)
    ('quit' | 'end') (('the' | 'this' |) ('game' | 'story') |)
    | 'q'
    :
;

modify VerbRule(Read)
    ('read' ('through' | 'thru' |) | 'peruse') dobjList
    :
;

modify VerbRule(Restart)
    'restart' (('the' | 'this' |) ('game' | 'story') |)
    | 'start' 'over'
    | ('start' | 'restart') 'from' ('the' |) 'beginning'
    | 'undo' ('all' | 'everything')
    :
;

modify VerbRule(Search)
    'search' ('through' | 'thru' |) dobjList
    :
;

modify VerbRule(ShowTo)
    ('show' | 'display' | 'present') dobjList 'to' singleIobj
    :
;

modify VerbRule(ShowToWhom)
    ('show' | 'display' | 'present') dobjList
    :
;

modify VerbRule(Sit)
    'sit' ('down' |)
    | ('take' | 'have') 'a' 'seat'
    :
;

modify VerbRule(SitOn)
    ('sit' ('on' | 'in' | 'down' 'on' | 'down' 'in')
        | ('take' | 'have') 'a' 'seat' ('on' | 'in'))
    singleDobj
    :
;

modify VerbRule(Sleep)
    ('go' 'to' |) 'sleep'
    | (('take' | 'have') 'a' |) 'nap'
    :
;

modify VerbRule(Take)
    ('take' | 'get' | 'pick' 'up' | 'carry' | 'acquire') dobjList
    | 'pick' dobjList 'up'
    :
;

modify VerbRule(TalkTo)
    ('talk' 'to' | 'greet') singleDobj
    :
;

modify VerbRule(ThrowAt)
    throwPhrase dobjList ('at' | 'towards' | 'against') singleIobj
    :
;

modify VerbRule(ThrowDir)
    throwPhrase dobjList toSingleDir
    :
;

modify VerbRule(ThrowTo)
    throwPhrase dobjList 'to' singleIobj
    :
;

modify VerbRule(Travel)
    singleDir
    | ('go' | 'travel' | 'journey' | 'head') toSingleDir
    | 'continue' ('going' | 'to' 'go' |) toSingleDir
    | 'proceed' ('to' 'go') toSingleDir
    | 'keep' 'going' toSingleDir
    :
;

modify VerbRule(Turn)
    turnPhrase dobjList
    :
;

modify VerbRule(TurnWith)
    turnPhrase singleDobj withPhrase singleIobj
    :
;

modify VerbRule(UnlockWith)
    'unlock' singleDobj withPhrase singleIobj
    | usePhrase singleIobj ('in' 'order' |) 'to' 'unlock' singleDobj
    | usePhrase singleIobj 'for' 'unlocking' singleDobj
    :
;

modify VerbRule(VagueTravel)
    'go' | 'travel' | 'journey' | 'walk'
    | ('continue' | 'keep') 'going'
    :
;

#endif

/* ---------------------------------------------------------------------- */
/*
 *   The SpareWords class.
 */

#ifdef MEGA_SPARE_ON

class SpareWords: VocabObject
    /* populate the dictionary for this object */
    initializeVocab()
    {
        local done, cls, super;

        /*
         *   inherit the standard behaviour, in case we have a vocabWords
         *   property of our own
         */
        inherited();

        /* we need a vector of classes that we have visited */
        done = new Vector(64);

        /* go through the classes in our 'classes' property */
        foreach (cls in classes)
        {
            /* add all the vocabulary from this class to us */
            cls.inheritVocab(self, done);
        }

        /* go through the classes in our 'superClasses' property */
        foreach (super in superClasses)
        {
            /* go through all the subclasses of this superclass */
            for (cls = firstObj(super, ObjClasses) ; cls != nil ;
                 cls = nextObj(cls, super, ObjClasses))
            {
                /* add all the vocabulary from this class to us */
                cls.inheritVocab(self, done);
            }
        }

        /*
         *   Clear out all dictionary properties. These word lists could
         *   potentially become very large, but we have no real need for
         *   them, since our only task is to add words to the dictionary.
         */
        noun = nil;
        adjective = nil;
        plural = nil;
        adjApostS = nil;
        literalAdjective = nil;
        weakTokens = nil;
    }

    classes = []
    superClasses = []
;

#endif

