#charset "us-ascii"
#include "adv3.h"

/*

Tads-3 Containment Table -- ALPHA revision 1

Steve Breslin, 2004
email: versim@hotmail.com

====

Licence:

Everyone is free to use this, but please alert me to your findings, and
provide me with all improvements and modifications.

====

About this module:

This is a test module. Please profile any test cases both with and
without the module, and publicise your comparative findings.

The purpose of this module is to decrease the time required to
calculate the containment table. 

The idea is simple: (1) on preinit, we set up a table of contents for
each room, and (2) when a thing moves, we update the old room's and the
new room's containment table, if the object is moving between top-level
rooms. That way we don't have to recompute the containment table from
scratch (which currently happens at least a few times per turn).

The containment-table is calculated by the method
Thing.addDirectConnections(tab), so we modify this so it will use our
cached information rather than calculating everything from scratch.

====

Notes:

This module is largely untested. It is meant to work in all situations
or "room architectures" supported by the main library, but it's
entirely possible that you find a situation which this module doesn't
handle properly. Definitely please let me know!

While we are significantly decreasing the computation of
Thing.addDirectConnections(), we are increasing the computation
required by certain library methods, and introducing some of our own
methods. Overall, the time saved should be really significant, but it's
necessary when testing to consider the newly added complications.

It is possible to reorganize the library a bit more substantially to
further streamline the algorithm. But because this module is currently
designed for testing purposes, we're going for minimal restructuring of
existing library code.

Further streamlining is certainly possible, and I will be happy to do
more after we have some preliminary tests.

====

Thanks to Mike Roberts for his comments and analysis, which illuminated
and focused this project. And thanks very much to all who help test
this module. A warm thanks to the whole v-space group for tolerating
what probably seemed an endless brainstorm on this and related sense
optimization issues.

*/


// Fix a minor "bug" in the library:
modify ComplexComponent
    location = lexicalParent
;
// there may be more bugs like this with other Component objects: I
// haven't combed through this fully.

/* Thanks to Mike, we have a (now modified) method for incorporating
 * one table into another. This is a substantial part of the
 * calculation we're doing -- like over 1/4 of the time is spent in
 * this method. But I don't see any way to make it faster.
 */

modify LookupTable
    incorporateKeys(tab)
    {
        tab.forEachAssoc
        (
            new function(key, value)
            {
                self[key] = true; // we're ignoring values here
            }
        );
    }
;

/* We tried making containmentTable a vector instead (since values are
 * irrelevant), but as expected, this slowed things down very slightly
 * overall. Here's the method we were using:
 */
modify Vector
    elementsToKeys(table)
    {
        forEach(
            new function(value)
            {
                if (!table.isKeyPresent(value))
                    table[value] = true;
            }
        );
    }
;

/* We add roomPreinit the adv3LibPreinit.execBeforeMe */

modify adv3LibPreinit
    execBeforeMe = [mainOutputStream, statusTagOutputStream,
                    statusLeftOutputStream, statusRightOutputStream,
                    langMessageBuilder, roomPreinit]
;

/* We initialize a containment table and a vector of MultiLoc objects
 * for each room. These are populated during adv3LibPreinit.
 */
roomPreinit: PreinitObject
    execute()
    {
        for (local obj=firstObj(Room) ; obj ; obj=nextObj(obj, Room))
        {
            /* Note that we could optimize the bucket count and initial
             * capacity based on the number of objects in the game
             * compared with the number of rooms, or the number of
             * objects initially in the room plus some advisory number.
             * (For now, no optimization of this value.)
             */
            obj.containmentTable = new LookupTable();

            /* Note that we could optimize this vector based on the
             * number of MultiLoc objects currently in the room. (For
             * now, no such sophistication.)
             */
            obj.multiLocObjects = new Vector(4);
        }
    }
;

modify Room

    /* containmentTable is a table of objects currently in the room.
     *
     * It is initially populated by Thing.initializeLocation(), which
     * calls addToContents(), which in turn adds the thing to its
     * outermost room's containmentTable.
     * 
     * It is updated each time an object moves, via that object's
     * addToContents() and removeFromContents() methods. The contents
     * of a moved object are also called (recursively) to update this
     * this table.
     *
     * Note that the values are irrelevant; what matters is only the
     * presence of a key in the table.
     */
    containmentTable = nil

    /* multiLocObjects is a vector which records the MultiLoc
     * objects currently in the room.
     *
     * We are simplifying the addDirectConnections() method of
     * things with a single location, but we use the old method for
     * MultiLoc objects.
     */
    multiLocObjects = nil

    /* The current library's addDirectConnections(tab) is computation
     * heavy. To lighten it up, we maintain a containment table of
     * objects, so we don't have to recalculate this table upon each
     * (frequent) call.
     *
     * However, MultiLoc objects are tricky. Not only can they occur
     * in more than one room, but they can also connect rooms, thereby
     * connecting containment tables. (This is true not only of
     * sense connectors, which inherit from MultiLoc, but of MultiLoc
     * objects generally: e.g., if the player is inside a MultiLoc, the
     * player can see into all rooms the MultiLoc inhabits.) Therefore,
     * we take a conservative approach, allowing them to
     * addDirectConnections() exactly as they do in the main library.
     */
    addDirectConnections(tab)
    {

/* I assume that a Room is always its outermost location. So this test
 * should not be necessary:
 *      if (getOutermostRoom != self)
 *      {
 *          getOutermostRoom.addDirectConnections(tab);
 *          return;
 *      }
 */

        if (!tab.isKeyPresent(self))
        {
            tab.incorporateKeys(containmentTable);
        }
        foreach(local obj in multiLocObjects)
//--??      if (!tab.isKeyPresent(obj))
                obj.addDirectConnections(tab);
    }

    /* On pre-initialization, rooms add themselves and their room parts
     * to their containment table.
     */
    initializeThing()
    {
        inherited();
        containmentTable[self] = true;
        foreach(local obj in roomParts)
            containmentTable[obj] = true;
    }
;

modify Thing

    /* We seemed to be experiencing significant slowdown because of our
     * complication of removeFromContents() and addToContents(). To
     * alleviate this somewhat, we'll update the relevant containment
     * tables only when the object is moving between outermost rooms.
     */
    updateContainmentTable = nil // boolean flag

    notifyMoveInto(newContainer)
    {
        inherited(newContainer);

        /* we'll remove ourself (and children) from the
         * containmentTable if appropriate. Here we just note whether
         * or not this is appropriate.
         */
        if (location != nil
            && newContainer != nil
            && location.getOutermostRoom
               != newContainer.getOutermostRoom)
        {
            updateContainmentTable = true; // set the flag
        }
    }

    baseMoveInto(newContainer)
    {
        inherited(newContainer);

        /* In case we set the flag, cancel it after the move. */
        updateContainmentTable = nil;
    }

    /* On pre-initialization, Thing calls location.addToContents(self).
     * This is what populates the containment table of its outermost
     * room. So we turn on the flag so it will do so. (And we turn it
     * off when we're done.)
     */
    initializeThing()
    {
        updateContainmentTable = true;
        inherited();
        updateContainmentTable = nil;
    }

    /* removeFromContents() is a convenient hook for updating the
     * containmentTable of our outermost room.
     *
     * This will call the object's contents recursively, so all objects
     * inside the object will also be removed from the containment
     * table.
     *
     * Note that this is often unnecessary, since much of the time
     * an object moves within a single room, rather than between
     * rooms. Thus we first check the updateContainmentTable flag.
     */
    removeFromContents(obj)
    {
        inherited(obj);
        if (obj.updateContainmentTable)
            obj.removeFromContainmentTable(getOutermostRoom);
    }

    /* This is a service method for removing obj and all of its
     * containment children (recursively) from the room's containment
     * table.
     */
    removeFromContainmentTable(room)
    {
        /* MultiLoc objects are the outermost rooms of their
         * containment children. We want the contained child removed
         * from the containment tables of each room the MultiLoc
         * inhabits.
         */
        if (room.ofKind(MultiLoc))
        {
            foreach(local loc in room.locationList)
                removeFromContainmentTable(loc.getOutermostRoom);
        }
        /* If the room isn't a MultiLoc, we assume it is a "legitimate"
         * outermost room. We remove ourself and contents from our
         * outermost room's containment table, and add our collective
         * group to the room's table, if we have one.
         */
        else
        {
            local tab = room.containmentTable;
            tab.removeElement(self);
            foreach(local obj in contents)
                obj.removeFromContainmentTable(room);
        }
    }

    /* This is called during preinit, and at that time populates our
     * getOuterMostRoom.containmentTable. Thereafter it is called each
     * time the object moves into a new location, to add the obj and
     * contents (recursively) to the new outermost room's containment
     * table. 
     */
    addToContents(obj)
    {
        inherited(obj);
        if (obj.updateContainmentTable)
            obj.addToContainmentTable(getOutermostRoom);
    }

    /* This is a service method for adding obj and all of its
     * containment children (recursively) to the room's containment
     * table.
     */
    addToContainmentTable(room)
    {
        /* MultiLoc objects are the outermost rooms of their
         * containment children. We want the contained children added
         * to the containment table of each room the MultiLoc inhabits.
         */
        if (room.ofKind(MultiLoc))
        {
            foreach(local loc in room.locationList)
                addToContainmentTable(loc.getOutermostRoom);
        }

        /* If the room isn't a MultiLoc, we assume it is a "legitimate"
         * outermost room. We add ourself and contents to our outermost
         * room's containment table, and add our collective group to
         * the room's table, if we have one.
         */
        else
        {
            local cg = collectiveGroup,
                 tab = room.containmentTable;

            tab[self] = true;

            if (cg)
                tab[cg] = true;

            foreach (local obj in contents)
                obj.addToContainmentTable(room);
        }
    }

    /* We don't have to calculate our containment connections. Instead,
     * we let the outermost room handle the calculation.
     */
    addDirectConnections(tab)
    {
        getOutermostRoom.addDirectConnections(tab);
    }
;

modify MultiLoc

    /* When this object is removed from the containment table of one of
     * its rooms, it is also removed from the vector of multiLocObjects
     * of that room.
     */
    removeFromContainmentTable(room)
    {
        inherited(room);
        room.multiLocObjects.removeElement(self);
    }

    /* When this object is added to the containment table of a room, it
     * is also added to the vector of multiLocObjects of that room.
     */
    addToContainmentTable(room)
    {
        inherited(room);
        room.multiLocObjects.appendUnique(self);
    }
;

