/* gw_screen.c								*/

/*
 *               Copyright 1989, 1990 Digital Equipment Corporation
 *                          All Rights Reserved
 * 
 * 
 * Permission to use, copy, and modify this software and its documentation
 * is hereby granted only under the following terms and conditions.  Both
 * the above copyright notice and this permission notice must appear in
 * all copies of the software, derivative works or modified versions, and
 * any portions threof, and both notices must appear in supporting
 * documentation.
 * 
 * Users of this software agree to the terms and conditions set forth herein,
 * and hereby grant back to Digital a non-exclusive, unrestricted, royalty-free
 * right and license under any changes, enhancements or extensions made to the
 * core functions of the software, including but not limited to those affording
 * compatibility with other hardware or software environments, but excluding
 * applications which incorporate this software.  Users further agree to use
 * their best efforts to return to Digital any such changes, enhancements or
 * extensions that they make and inform Digital of noteworthy uses of this
 * software.  Correspondence should be provided to Digital at:
 * 
 *                       Director of Licensing
 *                       Western Research Laboratory
 *                       Digital Equipment Corporation
 *                       100 Hamilton Avenue
 *                       Palo Alto, California  94301  
 * 
 * Comments and bug reports may also be sent using electronic mail to:
 * 			screend-reports@decwrl.dec.com
 * 
 * 	>> This software may NOT be distributed to third parties. <<
 *   
 * THE SOFTWARE IS PROVIDED "AS IS" AND DIGITAL EQUIPMENT CORP. DISCLAIMS
 * ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS.   IN NO EVENT SHALL DIGITAL
 * EQUIPMENT CORPORATION BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

/************************************************************************
 *			Modification History				*
 *									*
 *	16 December 1988	Jeffrey Mogul/DECWRL			*
 *		Created.						*
 *									*
 ************************************************************************/

/*
 * Gateway Screening mechanism
 *
 *	This uses elevated IPL for synchronization.  For an SMP
 *	kernel, this should be changed; we can actually use the
 *	parallelism!
 */

#include "../h/param.h"
#include "../h/ioctl.h"
#include "../h/mbuf.h"
#include "../h/protosw.h"
#include "../h/socket.h"
#include "../h/socketvar.h"
#include "../h/uio.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/proc.h"
#include "../h/kernel.h"
#include "../h/systm.h"
#include "../net/if.h"
#include "../net/route.h"
#include "../net/af.h"
#include "../net/gw_screen.h"

int gw_screen_on = 0;		/* Screening on, if true */

struct screen_stats screen_stats;	/* statistics */

int screen_list_initialized = 0;

/*
 * Access to this facility should be by a new system call, but
 * to keep things simple, we use several new ioctls instead.
 * screen_control() is called from in_control().
 */

screen_control(so, cmd, data, ifp)
	struct socket *so;
	int cmd;
	caddr_t data;
	register struct ifnet *ifp;
{
	int old_screen_on, new_screen_on;
	struct screen_data *sdp = (struct screen_data *)data;

	if (!screen_list_initialized) {	/* first time only */
		int s = splimp();	/* XXX is this right? XXX */
		screen_init_freelist();
		(void)splx(s);
	}

	switch (cmd) {

	case SIOCSCREENON:
		/*
		 * Turns screening on/off (based on arg)
		 * and returns previous setting (value-result arg)
		 */
		new_screen_on = *((int *)data);
		old_screen_on = gw_screen_on;
		switch (new_screen_on) {
		    case SCREENMODE_OFF:
		    case SCREENMODE_ON:
			if (!suser())
			    return(u.u_error);
			gw_screen_on = new_screen_on;
			break;

		    case SCREENMODE_NOCHANGE:
		    default:
			/* no change */
			break;
		}

		/* return old value in any case */
		*((int *)data) = old_screen_on;
		break;

	case SIOCSCREEN:
		/*
		 * Transmits from user to kernel an screen_data
		 * struct, and then copies the next one from kernel
		 * to user into the same buffer (value-result arg).
		 * This allows us to do each transaction with
		 * only one system call.
		 *
		 * This ioctl blocks until the next transaction
		 * is available.
		 */

		if (!suser())
			return(u.u_error);

		if (!gw_screen_on) {
			return(ENOPROTOOPT);
		}

		/* Honor user's wishes on previous packet */
		screen_disposition(sdp->sd_xid, sdp->sd_action);
		
		/* Wait for next packet, create next transaction */
		screen_getnext(sdp);
		break;

	case SIOCSCREENSTATS:
		/*
		 * Provides current statistics block
		 */
		*(struct screen_stats *)data = screen_stats;
		break;
		
	default:
		printf("screen_control: unknown cmd %x\n", cmd);
		panic("screen_control");
	}

	return(0);
}

/*
 * This procedure is called, for example,
 *	instead of ip_forward() from ipintr().
 */
gw_forwardscreen(pkt, ifp, af, fwdproc, errorproc)
	struct mbuf *pkt;
	struct ifnet *ifp;
	int af;
	void (*fwdproc)();
	void (*errorproc)();
{
	if (gw_screen_on == SCREENMODE_OFF) {	/* not our problem */
		(*fwdproc)(pkt, ifp);
	}
	else {
		screen_bufenter(pkt, ifp, af, fwdproc, errorproc);
	}
}

/*
 * Unprocessed packets kept on a list
 */

#define	SCREEN_MAXPEND	32	/* default max # of packets pending */

int	screen_maxpend = SCREEN_MAXPEND;

struct screen_listentry {
	struct screen_listentry *next;	/* forward pointer */
	struct screen_listentry *prev;	/* backward pointer */
	int	xid;		/* transaction id */
	struct mbuf *pkt;		/* pointer to the packet */
	struct ifnet *ifp;	/* where it arrived on */
	int	pid;		/* "owning" process, if nonzero */
	struct timeval arrival;	/* arrival time for the packet */
	short	family;		/* address family */
	void	(*fwdproc)();	/* forwarding action (takes pkt, ifp) */
	void	(*errorproc)();	/* error action (takes pkt, ifp) */
};

struct screen_listentry screen_pending;		/* queue header */
struct screen_listentry screen_free;		/* free queue header */

u_long	screen_next_xid = 99;	/* next transaction id */

screen_init_freelist()
{
	register struct screen_listentry *sclp;
	register int i;
	register int allocsize = 
		screen_maxpend * sizeof(struct screen_listentry);

	screen_pending.next = &screen_pending;
	screen_pending.prev = &screen_pending;

	screen_free.next = &screen_free;
	screen_free.prev = &screen_free;

	KM_ALLOC(sclp, struct screen_listentry *, allocsize,
			KM_DEVBUF, KM_NOWAIT);

	if (sclp == NULL) {
		return;		/* we COULD try again later */
	}

	/* Chop up the memory and insert it on the free queue */
	for (i = 0; i < screen_maxpend; i++) {
		insque(sclp, &screen_free);
		sclp++;
	}
	screen_list_initialized = 1;
}

/*
 * Unprocessed packets go onto the pending list.  When the list
 * is full, we drop incoming packets (I think this is the "right
 * thing" to do, based on the way congestion control algorithms
 * work).  We wake up any available screening processes.
 */
screen_bufenter(pkt, ifp, family, fwdproc, errorproc)
struct mbuf *pkt;
struct ifnet *ifp;
int family;
void (*fwdproc)();
void (*errorproc)();
{
	register struct screen_listentry *sclp;
	int s;

	if (!screen_list_initialized)	/* first time only */
		screen_init_freelist();

	screen_stats.ss_packets++;

	s = splimp();	/* XXX is this right? XXX */

	/* get next free listentry */
	if ((sclp = screen_free.next) == &screen_free) {
	    (void)splx(s);
	    screen_stats.ss_nobuffer++;
	    m_freem(dtom(pkt));			/* drop packet */
	    wakeup((caddr_t)&screen_pending);	/* just in case */
	    return;
	}
	remque(sclp);
	
	sclp->xid = screen_next_xid++;
	sclp->pkt = pkt;
	sclp->ifp = ifp;
	sclp->pid = 0;
	sclp->arrival = time;
	sclp->family = family;
	sclp->fwdproc = fwdproc;
	sclp->errorproc = errorproc;

	/* link this on onto pending list */
	insque(sclp, &screen_pending);
	
	(void)splx(s);

	/* notify waiting screen processes */
	wakeup((caddr_t)&screen_pending);
}

/*
 * Block until something is there, then mark entry as "owned"
 * and return the contents.
 */
screen_getnext(sdp)
register struct screen_data *sdp;
{
	int s;
	register struct screen_listentry *sclp;
	register struct mbuf *m;
	register int remlen;
	register int len;
	register char *dp;

	while (1) {
	    s = splimp();	/* XXX is this right? XXX */
    
	    /* search buffer for un-owned entry, FIFO order */
	    sclp = screen_pending.prev;
	    while (sclp != &screen_pending) {
		/* if not claimed and family matches or is wildcarded */
		if (sclp->pid == 0
			&& ((sdp->sd_family == sclp->family)
				|| (sdp->sd_family == AF_UNSPEC)) ) {
		   goto found;
		}
		sclp = sclp->prev;
	    }

	    /* buffer is empty */
	    
	    screen_purgeold();	/* get rid of stale entries */
    
	    (void)splx(s);	/* don't sleep at high IPL */

	    sleep((caddr_t)&screen_pending, PUSER);
			    /* this sleep is interruptible */
	}

found:
	sclp->pid = u.u_procp->p_pid;
	sdp->sd_xid = sclp->xid;
	sdp->sd_arrival = sclp->arrival;
	sdp->sd_family = sclp->family;
	m = dtom(sclp->pkt);
	(void)splx(s);

	/* copy packet header into sd_data */
	remlen = SCREEN_DATALEN;
	dp = sdp->sd_data;
	while (m && (remlen > 0)) {
	    len = min(remlen, m->m_len);
	    bcopy(mtod(m, caddr_t), dp, len);
	    dp += len;
	    remlen -= len;
	    m = m->m_next;
	}
	
	sdp->sd_count = sizeof(struct screen_data);
	sdp->sd_dlen = dp - sdp->sd_data;
	sdp->sd_action = SCREEN_DROP|SCREEN_NONOTIFY;

	return;
}

screen_disposition(xid, action)
register u_long xid;
int action;
{
	int s;
	register struct screen_listentry *sclp;
	register int mypid = u.u_procp->p_pid;

	s = splimp();		/* XXX is this right? */

	/* search for our current transaction; flush stale ones */
	sclp = screen_pending.prev;
	while (sclp != &screen_pending) {
		if (sclp->pid == mypid) {
			remque(sclp);
			if (sclp->xid == xid) {
				/* match */
				goto found;
			}
			else {
				/* stale, flush it */
				m_freem(dtom(sclp->pkt));
				insque(sclp, &screen_free);
				screen_stats.ss_badsync++;
				sclp = screen_pending.prev;
					/* rescan is slow but simple */
			}
		}
		else
			sclp = sclp->prev;
	}
	(void)splx(s);
	return;		/* nothing to dispose of */

found:
	(void)splx(s);
	if (action & SCREEN_ACCEPT) {
	    screen_stats.ss_accept++;
	    (*(sclp->fwdproc))(sclp->pkt, sclp->ifp);
		    /* this frees sclp->pkt */
	} else {
	    /* this packet is rejected */
	    screen_stats.ss_reject++;
    
	    if (action & SCREEN_NOTIFY) {
		(*(sclp->errorproc))(sclp->pkt, sclp->ifp);
		/* this frees sclp->pkt */
	    }
	    else
		m_freem(dtom(sclp->pkt));
	}

	s = splimp();		/* XXX is this right? */
	insque(sclp, &screen_free);
	(void)splx(s);
}

#define	SCREEN_MAXAGE	5	/* default maximum age for packets */

int	screen_maxage = SCREEN_MAXAGE;

/*
 * Free up entries on the pending queue that are more than
 * screen_maxage seconds old.
 *
 * ASSUMPTION: called at high IPL (for synchronization)
 */
screen_purgeold()
{
	register int cutoff;
	register struct screen_listentry *sclp;
	
	/* Calculate oldest legal age; grace period of 1 sec for roundoff */
	cutoff = (time.tv_sec - screen_maxage) - 1;
	
	sclp = screen_pending.next;
	while (sclp != &screen_pending) {
		if (sclp->arrival.tv_sec < cutoff) {
			screen_stats.ss_stale++;
			remque(sclp);
			m_freem(dtom(sclp->pkt));
			insque(sclp, &screen_free);
			sclp = screen_pending.next;
		}
		else
			sclp = sclp->next;
	}
}
