#ifndef CHECKS
#define CHECKS

/*
** As TADS has been updated, it's gained a lot of capabilities.
** Unfortunately, many players are using older interpreters which don't
** support all of these nifty bells and whistles. This module checks the
** interpreter's version number, as well as whether the game is running
** on a HTML runtime and whether or not the interpreter can handle
** graphics. All results are saved in variables in the global object,
** and the module will optionally print an error message if it's running
** on an old version of the interpreter.
**
** There are also some helper functions. graphicsOn() returns true if
** the interpreter is capable of handling graphics and the user has the
** "graphics on" option selected. Ditto soundsOn() and musicOn().
**
** Copyright (c) 1999, Stephen Granade. All rights reserved.
**
** Version history:
**   19 Oct 99 -- Initial release
*/

/* Use C operators */
#pragma C+

/*
   So here's what you do:
     1. #include this file *after* you define the global object, which is
        normally in std.t or some variation thereof.
     2. In your own definition of the global object, set minVersionNumber to
        be the minimum interpreter version under which your game will run.
	It should be a string:
          minVersionNumber = '2.5'
     3. In either init() or (if you use it) commonInit(), place a call
        to
          getSysInfo();
	This will set the variables defined below in the "global" object
	and perform version checking by calling giveVersionWarning().
     4. If you don't like having a version warning printed at the start
        of a game, set dontPrintWarning to true in your global object:
	  dontPrintWarning = true
     5. If you want to use a different warning message than the one
        below, replace the giveVersionWarning() function. If you want
	the game to quit if the wrong interpreter is used, add
          terminate();
          quit();
	to the function.

   A suggestion: if you're using HTML mode (that is, you've printed "\H+"),
   then somewhere in your commonInit() function, put the following:
     if (global.isHTMLRuntime) {
         "<ABOUTBOX>Information about your game</ABOUTBOX>";
         "<TITLE>Title of your game</TITLE>";
     }
   This will set the text which is displayed in the title bar of many
   interpreters and the text which is printed when "About this game" is
   selected from the "Help" menu of HTML TADS interpreters.
*/


/* I'm going to assume that you define the global object in one of your
   own files. If you don't, change "modify global" to "global: object" */
modify global
    // dontPrintWarning = true    // Ignore version warning
    // minVersionNumber = '2.5'   // Our minimum version number
    versionNumber = ''          // Interpreter version we're running under
    isHTMLRuntime = true        // Are we running in an HTML runtime?
    canDoGraphics = true        // Set to true if our runtime can handle
                                //  graphics
    canDoSoundFX = true         // Ditto if runtime can do sound effects
    canDoMusic = true           // Ditto if runtime can do music
;

/* This version of giveVersionWarning() just prints a warning message. */
giveVersionWarning: function
{
    if (!global.dontPrintWarning &&
        compareVersionNumbers(global.versionNumber, global.minVersionNumber)
            == -1) {
	"\b\(WARNING: The TADS interpreter that you are using
	 is a version that is earlier than <<global.minVersionNumber>>.
	 Because of this, this game will not work as it should. I
	 recommend that you get an updated version of the TADS
	 interpreter from
         ftp://ftp.gmd.de/if-archive/programming/tads/executables.\)\b";
    }
}

/* Get system-dependent info, such as "what interpreter is this" and
   "is this an HTML-capable interpreter" &c. All information is stored
   in variables held in the "global" object. */
getSysInfo: function
{
    if (systemInfo(__SYSINFO_SYSINFO) == true) {
        global.isHTMLRuntime = (systemInfo(__SYSINFO_HTML) == 1);
        global.canDoGraphics = (systemInfo(__SYSINFO_JPEG) == 1) ||
            (systemInfo(__SYSINFO_PNG) == 1);// True if we can do PNG or JPEG
        global.canDoSoundFX = (systemInfo(__SYSINFO_MPEG_AUDIO) == 1) ||
            (systemInfo(__SYSINFO_WAV) == 1);
        global.canDoMusic = (systemInfo(__SYSINFO_MIDI) == 1);
	// Check our version number
	global.versionNumber = systemInfo(__SYSINFO_VERSION);
    }
    else {
        global.isHTMLRuntime = nil;         // This interpreter's not HTML
        global.canDoGraphics = nil;         // capable. Assume the version
        global.canDoSoundFX = nil;          // number is 2.2.3, as that's
        global.canDoMusic = nil;            // the last version which
	global.versionNumber = '2.2.3';     // lacks systemInfo()
    }

    giveVersionWarning();
}

/* graphicsOn() returns true if the user has the "graphics" option selected
   and the runtime is capable of showing graphics. */
graphicsOn: function
{
    return (global.canDoGraphics && systemInfo(__SYSINFO_PREF_IMAGES));
}

/* soundsOn() returns true if the user has the "sound effects" option
   selected and the runtime is capable of playing them */
soundsOn: function
{
    return (global.canDoSoundFX && systemInfo(__SYSINFO_PREF_SOUNDS));
}

/* musicOn() returns true if the user has the "music" options selected
   and the runtime is capable of playing music. */
musicOn: function
{
    return (global.canDoMusic && systemInfo(__SYSINFO_PREF_MUSIC));
}

/* Compare two version numbers (e.g. '2.2.6'), starting with the major
   number, then minor, then final. Returns -1 if the first number is less
   than the second, 0 if they are equal, or 1 if the first is greater */
compareVersionNumbers: function(one, two)
{
    local firstStr, secondStr, i, j, firstLoc, secondLoc;

    firstStr = one;
    secondStr = two;
    do {
        firstLoc = find(firstStr, '.');
        secondLoc = find(secondStr, '.');
        if (firstLoc) {
            i = cvtnum(substr(firstStr, 1, firstLoc - 1));
            firstStr = substr(firstStr, firstLoc + 1,
                              length(firstStr) - firstLoc);
        }
        else i = cvtnum(firstStr);
        if (secondLoc) {
            j = cvtnum(substr(secondStr, 1, secondLoc - 1));
            secondStr = substr(secondStr, secondLoc + 1,
                               length(secondStr) - secondLoc);
        }
        else j = cvtnum(secondStr);
        if (i < j) return -1;
        if (i > j) return 1;
    } while (firstLoc && secondLoc)
    if (firstLoc) return 1;   // This handles '2.5.1' vs '2.5'
    if (secondLoc) return -1; // while this handles '2.5' vs '2.5.1'
    return 0;
}

#endif /* CHECKS */
