/* copyright  1989 Phil Andrews, Pittsburgh Supercomputing Center */
#include <ctype.h>
#include <math.h>
#include <stdio.h>
#include "defs.h"
#define byte_size 8
#define out_ln(instr) {out_string(instr);}

/* module for EXCL specific functions, note that we expect the device
   resolution is pxl_per_in, set by the calling program. All dimensions are expected
   to come in terms of device pixels
*/   

/* useful macros */
#define max_int 65536
#define xpt(x) ((x>0) ? x : max_int - x)
#define word_size (sizeof(int) * byte_size)
#define start_comment "\033Q"
#define end_comment "\033R"
#define reset_printer "\033c"
#define set_excl "\033[0*s"
#define set_x_mode "\033[1;0r"
#define beg_copies "\033["
#define end_copies ";0u"
#define port_mode "\033[0p"
#define land_mode "\033[1p"
#define lower_letter "\033[0;3x"
#define absorb_forms "\033[1 z"
#define use_dots "\033[0y"
#define prop_spacing "\033[0w"
#define force_page "\033[*F"
#define x_polygon "\033P4;1;"
#define x_polyline "\033P5;1}"
#define poly_term "\033\\"
#define gs_char '\035'
#define x_carray "\033P3;0;"
#define ignore_nulls "\033[?60h"
/* for hex strings */
static hex_char[16] = {'0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
	'A', 'B', 'C', 'D', 'E', 'F'};

static int glbl_page;
static int pxl_per_in;
static char string_buffer[max_str + 1];
static int str_index;

/* store these pointers to allow lookups into the CGM data structures */
static struct one_opt 		*popt;	/* the command line options, in only */
static struct mf_d_struct 	*pc1;	/* the class 1 elements, in only */
static struct pic_d_struct 	*pc2;	/* the class 2 elements, in only */
static struct control_struct	*pc3;	/* the class 3 elements, in only */
static struct attrib_struct	*pc5;	/* the class 5 elements, in only */
static struct info_struct 	*dptr;	/* device info, in and out */

static int old_l_type;			/* line type presently set */
static int old_l_width;			/* line width presently set */
static int old_f_index;			/* fill index presently set */
static int state_level;
/* handy macros */
#define out_xvec(x,y) {out_xx(x);outc(':');out_xy(y);outc(';');}
static int out_xx(x)
int x;
{
	if (x<0) x = max_int - x;
	return(doint(x));
}
static int out_xy(y)
int y;
{
	y = (dptr->y_size) * dptr->ypxl_in - y;
	if (y<0) y = max_int - y;
	return(doint(y));
}

/* set the line type and width */
static void set_l_type(l_type, line_width)
int line_width;
enum line_enum l_type;
{
int dash_size, dot_size;

	if (line_width <= 0) line_width = 1;
	if ((int) l_type !=  old_l_type) {
	    old_l_type = (int) l_type;

	    dash_size = 20 * line_width;	/* guess */
	    dot_size = dash_size / 4;	 	/* guess */
	    if (dot_size <= 0) dot_size = 1;
	    if (dash_size <= dot_size) dash_size = 4 * dot_size;

	    out_string("\033[");	/* start command */
	    switch (l_type) {
case solid_l:	break;
case dash:	doint(dash_size); outc(';'); doint(dash_size);
		break;
case dot_l:	doint(dot_size); outc(';'); doint(dot_size);
		break;
case dash_dot:	doint(dash_size); outc(';'); doint(dot_size); outc(';');
		doint(dot_size); outc(';'); doint(dot_size); 
		break;

case dash_d_d: doint(dash_size); outc(';'); doint(dot_size); outc(';');
		doint(dot_size); outc(';'); doint(dot_size); outc(';');
		doint(dot_size); outc(';'); doint(dot_size); 
		break;

default:	fprintf(stderr, "unknown line type [%d]!\n", l_type);
	    }
	    out_string("*t");		/* end command */
	}
	if (line_width != old_l_width) {
	    if (line_width < 2) line_width = 2;	/* safety */
	    out_string("\033[");
	    doint(line_width);
	    outc(';');
	    doint(line_width);
	    out_string(";;;;;*w");	/* use defaults */
	    old_l_width = line_width;
	}

	return;
}
/* set the fill colour */
static set_fill(cptr)
struct rgbi_struct *cptr;
{
float grey_val, r, g, b;
int fill_num;
/* macro to convert from r, g, b to grey_scale, NTSC standard */
#define col_grey(r, g, b)  (1.0 - (0.3 * r + 0.59 * g + 0.11 * b))

	if (pc2->c_s_mode == i_c_mode) {	/* indexed */
	    r = pc5->ctab[cptr->ind * 3];
	    g = pc5->ctab[cptr->ind * 3 + 1];
	    b = pc5->ctab[cptr->ind * 3 + 2];
	}
	else if (pc2->c_s_mode == d_c_mode) {	/* direct colour */
	    r = cptr->red;
	    g = cptr->green;
	    r = cptr->blue;
	}

	grey_val = col_grey(r, g, b);
 	fill_num = 48 + grey_val * 64 + 0.5;
	if (fill_num < 0) fill_num = 0;
	if (fill_num == old_f_index) return(1);
	out_string("\033[");
	doint(fill_num);
	out_string(";5279;*r");		/* use grey-scale font */
	old_f_index = fill_num;

	return(1);

}
/* write Talaris style header and set up the printer */
excl_begin(comment, file_name, prog_name)
char *comment, *file_name, *prog_name;
{
long time(), now;
char *ctime(), *cuserid(), out_date[40];
int slen;
static char buffer[max_str];

	++state_level;
	if (state_level > 0) return(1);	/* already done everything */
	pxl_per_in = dptr->pxl_in;	/* might have been changed */

	out_string(reset_printer);
	out_string(set_excl);
	out_string(set_x_mode);
	out_string(beg_copies);
	doint(popt[(int) copies].val.i);
	out_string(end_copies);
	out_string(ignore_nulls);
	out_string(start_comment);
	out_string("Created by ");
	out_string(prog_name);
	out_string(" from ");
	out_string(file_name);
	out_ln(end_comment);
	out_string(start_comment);
	now = time(0);
	strcpy(out_date, ctime(&now));
	slen = strlen(out_date);
	*(out_date + ((slen) ? slen - 1 : 0) ) = 0;	/* eliminate ctrl-j */
	out_string(out_date);
	out_ln(end_comment);

	return(1);
}



/* do an EXCL trailer */
excl_end(pages_done)
int pages_done;
{
	--state_level;
	if (state_level >= 0) return(1);
	out_string(start_comment);
	out_string("End of Document ");
	out_ln(end_comment);
	out_ln(reset_printer);
	return(1);
}
/* start the page */
excl_bpage(pic_name, xoffset, yoffset, rotation, rb, gb, bb, page_no,
    xsize, ysize)
char *pic_name;
float rotation;		/* how much to rotate the page */
int xoffset, yoffset;	/* offset in pixels */
int page_no;		/* page number */
float rb, gb, bb;	/* background colour values */
int xsize, ysize;
{
int max_x, max_y;
char buffer[max_str];
	
	if (state_level > 0) return(1);	/* already done everything */
	old_l_type = -1;	/* reset line characteristics */
	old_l_width = -1;
	old_f_index = -1;
	glbl_page = page_no;

	out_string(start_comment);
	out_string(" Page Number: ");
	doint(page_no);
	out_string(" ");
	out_ln(end_comment);
	/* setup the printer */
	if ((rotation > 89.0) && (rotation < 91.0)) { /* can do it */
	    out_string(land_mode); 
	} else {
	    out_string(port_mode);
	}
	out_ln(lower_letter);	/* paper tray, size */
	out_string(absorb_forms);	/* absorb forms control */
	out_string(use_dots);		/* unit of measure is dots */
	/* now set the margins */
	/* first top and bottom margins */
	out_string("\033[");
	doint(0);
	outc(';');
	doint((int) ((dptr->y_size) * dptr->ypxl_in));
	out_string(" v");
	/* now left and right margins */
	out_string("\033[");
	doint(0);
	outc(';');
	doint((int) ((dptr->x_size) * dptr->pxl_in));
	outc('v');
	/* set proportional spacing */
	out_ln(prop_spacing);
	/* done for now */
	return(1);
}
/* end the page */
excl_epage(no_copies)
{
	
	if (state_level == 0) {
	    out_string(start_comment);
	    out_string(" End Page Number: ");
	    doint(glbl_page);
	    out_string(" ");
	    out_ln(end_comment);
	    out_ln(force_page);
	    fb();	/* empty line */
	}
	return(1);
}


/* actually output a string */
static send_string(in_string, str_l)
char *in_string;
int str_l;
{
int i;
char excl_string[4 * max_str + 1], buffer[10];
char *c_ptr, *excl_ptr;

	if (str_l > max_str) str_l = max_str;

	c_ptr = in_string;
	excl_ptr = excl_string;
	for (i=0; i<str_l; ++i) {
	    if ( iscntrl(*c_ptr) ) {
		sprintf(buffer, "\033[%d}", *c_ptr);
		sprintf(excl_ptr, "%s", buffer);
		excl_ptr += strlen(buffer);
	    }
	    else *excl_ptr++ = *c_ptr;
	    ++c_ptr;
	}
	*excl_ptr   = 0;
	out_string(excl_string);

	return(1);
}

/* plot a set of lines */
excl_pline(no_pairs, x1_ptr, y1_ptr)
int no_pairs, *x1_ptr, *y1_ptr;
{
int i;

	if ( no_pairs <= 1 ) return(1);

	set_l_type(pc5->line_type, pc5->line_width.i);

	out_string(x_polyline);
	for (i=0; i<no_pairs; ++i) out_xvec(x1_ptr[i], y1_ptr[i]);
	out_string(poly_term);
	

	return(1);
}
/* plot a set of lines between alternate points */
excl_dpline(no_pairs, x1_ptr, y1_ptr)
int no_pairs, *x1_ptr, *y1_ptr;
{
int i;

	if ( no_pairs <= 1 ) return(1);

	set_l_type(pc5->line_type, pc5->line_width.i);

	out_string(x_polyline);
	for (i=0; i<(no_pairs / 2); ++i) {
	    if (i) outc(gs_char);
	    out_xvec(x1_ptr[2 * i], y1_ptr[2 * i]);
	    out_xvec(x1_ptr[2 * i + 1], y1_ptr[2 * i + 1]);
	}
	out_string(poly_term);

	return(1);
}
/* do a polygon */
excl_pgon(no_pairs, x1_ptr, y1_ptr)
int no_pairs, *x1_ptr, *y1_ptr;
{
int i, want_edge;
	if ( no_pairs <= 1 ) return(1);
	want_edge = (pc5->edge_vis == on) || (pc5->int_style == hollow);
	set_fill(&pc5->fill_colour);

	out_string(x_polygon);
	if (want_edge) out_string("1;");
	else out_string("0;");
	/* do we fill it ? */
	switch (pc5->int_style) {
	case hollow :	out_string("0}"); break;
	case empty :	out_string("0}"); break;
			/* don't try to fill these */
	
	case pattern :	
		/* no pattern fill yet */
	case hatch :	
		/* no hatch fill yet */
	case solid_i :	out_string("1}"); break;
	}

	for (i=1;i<no_pairs;++i) out_xvec(x1_ptr[i], y1_ptr[i]);

	out_string(poly_term);
	return(1);
}

/* cgm cell array */
excl_carray(cp, cq, cr, nx, ny, col_prec, dat_ptr, rep_mode, no_bytes)
int *cp;	/* first corner */
int *cq;	/* the diagonal to cp */
int *cr;	/* first row is from cp to cr */
int nx;		/* number of elements in x rows (i.e., parallel to cp-cr) */
int ny;		/* number of rows */
int col_prec;	/* the colour precision (how many possible colour values */
char *dat_ptr;	/* pointer to the body of information */
int rep_mode;	/* the representation mode; 0, run length encoded, 1 normal */
int no_bytes;	/* number of data bytes */
/* without compression there should be ny * row_size bytes after dat_ptr, 
   the first nx * c_size bits of each row should contain information */
{
enum cs_enum c_s_mode;	/* the colour selection mode, indexed or direct */
unsigned bits_pixel, carray_atb, mode;
unsigned char *bit_ptr = NULL, *new_ptr = NULL, *outb, *old_ptr;
unsigned int new_bytes, i, j, row_size, bdone, c_size, new_size;
static unsigned char bit1_flag[byte_size] = {128, 64, 32, 16, 8, 4, 2, 1};
static unsigned char bit2_flag[byte_size] = {1, 2, 4, 8, 16, 32, 64, 128};
unsigned int x1, x2, y1, y2;
#define lctab_size 2
static float lctab[3 * lctab_size] = {1.0, 1.0, 1.0, 0.0, 0.0, 0.0};
int out_flag;
int my_flag = 0;
int want_rmode = 1;
int want_cp;
int want_csmode = 0;
int new_nx, new_ny;
int bytes_needed, bits_row, bytes_row, first_index;

	bits_pixel = 1;
	want_cp = 1;

	first_index =  (pc2->c_s_mode == d_c_mode) ? 1 : 0;
	/* the rows and columns we need */
	new_nx = (cr[0] > cp[0]) ? 1 + cr[0] - cp[0] : 1 + cp[0] - cr[0];
	new_ny = (cq[1] > cp[1]) ? 1 + cq[1] - cp[1] : 1 + cp[1] - cq[1];
	
	bits_row = new_nx * bits_pixel;
	bytes_row = (bits_row + byte_size - 1) / byte_size;
	bytes_needed = bytes_row * new_ny;

	if (!(new_ptr = (unsigned char *) allocate_mem(bytes_needed, 1)))
	    return(2);

/* lets make the cell array fit */
	new_carray(NULL, nx, ny, col_prec, rep_mode, 
	    0, dat_ptr, no_bytes, pc2->c_s_mode,
	    pc5->ctab, pc1->max_c_index, cp[0], cp[1], cq[0], cq[1],
	    cr[0], cr[1], new_nx, new_ny,
	    want_cp, want_rmode, bits_row, new_ptr, bytes_needed, 
	    want_csmode, lctab, lctab_size, my_flag, &out_flag, 
	    first_index, NULL, &(pc1->c_v_extent));

	/* at this point we are sure of a normal cell array in cgm format;
	   (except no silly row padding to even no. of bytes)
	   we may need to repack it for the uis routine */

	/* start the EXCL header */
	out_string(x_carray);
	out_xx(cp[0]);
	outc(';');
	out_xy(cp[1]);
	outc('}');
	/* now output the header */
	outb = new_ptr;
	for (i=0; i<new_ny; ++i) {
	    for (j=0; j<bytes_row; ++j) {
		outc(hex_char[*outb >> 4]);
		outc(hex_char[*outb & 15]);
		++outb;
	    }
	    out_string("\033C");   	/* mark end of row */
	}
	out_string("\033\\");

	(void) free(new_ptr);
	return(1);
}
/* set  a Circle */
excl_circle(x, y, r)
int x, y, r;
{
float d0, d1;

	set_fill(&pc5->fill_colour);
	out_string("\033[");
	out_xx(x);
	outc(';');
	out_xy(y);
	outc(';');
	out_xx(r);
	if (pc5->edge_vis == on) out_string(";1");
	else outc(';');
	if (pc5->int_style != hollow) out_string(";1");
	else out_string("*h");
	return(1);
}
/* set an arc, we get the positions of 1st pt, intermediate pt, end pt */
/* see utils.c for details */
excl_c3(pt_ptr)
int *pt_ptr;
{
int xc, yc, r;		/* circle centre and radius */
float deg[3];		/* starting and ending degrees (0 at top) */
double theta[3];
extern get_c_info();	/* utils.c */

	get_c_info(pt_ptr, &xc, &yc, &r, theta, deg);

	out_string("\033[");
	out_xx(xc);
	outc(';');
	out_xy(yc);
	outc(';');
	out_xx(r);
	outc(';');
	out_xx(pt_ptr[0]);
	outc(';');
	out_xy(pt_ptr[1]);
	outc(';');
	out_xx(pt_ptr[4]);
	outc(';');
	out_xy(pt_ptr[5]);
	out_string("*b");

	return(1);
}
/* set  a closed arc, get the positions of 1st pt, intermdiate pt, end pt */
excl_c3_close(pt_ptr, chord)
int *pt_ptr;
enum boolean chord;
{
int xc, yc, r;		/* circle centre and radius */
float deg[3];		/* starting and ending degrees (0 at top) */
double theta[3];
extern get_c_info();	/* utils.c */

	set_fill(&pc5->fill_colour);
	get_c_info(pt_ptr, &xc, &yc, &r, theta, deg);

	out_string("\033[");
	out_xx(xc);
	outc(';');
	out_xy(yc);
	outc(';');
	out_xx(r);
	outc(';');
	out_xx(pt_ptr[0]);
	outc(';');
	out_xy(pt_ptr[1]);
	outc(';');
	out_xx(pt_ptr[4]);
	outc(';');
	out_xy(pt_ptr[5]);
	outc(';');
	if (pc5->edge_vis == on) {
	    if (chord) out_string("3;");
	    else out_string("2;");
	} else out_string("0;");
	if (pc5->int_style == hollow) out_string("0*b");
	else {
	    if (chord) out_string("2*b");
	    else out_string("1*b");
	}
	return(1);
}
/* set an arc, ends specified by vectors */
excl_c_centre(x, y, vec_array, r)
int x, y, *vec_array, r;
{
float deg0, deg1;
double theta;
extern get_angle();	/* utils.c */
char buffer[max_str];

	get_angle(vec_array, &deg0, &theta);
	get_angle(vec_array + 2, &deg1, &theta);
	/* fix later */
	return(1);
}
/* set an arc, ends specified by vectors, but close it */
excl_c_c_close(x, y, vec_array, r, chord)
int x, y, *vec_array, r;
enum boolean chord;
{
float deg0, deg1;
double theta;
extern get_angle();	/* utils.c */

	get_angle(vec_array, &deg0, &theta);
	get_angle(vec_array + 2, &deg1, &theta);
	/* fix later */
	return(1);
}
/* the TeX setting routines */
excl_s_font(font_no)
int font_no;
{
char buffer[max_str];

	out_string("\033[");
	doint(font_no + 1);
	out_string(";0;1s");

	return(1);
}
/* set a rule */
excl_s_rule(h, v, w_pxl, h_pxl)
int h, v, w_pxl, h_pxl;
{

	out_string("\033[;");		/* use absolute postioning */
	out_xx(h);
	outc(';');
	out_xy(v + h_pxl - 1);
	outc(';');
	doint(w_pxl);
	outc(';');
	doint(h_pxl);
	outc(';');
	outc('|');

	return(1);
}
/* end the font definition */
excl_e_font(font_no, pk_file)
int font_no;
FILE *pk_file;
{
	/* output the whole file */
	rewind(pk_file);
	while (!feof(pk_file)) outc(fgetc(pk_file));
	out_string("\033\\");
	fb();

	return(1);
}
/* set both the absolute vertical and horizontal positions */
excl_abs_both(hpos, vpos)
int hpos, vpos;
{
char buffer[max_str];
	out_string("\033[");
	out_xy(vpos);
	outc(';');
	out_xx(hpos);
	outc('f');
	return(1);
}
/* set the absolute horizontal position */
excl_abs_h(pos)
int pos;
{
	out_string("\033[");
	out_xx(pos);
	outc('`');
	return(1);
}
/* set the absolute vertical position */
excl_abs_v(pos)
int pos;
{
	out_string("\033[");
	out_xy(pos);
	outc('d');
	return(1);
}
/* move horizontally, relatively */
excl_rel_h(shift)
int shift;
{
	if (shift == 0) return(1);
	out_string("\033[");
	if (shift > 0) {
	    if (shift > 1) doint(shift);
	    outc(';');
	    outc('a');
	} else {
	    doint(-shift);
	    outc(';');
	    outc('1');
	    outc('a');
	}
	return(1);
}
/* set a char */
excl_s_char(char_no)
int char_no;
{
char buffer[10];

	if (iscntrl(char_no)) {
	    sprintf(buffer, "\033[%d}", char_no);
	    out_string(buffer);
	} else outc(char_no);
	return(1);
}

/* start a font definition */
excl_st_font(font_no, finfo)
int font_no;
struct font *finfo;
{
	fb();
	out_string("\033P1;");
	doint(font_no + 1);
	out_string(";1;0}");

	return(1);
}
/* dummy routine */
excl_m_dev_char(in_ptr, w_bytes, height, width, char_no, 
	tfm_width, pxl_width, char_array, bytes_found, h_dots, v_dots)
unsigned char *in_ptr;
char *char_array;
int w_bytes, height, char_no, width, tfm_width,pxl_width,
*bytes_found, h_dots, v_dots;
{
	return(1);
}
/* suspend things gracefully */
excl_suspend()
{
	return(1);
}
/* and start things up again */
excl_restart()
{
	return(1);
}
/* this is the routine that sets everything up */
/* the initialising routine for the module */
void excl_setup(opt, dev_info, c1, c2, c3, c5, delim, mfdesc, pdesc, mfctrl,
	gprim, attr, escfun, extfun, ctrl, pargc, argv)

struct one_opt 		*opt;	/* the command line options, in only */
struct info_struct *dev_info;	/* device info to fill out, out only */
struct mf_d_struct 	*c1;	/* the class 1 elements, in only */
struct pic_d_struct 	*c2;	/* the class 2 elements, in only */
struct control_struct	*c3;	/* the class 3 elements, in only */
struct attrib_struct	*c5;	/* the class 5 elements, in only */
int (*delim[])();	/* delimiter functions */
int (*mfdesc[])();	/* metafile descriptor functions */
int (*pdesc[])();	/* page descriptor functions */
int (*mfctrl[])();	/* controller functions */
int (*gprim[])();	/* graphical primitives */
int (*attr[])();	/* the attribute functions */
int (*escfun[])();	/* the escape functions */
int (*extfun[])();	/* the external functions */
int (*ctrl[])();	/* controller functions */
int *pargc;			/* pointer to argc */
char **argv;			/* the main argv */	
{
	/* store the command line argument pointer */
	popt = opt;
	/* store the device info pointer */
	dptr = dev_info;
	/* and the cgm data strucures */
	pc1 = c1;
	pc2 = c2;
	pc3 = c3;
	pc5 = c5;
	/* fill out the device info structure */
	dev_info->pxl_in 	= 300.0;
	dev_info->ypxl_in 	= 300.0;
	dev_info->x_size 	= 8.0;
	dev_info->y_size 	= 10.5;
	dev_info->x_offset	= 0.25;
	dev_info->y_offset	= 0.25;
	dev_info->c_height	= 10 * dev_info->pxl_in / 72;	/* 10 pt */
	dev_info->c_width	= 10 * dev_info->pxl_in / 72;	/* 10 pt */
	dev_info->d_l_width	= 3;				
	dev_info->d_e_width	= 3;				
	dev_info->d_m_size	= dev_info->c_height;	/* marker size */
	strcpy(dev_info->out_name, ".x");
	dev_info->capability	= arb_rot + port_land + need_fix + no_cr;
	dev_info->rec_size	= 512;

	/* now fill out the function pointer arrays for CGM */
	/* the delimiter functions */
	delim[(int) B_Mf] 	= excl_begin;
	delim[(int) E_Mf]	= excl_end;
	delim[(int) B_Pic_Body]	= excl_bpage;
	delim[(int) E_Pic]	= excl_epage;

	/* the graphical primitives */
	gprim[(int) PolyLine]	= excl_pline;
	gprim[(int) Dis_Poly]	= excl_dpline;
	gprim[(int) Polygon]	= excl_pgon;
	gprim[(int) Cell_Array]	= excl_carray;
	gprim[(int) Cgm_Circle]	= excl_circle;
	gprim[(int) Circ_3]	= excl_c3;
	gprim[(int) Circ_3_Close]	= excl_c3_close;
/* 	fix later
	gprim[(int) Circ_Centre]	= excl_c_centre;
	gprim[(int) Circ_C_Close]	= excl_c_c_close;
*/

	state_level = -1;	/* just starting */
	return;
}

/* the following function optimises our TeX implementation and is 
   unnecessary for CGM support only */
void excl_gtex_setup(opt, gtex_calls)	/* function pointers, out only */
struct one_opt 		*opt;	/* the command line options, in only */
int (*gtex_calls[])();
{
	/* assign TeX-specific functions */
	gtex_calls[(int) S_Font] 	= excl_s_font;
	gtex_calls[(int) Suspend]	= excl_suspend;
	gtex_calls[(int) Restart]	= excl_restart;
	gtex_calls[(int) S_Rule]	= excl_s_rule;
	gtex_calls[(int) S_Abs_Both]	= excl_abs_both;
	gtex_calls[(int) S_Abs_H]	= excl_abs_h;
	gtex_calls[(int) S_Abs_V]	= excl_abs_v;
	gtex_calls[(int) M_Rel_H]	= excl_rel_h;
	gtex_calls[(int) S_Char]	= excl_s_char;
	gtex_calls[(int) St_Font]	= excl_st_font;
	gtex_calls[(int) M_Dev_Char]	= excl_m_dev_char;
	gtex_calls[(int) E_Font]	= excl_e_font;

	return;
}

