/*
 * last.c	Re-implementation of the 'last' command, this time
 *		for Linux. Yes I know there is BSD last, but I
 *		just felt like writing this. No thanks :-).
 *
 * Date:	08-12-1992
 *
 * Author:	Miquel van Smoorenburg, miquels@drinkel.nl.mugnet.org
 *
 */
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <stdio.h>
#include <utmp.h>
#include <errno.h>
#include <malloc.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>

#ifndef BTMP_FILE
#  define BTMP_FILE "/etc/btmp"
#endif

char *Version = "@(#) last 1.0 08-12-92 MvS";

/* Double linked list of struct utmp's */
struct utmplist {
  struct utmp ut;
  struct utmplist *next;
  struct utmplist *prev;
};
struct utmplist *utmplist = NULL;

/* Types of listing */
#define R_CRASH		1 /* No logout record, system boot in between */
#define R_DOWN		2 /* System brought down in decent way */
#define R_NORMAL	3 /* Normal */
#define R_NOW		4 /* Still logged in */

/* Global variables */
int maxrecs = 0;	/* Maximum number of records to list. */
int recsdone = 0;	/* Number of records listed */
int showhost = 0;	/* Show hostname too? */
char **show = NULL;	/* What do they want us to show */
char *ufile;		/* Filename of this file */
FILE *fp;		/* Filepointer of wtmp file */
static char lastdate[32]; /* Last date we've seen */

/* SIGINT handler */
void int_handler()
{
  printf("Interrupted %s\n", lastdate);
  exit(1);
}

/* SIGQUIT handler */
void quit_handler()
{
  printf("Interrupted %s\n", lastdate);
  signal(SIGQUIT, quit_handler);
}

/* Get the basename of a filename */
char *basename(char *s)
{
  char *p;

  if ((p = strrchr(s, '/')) != NULL)
	p++;
  else
	p = s;
  return(p);
}

/* Tell when the file was created */
void showtime()
{
  struct stat st;

  fstat(fileno(fp), &st);

  printf("\n%s begins %s", basename(ufile), ctime(&st.st_ctime));
}


/* Show one line of information on screen */
void list(struct utmp *p, time_t t, int what)
{
  char logintime[32];
  char logouttime[32];
  char length[32];
  time_t secs;
  int mins, hours, days;
  char **walk;

  /* Is this something we wanna show? */
  if (show) {
	for(walk = show; *walk; walk++) {
		if (strcmp(p->ut_name, *walk) == 0 ||
		    strcmp(p->ut_line, *walk) == 0 ||
		    strcmp(p->ut_line + 3, *walk) == 0) break;
	}
	if (*walk == NULL) return;
  }

  /* Calculate times */
  strcpy(logintime, ctime(&p->ut_time));
  logintime[16] = 0;
  strcpy(lastdate, logintime);
  strcpy(logouttime, ctime(&t) + 11);
  logouttime[5] = 0;
  secs = t - p->ut_time;
  mins   = (secs / 60) % 60;
  hours = (secs / 3600) % 24;
  days  = secs / 86400;
  if (days)
	sprintf(length, "(%d+%02d:%02d)", days, hours, mins);
  else
	sprintf(length, " (%02d:%02d)", hours, mins);

  switch(what) {
	case R_CRASH:
		sprintf(logouttime, "crash");
		break;
	case R_DOWN:
		sprintf(logouttime, "down ");
		break;
	case R_NOW:
		sprintf(length, "(still logged in)");
		break;
	case R_NORMAL:
		break;
  }

  if (showhost)
	printf("%-8s %-12s %-14s %s - %s %s\n", p->ut_name, p->ut_line,
		p->ut_host, logintime, logouttime, length);
  else
	printf("%-8s %-12s %s - %s %s\n", p->ut_name, p->ut_line,
		logintime, logouttime, length);
  recsdone++;
  if (maxrecs && recsdone >= maxrecs) {
	showtime();
	exit(0);
  }

}

/* show usage */
void usage(char *s)
{
  fprintf(stderr, "Usage: %s [-num] [-R] [username..] [tty..]\n", basename(s));
  exit(1);
}

int main(int argc, char **argv)
{
  struct utmp ut;	/* Current utmp entry */
  struct utmplist *p;	/* Pointer into utmplist */
  time_t lastboot = 0;  /* Last boottime */
  time_t lastdown;	/* Last downtime */
  int whydown = 0;	/* Why we went down: crash or shutdown */
  int x, y;		/* Scratch */

  /* Check the arguments */
  for(y = 1; y < argc; y++) {
	if (argv[y][0] == '-') {
		if (argv[y][1] == 0) usage(argv[0]);
		if ((x = atoi(argv[y] + 1)) <= 0) {
		    if (argv[y][2] != 0) usage(argv[0]);
		    switch(argv[y][1]) {
			case 'R':
				showhost++;
				break;
			default:
				usage(argv[0]);
				break;
		    }
		} else
			maxrecs = x;
	} else
		break;
  }
  if (y < argc) show = argv + y;

  /* Which file do we want to read? */
  if (strcmp(basename(argv[0]), "lastb") == 0)
	ufile = BTMP_FILE;
  else
	ufile = WTMP_FILE;
  time(&lastdown);

  /* Fill in 'lastdate' */
  strcpy(lastdate, ctime(&lastdown));
  lastdate[16] = 0;

  /* Install signal handlers */
  signal(SIGINT, int_handler);
  signal(SIGQUIT, quit_handler);

  /* Open the utmp file */
  if ((fp = fopen(ufile, "r")) == NULL) {
	fprintf(stderr, "last: %s: %s\n", ufile, sys_errlist[errno]);
	exit(1);
  }
  
  /* Go to end of file minus one structure */
  fseek(fp, -1L * sizeof(struct utmp), SEEK_END);

  /* Read struct after struct */
  while(fread(&ut, sizeof(struct utmp), 1, fp)) {
	switch (ut.ut_type) {
		case RUN_LVL:   /* Might be a shutdown record */
			if (strcmp(ut.ut_user, "shutdown") != 0)
				break;
			lastdown = ut.ut_time;
		case BOOT_TIME: /* Rebooted, processes might have crashed */
			if (ut.ut_type == BOOT_TIME) {
				sprintf(ut.ut_user, "reboot");
				sprintf(ut.ut_line, "system boot");
				list(&ut, lastdown, R_NORMAL);
				lastdown = ut.ut_time;
			}
			lastboot = ut.ut_time;
			whydown = (ut.ut_type == RUN_LVL) ? R_DOWN : R_CRASH;

			/* Delete the list; it's meaningless now */
			for(p = utmplist; p; p = p->next) free(p);
			utmplist = NULL;
			break;
		case DEAD_PROCESS: /* Process died; remember it */
			/* Get some memory */
			if ((p = malloc(sizeof(struct utmplist))) == NULL) {
				fprintf(stderr, "last: out of memory\n");
				exit(1);
			}
			/* Fill in structure */
			memcpy(&p->ut, &ut, sizeof(struct utmp));
			p->next  = utmplist;
			p->prev  = NULL;
			utmplist = p;
			break;
		case USER_PROCESS: /* Someone logged in */
			/* See if we crashed in the mean time */
			if (ut.ut_time < lastboot) {
				list(&ut, lastboot, whydown);
				break;
			}
			/* Walk through list to see when logged out */
			for(p = utmplist; p; p = p->next)
				if (p->ut.ut_pid == ut.ut_pid &&
				    p->ut.ut_type == DEAD_PROCESS) {

					/* Show it */
					list(&ut, p->ut.ut_time, R_NORMAL);
					/* Delete it from the list */
					if (p->next) p->next->prev = p->prev;
					if (p->prev)
						p->prev->next = p->next;
					else
						utmplist = p->next;
					free( p );
					break;
				}
			/* Not found? Then still logged in */
			if (p == NULL) list(&ut, time(NULL), R_NOW);
			break;
		default:
			break;
	}
	/* Position the file pointer 2 structures back */
	if (fseek(fp, -2 * sizeof(struct utmp), SEEK_CUR) < 0) break;
  }
  showtime();
  fclose(fp);
  /* Should we free memory here? Nah. */
  return(0);
}
