(*
$File: Compiler/CompileAndRun.sml $
$Date: 1992/09/17 14:17:46 $
$Revision: 1.1 $
$Locker:  $
*)

(* CompileAndRun: core language system only. *)

(*$CompileAndRun:
	DEC_GRAMMAR TOPDEC_GRAMMAR GRAMMAR_INFO RESIDENT
	RUN_LAMBDA RUNTIME COMPILER_DYNAMIC_BASIS REPORT FLAGS PRETTYPRINT
	FINMAP LAMBDA_EXP COMPILER_ENV COMPILE_DEC OPT_LAMBDA CRASH
	COMPILE_AND_RUN
 *)

functor CompileAndRun(structure DecGrammar: DEC_GRAMMAR

		      structure TopdecGrammar: TOPDEC_GRAMMAR
			sharing type TopdecGrammar.dec = DecGrammar.dec

		     (* MEMO: the INFO stuff is temporary. *)
		      structure GrammarInfo: GRAMMAR_INFO
			sharing type TopdecGrammar.info
				     = DecGrammar.info
				     = GrammarInfo.PostElabGrammarInfo

		      structure R: RESIDENT
		      structure FinMap: FINMAP

		      structure LambdaExp: LAMBDA_EXP
			sharing type LambdaExp.map = FinMap.map

		      structure CompilerEnv: COMPILER_ENV
			sharing type CompilerEnv.lvar = LambdaExp.lvar

		      structure CompileDec: COMPILE_DEC
			sharing type CompileDec.dec = DecGrammar.dec
			    and type CompileDec.LambdaExp = LambdaExp.LambdaExp
			    and type CompileDec.CEnv = CompilerEnv.CEnv

		      structure OptLambda: OPT_LAMBDA
			sharing type OptLambda.LambdaExp = LambdaExp.LambdaExp

		      structure Runtime: RUNTIME
			sharing type Runtime.Objects.lvar = CompilerEnv.lvar

		      structure DynamicBasis: COMPILER_DYNAMIC_BASIS
			sharing type DynamicBasis.CEnv = CompilerEnv.CEnv
			    and type DynamicBasis.DEnv = Runtime.DynamicEnv.DEnv

		      structure RunLambda: RUN_LAMBDA
			sharing type RunLambda.LambdaExp = LambdaExp.LambdaExp
			    and type RunLambda.object = Runtime.Objects.object
			    and type RunLambda.DEnv = Runtime.DynamicEnv.DEnv

		      structure Report: REPORT
		      structure Flags: FLAGS

		      structure PP: PRETTYPRINT
			sharing type DynamicBasis.StringTree
				     = LambdaExp.StringTree
				     = CompilerEnv.StringTree
				     = Runtime.Objects.StringTree
				     = PP.StringTree
			    and type PP.Report = Report.Report

		      structure Crash: CRASH
		     ): COMPILE_AND_RUN =
  struct
    type DynamicBasis = DynamicBasis.Basis
    type topdec = TopdecGrammar.topdec

    open Runtime

    val pr = Report.print o PP.reportStringTree

    fun display(title, tree) =
      if Flags.DEBUG_COMPILER then
	pr(PP.NODE{start=title ^ ": ",
		   finish="",
		   indent=3,
		   children=[tree],
		   childsep=PP.NONE
		  }
	  )
      else ()

    fun printCEnv ce =
      display("CompileAndRun.CEnv", CompilerEnv.layoutCEnv ce)

   (* MEMO: The info stuff is temporary; we need it at present to reconstruct
      LOCAL and SEQ abstract syntax. *)

    val i = GrammarInfo.emptyPostElabGrammarInfo

   (* How the hell are we going to model EVALTOPDEC's notion of exception
      packet? Let's see what we can do... *)

    type Pack = Runtime.Objects.object
    val pr_Pack = RunLambda.printPacket
    val RE_RAISE = RunLambda.RE_RAISE
    val FAIL_USE = RunLambda.FAIL_USE

   (* MEMO: this stuff is rather temporary. Once we believe in the modules
      language it will get sorted out, but until then we need to be able
      to handle the core language plus SEQ and LOCAL declarations (which the
      parser identifies as module-level constructs). *)

    fun evalDecFromBasis(b, dec) =
      let
	val (ce, de) = DynamicBasis.deBasis b
	val (ce1, lambFn) = CompileDec.compileDec ce dec
	val _ = printCEnv ce1
	val lvars = CompilerEnv.lvarsOfCEnv ce1
	val scope = LambdaExp.VECTOR(map LambdaExp.VAR lvars)
	val lamb = lambFn scope

	val layoutLambdaExp = LambdaExp.layoutLambdaExp
	val layoutObject = Objects.layoutObject

	val _ = display("UnOpt", layoutLambdaExp lamb)

	val lamb = OptLambda.optimise lamb

	val _ = display("Opt", layoutLambdaExp lamb)

	val result = RunLambda.run de lamb

	val _ = display("Result", layoutObject result)

	fun bind(n, lv :: lvs, vec) =
	      DynamicEnv.declare(lv, Objects.select(n, vec),
				 bind(n+1, lvs, vec)
				)
	  | bind(_, nil, _) =
	      DynamicEnv.emptyDEnv
      in
	DynamicBasis.mkBasis(ce1, bind(0, lvars, result))
      end

    local
      open DecGrammar
      open TopdecGrammar
      open DynamicBasis
    in
      datatype result = SUCCESS of DynamicBasis
                      | FAILURE of string

     (* evalStrdec is a hack - it interprets sequential/local declarations in a
	rather ad-hoc way. This is because we need to deal with sequential
	and local declarations for the core language, and we haven't done the
	modules compiler yet (the parser treats them as module syntax). *)

     (* The technique we use (assuming it works...) is to push LOCAL and
        SEQ constructs from the modules level of the abstract syntax to the
	core level. *)

      fun pushLocalSeq strdec =
	case strdec
	  of LOCALstrdec(_, strdec1, strdec2) =>
	       let
		 val dec1 = decOfStrdec(pushLocalSeq strdec1)
		 val dec2 = decOfStrdec(pushLocalSeq strdec2)
	       in
		 DECstrdec(i, LOCALdec(i, dec1, dec2))
	       end

	   | SEQstrdec(_, strdec1, strdec2) =>
	       let
		 val dec1 = decOfStrdec(pushLocalSeq strdec1)
		 val dec2 = decOfStrdec(pushLocalSeq strdec2)
	       in
		 DECstrdec(i, SEQdec(i, dec1, dec2))
	       end

	   | _ => strdec

      and decOfStrdec(DECstrdec(_, dec)) = dec
	| decOfStrdec _ = Crash.impossible "CompileAndRun.decOfStrdec"

      exception UNCAUGHT = RunLambda.UNCAUGHT

      fun evalStrdec(basis, strdec) =
	case pushLocalSeq strdec
	  of DECstrdec(_, dec) =>
	       evalDecFromBasis(basis, dec)

	   | _ =>
	       Crash.unimplemented "evalStrdec"

      fun eval(basis, topdec) =
	case topdec
	  of STRtopdec(_, strdec) =>
	       evalStrdec(basis, strdec)

	   | SIGtopdec _ =>
	       Crash.unimplemented "CompileAndRun.eval(SIGtopdec)"

	   | FUNtopdec _ =>
	       Crash.unimplemented "CompileAndRun.eval(FUNtopdec)"

    end

    type StringTree = PP.StringTree
    val layoutDynamicBasis = DynamicBasis.layoutBasis
  end;
