(*
$File: Compiler/LambdaExp.sml $
$Date: 1992/09/17 14:17:56 $
$Revision: 1.1 $
$Locker:  $
*)

(*$LambdaExp: LVARS FINMAP PRETTYPRINT CRASH LAMBDA_EXP*)

functor LambdaExp(structure Lvars: LVARS
		  structure FinMap: FINMAP

		  structure PP: PRETTYPRINT
		    sharing type PP.StringTree = FinMap.StringTree

		  structure Crash: CRASH
		 ): LAMBDA_EXP =
  struct
    type lvar = Lvars.lvar
    type (''a, 'b) map = (''a, 'b) FinMap.map

    datatype LambdaExp =
        VAR      of lvar		(* lambda variables. *)
      | INTEGER  of int			(* constants... *)
      | STRING   of string
      | REAL     of real
      | FN       of lvar * LambdaExp	(* function-terms. *)
      | FIX      of lvar list * LambdaExp list * LambdaExp
					(* mutual recursive fns. *)
      | APP      of LambdaExp * LambdaExp	(* function application. *)
      | PRIM_APP of int * LambdaExp	(* primitive function application. *)
      | VECTOR   of LambdaExp list	(* records/tuples. *)
      | SELECT   of int * LambdaExp	(* con/record indexing. *)
      | SWITCH_I of int Switch		(* switch on integers. *)
      | SWITCH_S of string Switch	(* ...strings *)
      | SWITCH_R of real Switch		(* ...reals *)
      | RAISE    of LambdaExp		(* raise exception *)
      | HANDLE   of LambdaExp * LambdaExp	(* exception handling. *)
      | REF	 of LambdaExp		(* ref(expr) *)
      | VOID

    and 'a Switch = SWITCH of {arg: LambdaExp,
			       selections: ('a, LambdaExp) map,
			       wildcard: LambdaExp Option
			      }

   (* some convenient shorthand: *)
    fun pair(lexp1, lexp2) = VECTOR [lexp1, lexp2]
    fun first lamb = SELECT(0, lamb)
    fun second lamb = SELECT(1, lamb)
    fun Let((lhs, rhs), scope) = APP(FN(lhs, scope), rhs)

   (* prettyprinting. *)
    type StringTree = PP.StringTree

    fun layoutSwitch(name, layoutLambdaExp, pr,
		     (SWITCH{arg, selections, wildcard})
		    ) =
      let
	val selections =
	  FinMap.layoutMap {start="", eq=" -> ", sep="; ", finish=""}
	                   (PP.layoutAtom pr)
			   layoutLambdaExp
			   selections
	val wildcard_L =
	  case wildcard
	    of None => nil
	     | Some w =>
		 [PP.NODE{start="_ -> ", finish="", indent=3, childsep=PP.NONE,
			  children=[layoutLambdaExp w]
			 }
		 ]
      in
	PP.NODE{start=name ^ "(", finish=")", indent=3,
		children=[layoutLambdaExp arg,
			  PP.NODE{start="", finish="", indent=3,
				  children=selections :: wildcard_L,
				  childsep=PP.RIGHT "; "
				 }
			 ],
		childsep=PP.RIGHT ": "
	       }
      end

    fun layoutLambdaExp lamb: StringTree =
      case lamb
	of VAR lv => PP.LEAF(Lvars.pr_lvar lv)
         | INTEGER x => PP.LEAF(Int.string x)
	 | STRING x => PP.LEAF(String.string x)
	 | REAL x => PP.LEAF(Real.string x)

	 | FN(lv, lamb) =>
	     PP.NODE{start="FN " ^ Lvars.pr_lvar lv ^ ". ",
		     finish="", indent=3,
		     children=[layoutLambdaExp lamb], childsep=PP.NONE
		    }

	 | FIX(lvs, binds, scope) =>
	     let
	       fun child(lv, bind) =
		 PP.NODE{start=Lvars.pr_lvar lv ^ " = ", finish="", indent=3,
			 children=[layoutLambdaExp bind], childsep=PP.NONE
			}

	       val children = map child (ListPair.zip(lvs, binds))
	     in
	       PP.NODE{start="FIX(", finish=")", indent=4,
		       childsep=PP.LEFT " in ",
		       children=[PP.NODE{start="", finish="", indent=0,
					 children=children,
					 childsep=PP.RIGHT ", "
					},
				 layoutLambdaExp scope
				]
		      }
	     end

	(* APP(FN(lv, scope), bind) is printed as LET(lv = bind in scope);
	   but, we want to be cleverer than that, in order to catch
	   a series of nested LET's, otherwise the prettyprint tree gets
	   too deep. *)

	 | APP(FN _, _) =>
	     PP.NODE{start="LET(", finish=")", indent=4,
		     childsep=PP.RIGHT ": ",
		     children=layoutLambdaLETs lamb
		    }

	 | APP(lamb1, lamb2) =>
	     PP.NODE{start="APP(", finish=")", indent=4,
		     children=[layoutLambdaExp lamb1, layoutLambdaExp lamb2],
		     childsep=PP.RIGHT ", "
		    }

	 | PRIM_APP(n, lamb) =>
	     PP.NODE{start="PRIM_APP(", finish=")", indent=3,
		     childsep=PP.RIGHT ", ",
		     children=[PP.layoutAtom Int.string n,
			       layoutLambdaExp lamb
			      ]
		    }

	 | VECTOR list =>
	     PP.NODE{start="VECTOR(", finish=")", indent=3,
		     children=map layoutLambdaExp list, childsep=PP.RIGHT ", "
		    }

	 | SELECT(int, lamb) =>
	     PP.NODE{start="SELECT(" ^ Int.string int ^ ", ",
		     finish=")", indent=3,
		     children=[layoutLambdaExp lamb], childsep=PP.NONE
		    }

	 | SWITCH_I sw =>
	     layoutSwitch("SWITCH_I", layoutLambdaExp, Int.string, sw)

	 | SWITCH_S sw =>
	     layoutSwitch("SWITCH_S", layoutLambdaExp, String.string, sw)

	 | SWITCH_R sw =>
	     layoutSwitch("SWITCH_R", layoutLambdaExp, Real.string, sw)

	 | RAISE lamb =>
	     PP.NODE{start="RAISE(", finish=")", indent=3,
		     children=[layoutLambdaExp lamb], childsep=PP.NONE
		    }

	 | HANDLE(lamb1, lamb2) =>
	     PP.NODE{start="HANDLE(", finish=")", indent=3,
		     children=[layoutLambdaExp lamb1, layoutLambdaExp lamb2],
		     childsep=PP.LEFT " with "
		    }

	 | REF lamb =>
	     PP.NODE{start="REF(", finish=")", indent=3,
		     children=[layoutLambdaExp lamb], childsep=PP.NONE
		    }

	 | VOID => PP.LEAF "VOID"

   (* layoutLambdaLETs - traverse a series of APP((FN _), _) lambdas and
      print out as a sequence `lv1 = lamb1: lv2 = lamb2: ....', with all
      the bindings at the same indent level. *)

    and layoutLambdaLETs lamb: StringTree list =
      case lamb
	of APP(FN(lv, scope), bind) =>
	     PP.NODE{start=Lvars.pr_lvar lv ^ " = ", finish="",
		     indent=0, childsep=PP.NONE,
		     children=[layoutLambdaExp bind]
		    } :: layoutLambdaLETs scope
	 | _ =>
	     [layoutLambdaExp lamb]
  end;
