(*
$File: Compiler/Runtime.sml $
$Date: 1992/09/17 14:18:04 $
$Revision: 1.1 $
$Locker:  $
*)

(*$Runtime:
	LVARS LAMBDA_EXP FINMAP
	PRETTYPRINT CRASH RUNTIME OBJECTS DYNAMIC_ENV
 *)

functor Runtime(structure Lvars: LVARS
		structure LambdaExp: LAMBDA_EXP
		structure FinMap: FINMAP

		structure PP: PRETTYPRINT
		  sharing type FinMap.StringTree = PP.StringTree

		structure Crash: CRASH
	       ): RUNTIME =
  struct
   (* types `object' and `DEnv' are mutually recursive, so they're here
      outside the substructures. *)

    datatype object = INTEGER of int
		    | STRING of string
                    | REAL of real
		    | CLOSURE of {arg: Lvars.lvar,
				  body: LambdaExp.LambdaExp,
				  bodyEnv: DEnv,
				  recEnv: DEnv
				 }
		    | VECTOR of object list
		    | REF of int * object
		    | VOID

    withtype DEnv = (Lvars.lvar, object) FinMap.map

    local
      val n = ref 0
    in
      fun stamp() = (n := !n + 1; !n)
    end

    structure Objects: OBJECTS =
      struct
	type lvar = Lvars.lvar
	type LambdaExp = LambdaExp.LambdaExp
	type object = object
	type DEnv = DEnv

	val void = VOID

	val integer = INTEGER
	fun deInteger(INTEGER x) = x
	  | deInteger _ = Crash.impossible "Objects.deInteger"

	val real = REAL
	fun deReal(REAL x) = x
	  | deReal _ = Crash.impossible "Objects.deReal"

	val string = STRING
	fun deString(STRING x) = x
	  | deString _ = Crash.impossible "Objects.deString"

	fun closure{arg, body, bodyEnv} =
	  CLOSURE{arg=arg, body=body, bodyEnv=bodyEnv, recEnv=FinMap.empty}

	fun deClosure(CLOSURE c) = c
	  | deClosure _ = Crash.impossible "Objects.deClosure"

	val vector = VECTOR

	fun select(n, VECTOR x) = List.nth n x
	  | select _ = Crash.impossible "Objects.select"

	fun Ref obj = REF(stamp(), obj)

	fun deRef(REF(_, obj)) = obj
	  | deRef _ = Crash.impossible "Objects.deRef"

	fun equal(obj1, obj2) =
	  case (obj1, obj2)
	    of (INTEGER i1, INTEGER i2) => (i1 = i2)
	     | (STRING s1, STRING s2) => (s1 = s2)
	     | (REAL r1, REAL r2) => (r1 = r2)

	     | (CLOSURE _, CLOSURE _) =>
		 Crash.impossible "Objects.equal(CLOSURES)"

	     | (VECTOR list1, VECTOR list2) => equalList(list1, list2)
	     | (REF(stamp1, _), REF(stamp2, _)) => (stamp1 = stamp2)

	     | (VOID, VOID) => true

	     | _ => Crash.impossible "Objects.equal(different kinds)"

	and equalList(obj1 :: rest1, obj2 :: rest2) =
	      equal(obj1, obj2) andalso equalList(rest1, rest2)

	  | equalList(nil, nil) = true
	  | equalList _ = false

	type StringTree = PP.StringTree
	fun layoutObject obj =
	  case obj
	    of INTEGER x => PP.LEAF(Int.string x)
	     | STRING x => PP.LEAF(String.string x)
	     | REAL x => PP.LEAF(Real.string x)
	     | CLOSURE _ => PP.LEAF "fn"

	     | VECTOR v =>
		 PP.NODE{start="<", finish=">", indent=1,
			 childsep=PP.RIGHT "; ",
			 children=map layoutObject v
			}

	     | REF(n, obj) =>
		 PP.NODE{start="(ref[#" ^ Int.string n ^ "] ",
			 finish=")", indent=3, childsep=PP.NONE,
			 children=[layoutObject obj]
			}

	     | VOID => PP.LEAF "(void)"
      end

    structure DynamicEnv: DYNAMIC_ENV =
      struct
	type lvar = Lvars.lvar
	type object = object
	type DEnv = DEnv

	val emptyDEnv = FinMap.empty
	val declare = FinMap.add
	val plus = FinMap.plus

	fun REC env =
	  let
	    fun unwind(CLOSURE{arg, body, bodyEnv, recEnv=_}) =
	      CLOSURE{arg=arg, body=body, bodyEnv=bodyEnv, recEnv=env}
	      | unwind obj = obj		(* Can this ever happen? *)
	  in
	    FinMap.composemap unwind env
	  end

	fun lookup re lvar =
	  case FinMap.lookup re lvar
	    of Some obj => obj
	     | None =>
		 Crash.impossible("DynamicEnv.lookup " ^ Lvars.pr_lvar lvar)

	type StringTree = PP.StringTree
	val layoutDEnv =
	  FinMap.layoutMap {start="{", eq=" -> ", sep=", ", finish="}"}
	  		   (PP.layoutAtom Lvars.pr_lvar)
			   Objects.layoutObject
      end
  end;
