(* The interface to the parser that the rest of the Kit Compiler sees. We hide
   all the ugly MLLex and MLYacc implementation details here. Parse() depends
   on the base support modules for MLYacc (MyBase.sml), and my support functors
   - these are the functors that are referred to freely. *)

(*
$File: Parsing/Parse.sml $
$Date: 1992/04/07 14:58:42 $
$Revision: 1.13 $
$Locker:  $
*)

(*$Parse:
	TOPDEC_GRAMMAR DEC_GRAMMAR CON EXCON SCON LAB TYCON TYVAR IDENT STRID
	FUNID SIGID LEX_BASICS GRAMMAR_INFO SOURCE_INFO DF_INFO INFIX_BASIS
	PPDECGRAMMAR PRETTYPRINT REPORT FINMAP BASIC_IO FLAGS CRASH
	MyBase LexSML_ ParseSML_ GrammarUtils LexUtils Infixing PARSE
 *)
functor Parse(structure TopdecGrammar: TOPDEC_GRAMMAR
	      structure DecGrammar: DEC_GRAMMAR
		sharing type TopdecGrammar.dec = DecGrammar.dec
		    and type TopdecGrammar.ty = DecGrammar.ty
		    and type TopdecGrammar.WithInfo = DecGrammar.WithInfo

	      structure Con: CON
		sharing type TopdecGrammar.con = DecGrammar.con = Con.con

	      structure Excon: EXCON
		sharing type TopdecGrammar.excon = DecGrammar.excon
			     = Excon.excon

	      structure SCon: SCON  sharing type DecGrammar.scon = SCon.scon

	      structure Lab: LAB  sharing type DecGrammar.lab = Lab.lab

	      structure TyCon: TYCON
		sharing type TopdecGrammar.tycon = DecGrammar.tycon
			     = TyCon.tycon
		    and type TopdecGrammar.longtycon = DecGrammar.longtycon
			     = TyCon.longtycon

	      structure TyVar: TYVAR
	        sharing type TopdecGrammar.tyvar = DecGrammar.tyvar
			     = TyVar.SyntaxTyVar

	      structure Ident: IDENT
		sharing type TopdecGrammar.id = DecGrammar.id = Ident.id
		    and type DecGrammar.longid = Ident.longid

	      structure StrId: STRID
		sharing type TopdecGrammar.strid = StrId.strid
		    and type TopdecGrammar.longstrid = DecGrammar.longstrid
			     = StrId.longstrid

	      structure FunId: FUNID
	        sharing type TopdecGrammar.funid = FunId.funid

	      structure SigId: SIGID
	        sharing type TopdecGrammar.sigid = SigId.sigid

	      structure LexBasics: LEX_BASICS

	      structure GrammarInfo: GRAMMAR_INFO
		sharing type TopdecGrammar.info = DecGrammar.info
		  	     = GrammarInfo.PreElabGrammarInfo

	      structure SourceInfo: SOURCE_INFO
		sharing type GrammarInfo.SourceInfo = SourceInfo.info
		    and type SourceInfo.pos = LexBasics.pos

	      structure DFInfo: DF_INFO
		sharing type GrammarInfo.DFInfo = DFInfo.info

	      structure InfixBasis: INFIX_BASIS
		sharing type InfixBasis.id = Ident.id

	      structure PPDecGrammar: PPDECGRAMMAR
		sharing PPDecGrammar.G = DecGrammar

	      structure Report: REPORT
		sharing type LexBasics.Report = Report.Report

	      structure PP: PRETTYPRINT
		sharing type PPDecGrammar.StringTree = PP.StringTree

	      structure FinMap: FINMAP
	      structure BasicIO: BASIC_IO
	      structure Flags: FLAGS
	      structure Crash: CRASH
	     ): PARSE =
  struct
    structure Stream = Stream()

    structure LrParser = ParserGen(structure LrTable = LrTable()
				   structure Stream = Stream
				  )

    structure GrammarUtils =
      GrammarUtils(structure TopdecGrammar = TopdecGrammar
		   structure DecGrammar = DecGrammar
		   structure LexBasics = LexBasics
		   structure Con = Con
		   structure Excon = Excon
		   structure SCon = SCon
		   structure Lab = Lab
		   structure TyCon = TyCon
		   structure TyVar = TyVar
		   structure Ident = Ident
		   structure StrId = StrId
		   structure FunId = FunId
		   structure SigId = SigId
		   structure GrammarInfo = GrammarInfo
		   structure SourceInfo = SourceInfo
		   structure DFInfo = DFInfo
		   structure Crash = Crash
		  )    

    structure TopdecLrVals =
      TopdecLrVals(structure Token = LrParser.Token
		   structure LexBasics = LexBasics
		   structure GrammarUtils = GrammarUtils
		  )

    structure LexUtils = LexUtils(structure LexBasics = LexBasics
				  structure Token = TopdecLrVals.Tokens
				  structure BasicIO = BasicIO
				  structure Flags = Flags
				  structure Crash = Crash
				 )

    structure Infixing = Infixing(structure InfixBasis = InfixBasis
				  structure GrammarUtils = GrammarUtils
				  structure Ident = Ident
				  structure DFInfo = DFInfo
				  structure PPDecGrammar = PPDecGrammar
				  structure PP = PP
				  structure Crash = Crash
				 )

    structure TopdecLex =
      TopdecLex(structure Tokens = TopdecLrVals.Tokens
		structure LexBasics = LexBasics
		structure LexUtils = LexUtils
	       )

    structure TopdecParser =
      JoinWithArg(structure ParserData = TopdecLrVals.ParserData
		  structure Lex = TopdecLex
		  structure LrParser = LrParser
		 )


    val eof = TopdecLrVals.Tokens.EOF(LexBasics.DUMMY, LexBasics.DUMMY)
    val sc = TopdecLrVals.Tokens.SEMICOLON(LexBasics.DUMMY, LexBasics.DUMMY)

    type Report = Report.Report
    infix //
    val op // = Report.//

    exception ESCAPE of string * (LexBasics.pos * LexBasics.pos) Option

   (* The state of a parser, holding the current token stream between
      top-level phrases (needed to handle "val x = 1; val y = 2;" without
      losing the end of the line. *)

    local
      open TopdecLrVals
    in
      datatype State =
	STATE of (Tokens.svalue, LexBasics.pos) Tokens.token Stream.stream
			(* God I HATE that parser generator... *)
    end

   (* result type for `parseStream' (as opposed to `parse'): *)

    datatype PSResult = PS_SUCCESS of TopdecGrammar.topdec * State
		      | PS_ERROR of Report
		      | PS_EOF

    fun parseStream(STATE lazyStream) =
      let
	val (firstToken, rest) = Stream.get lazyStream
	val lazyStream = Stream.cons(firstToken, rest)
				      (* Streams side-effect (yuck). *)
      in
	if LrParser.Token.sameToken(firstToken, eof)
	then
	  (if Flags.DEBUG_PARSING then BasicIO.println "**EOF**" else ();
	   PS_EOF
	  )
	else if LrParser.Token.sameToken(firstToken, sc)
	then
	  (if Flags.DEBUG_PARSING then BasicIO.println "**SC**" else ();
	   parseStream(STATE rest)
	  )
	else
	  (let
	     val (topdec, lazyStream') =
	       TopdecParser.parse(0, lazyStream,
				  fn (x, l, r) => raise ESCAPE(x, Some(l, r)),
				  ()
				 )
	       handle LexBasics.LEXICAL_ERROR(pos, msg) =>
		        raise ESCAPE(msg, Some(pos, pos))
		    | GrammarUtils.LAYERPAT_ERROR posLR_opt =>
			raise ESCAPE("Bad layered pattern", posLR_opt)
	   in
	     if Flags.DEBUG_PARSING
	     then BasicIO.println "**SUCCESS**"
	     else ();
	     PS_SUCCESS(topdec, STATE lazyStream')
	   end
	  ) handle ESCAPE(text, Some(lPos, rPos)) =>
	             PS_ERROR(LexBasics.reportPosition{left=lPos, right=rPos}
			      // Report.line text
			     )

		 | ESCAPE(text, None) =>
		     PS_ERROR(Report.line "(position unknown)"
			      // Report.line text
			     )
      end

    type topdec = TopdecGrammar.topdec
    type InfixBasis = Infixing.InfixBasis
    type SourceReader = LexBasics.SourceReader

    val sourceFromStdIn = LexBasics.lexFromStdIn
    val sourceFromFile = LexBasics.lexFromFile
    val sourceFromString = LexBasics.lexFromString

    fun nameOf(LexBasics.SOURCE_READER{name, ...}) = name

    datatype Result = SUCCESS of InfixBasis * topdec * State
      		    | ERROR of Report
		    | LEGAL_EOF

    fun begin sourceReader =
      let
	val LexBasics.SOURCE_READER{clearFn, lexingFn, ...} = sourceReader
	val _ = clearFn()		(* Forget any stored lines *)

	val lex_fn =
	  TopdecLex.makeLexer lexingFn (LexUtils.initArg sourceReader)

	val stream = Stream.streamify lex_fn
      in
	STATE stream
      end

    fun parse(ib, state) =
      case parseStream state
	of PS_SUCCESS(topdec, state') =>
	     let
	       val (ib', topdec') = Infixing.resolveTopdec(ib, topdec)
	     in
	       SUCCESS(ib', topdec', state')
	     end

	 | PS_ERROR report => ERROR report
	 | PS_EOF => LEGAL_EOF
  end;
