/*
 * dns.c
 *
 * x-kernel v3.1	12/10/90
 *
 * Copyright (C) 1990  Larry L. Peterson and Norman C. Hutchinson
 */

#include "xkernel.h"
#include "ip.h"
#include "udp.h"
#include "dns.h"
#include "dns_internal.h"

extern unsigned long inet_addr();

int tracednsp;

#define UDP (self->down[0])
#define IP (self->down[1])

unsigned char *dns_getname();
unsigned char *dns_getrr();
int   dns_timeout();
void reverse();
ResRecord   *dns_findname();
char *dns_itos();

/***********************************
* Uniform Interface Operations
************************************/

dns_instantiateprotl(self)
XObj	self;
{
  PSTATE	*pstate;

  TRACE0(dnsp, 1, "DNS instantiateprotl");
  pstate = (PSTATE *) malloc(sizeof(PSTATE));
  self->state = (char *) pstate;
}

dns_init(self)
XObj self;
{
  PSTATE *pstate;
  unsigned long inetaddr;
  IPhost client;
  RNS *tr;
#ifdef XSIMUL
  char **rp;
#endif

  assert(self->numdown==2);
  assert((int)self->down[0]);
  assert((int)self->down[1]);  

  pstate = (PSTATE *) self->state;
  pstate->current_id = 0;
  pstate->pending = map_create(20, 8);
  InitSemaphore(&pstate->pending_s,1);
  pstate->sbelt = NULL;

#ifdef	XSIMUL
  for (rp = rom; *rp; rp += 3) {
    if (!strcmp(rp[0], "dns")) {
      tr = NEW(RNS);
      tr->zoneq = -1;
      dns_makeiname(&tr->name, rp[1]);
      inetaddr = inet_addr(rp[2]);
      bcopy((char *)&inetaddr, (char *)&tr->addr, 4);
      tr->next = pstate->sbelt;
      pstate->sbelt = tr;
    }
  }
#endif
  if (pstate->sbelt == NULL) {
    /* Use SITE_DNS_SRV_ADDR as default name server */
    tr = NEW(RNS);
    tr->zoneq = -1;
    dns_makeiname(&tr->name, SITE_DNS_SRV_NAME);
    inetaddr = inet_addr(SITE_DNS_SRV_ADDR);
    bcopy(&inetaddr, &tr->addr, 4);
    tr->next = pstate->sbelt;
    pstate->sbelt = tr;
  }
  x_control(IP, GETMYADDR, (char *)&client, IPADLEN);
  pstate->clntaddr.port = 1200;
  pstate->clntaddr.host = client;
  pstate->srvraddr.port = 53;
  pstate->srvraddr.host = *(IPhost *) &inetaddr;
}

/*ARGSUSED*/
Sessn dns_open(self, hlp, p)
XObj	self;
XObj	hlp;
Part	*p;
{
  return(ERR_XOBJ);
}

/*ARGSUSED*/
dns_control(self, op, buf, len)
XObj	self;
int	op, len;
char	*buf;
{
  PSTATE *pstate;
  RSTATE *rs;
  RNS	*servers;
  Part	p[3];
  XObj	ds;
  Msg	m;
  long	now;
  char *lbuf = buf;
  char *mbuf,*mp;
  PendingKey pk;

  pstate = (PSTATE *) self->state;
  rs = NEW(RSTATE);
  switch (op) {
    case RESOLVE:
      checkLen(len, sizeof(Result));
      rs->name.type = A;
      rs->name.class = IN;
      break;
    case RRESOLVE:
      checkLen(len, sizeof(Result));
      lbuf = malloc(strlen(buf) + strlen(".in-addr.arpa") + 1);
      reverse(buf, lbuf);
      strcat(lbuf, ".in-addr.arpa");
      rs->name.type = PTR;
      rs->name.class = IN;
      break;
  
    default:
      x_errno = INVALID_OPCODE;
      free((char *)rs);
      return(-1);
  }
  dns_makeiname(&rs->name.name, lbuf);

  /* From RICE 07/03/90 */
  if (op == RRESOLVE) {
    free(lbuf);
  }

  rs->hdr.id = htons(pstate->current_id++);
  rs->hdr.qdcnt = htons(1);
  rs->hdr.ancnt = htons(0);
  rs->hdr.nscnt = htons(0);
  rs->hdr.arcnt = htons(0);
  rs->hdr.flags.all = 0;
  rs->hdr.flags.flags.rd = 1;
  rs->hdr.flags.all = htons(rs->hdr.flags.all);

  rs->list = NULL;

  P(&pstate->pending_s);
  pk.spad = 0;
  pk.lpad = 0;
  pk.id = rs->hdr.id;
  map_bind(pstate->pending, (char *)&pk, rs);
  V(&pstate->pending_s);

  while(TRUE) {
    /* look in the cache for rs->name.name */
    InitSemaphore(&rs->s, 0);
    if (rs->list == NULL) 
      servers = pstate->sbelt;
    else
      servers = rs->list;

    init_partlist(p, 2, UDPaddr);
    set_part(p, 0, pstate->clntaddr);
    set_part(p, 1, pstate->srvraddr);
    rs->result = DNS_QUERYING;
    mbuf = (char *) malloc(256);
    mp = mbuf;
    bcopy(&rs->hdr, mp, RHDRLEN);
    mp += RHDRLEN;
    bcopy(rs->name.name.name, mp, rs->name.name.size);
    mp += rs->name.name.size;
    bcopy(&rs->name.type, mp, 2);
    mp += 2;
    bcopy(&rs->name.class, mp, 2);
    msg_make_new(m, 128, mbuf, mp-mbuf+2);
    for (; servers != NULL; servers = servers->next) {
      msg_save(m, m);
      bcopy(&servers->addr, &pstate->srvraddr.host, 4);
      ds = x_open(self, UDP, p);
      x_push(ds, m, 0);
    }
    msg_free(m);
    dns_rtime(&now);
    event_register(dns_timeout, (int)rs, DNS_TIME, EV_ONCE | EV_CREATEPROCESS);
    P(&rs->s);
    if (rs->result == DNS_TIMEOUT) {
      rs->hdr.flags.flags.rcode = RTIMEOUT;
      rs->hdr.ancnt = 0;
      return(dns_cleanup(rs,op,buf));
    } else {
      return(dns_cleanup(rs,op,buf));
    }
  }
}

/*ARGSUSED*/
dns_demux(self, s, msg)
XObj	self;
XObj	s;
Msg	msg;
{
  PSTATE *pstate;
  RHEADER rh;
  RSTATE *r;
  PendingKey p;

  pstate = (PSTATE *)self->state;
  x_close(s);
  if (msg_len(msg)-RHDRLEN <= 0) {
    printf("bad reply\n");
    return;
  }
  msg_peek(msg, 0, RHDRLEN, (char *)&rh);
  rh.id = ntohs(rh.id);
  rh.flags.all = ntohs(rh.flags.all);
  rh.qdcnt = ntohs(rh.qdcnt);
  rh.ancnt = ntohs(rh.ancnt);
  rh.nscnt = ntohs(rh.nscnt);
  rh.arcnt = ntohs(rh.arcnt);
  P(&pstate->pending_s);
  p.spad = 0;
  p.lpad = 0;
  p.id = ntohs(rh.id);
  r = (RSTATE *)map_resolve(pstate->pending, (char *)&p);
  if (r != (RSTATE *)-1) (void) map_unbind(pstate->pending, (char *)&p, r);
  V(&pstate->pending_s);
  if (r != (RSTATE *)-1) {
    r->hdr = rh; 
    r->msgbuf = (unsigned char *) malloc((unsigned)msg_len(msg));
    msg_peek(msg, 0, msg_len(msg), (char *)r->msgbuf);
    r->result = DNS_RESOLVED;
    event_remove(dns_timeout, r);
    V(&r->s);
  }
  msg_free(msg);
  return;
}

dns_getproc(p,type)
XObj p;
XObjType type;
{
  TRACE1(dnsp, 3, "in dns getproc, type = %s", type == Session ? "Session" : "Protocol");
  if (type == Protocol) {
    p->instantiateprotl = dns_instantiateprotl;
    p->init = dns_init;
    p->close = noop;
    p->push = noop;
    p->pop = noop;
    p->control = dns_control;
  } else {
    p->push = noop;
    p->pop = noop;
    p->instantiateprotl = noop;
    p->init = noop;
    p->close = noop;
    p->control = noop;
  }
  p->demux = dns_demux;
  p->open = (Pfi) dns_open;
  p->openenable = noop;
  p->opendone = noop;
  p->closedone = noop;
  p->opendisable = noop;
  p->getproc = dns_getproc;
  TRACE0(dnsp, 3, "done dns getproc");
}

/***********************************
* Internal Routines
************************************/

void reverse(from, to)
char *from, *to;
{
  char *s, *t;
  t = from + strlen(from);
  while (t >= from) {
    for (s = t-1; s > from && *s != '.'; s--) ;
    if (s != from) s++;
    bcopy(s, to, t - s);
    to += t - s;
    if (s != from) *to++ = '.';
    t = s - 1;
  }
  *to = '\0';
}

unsigned char *dns_getname(base, ptr, name)
unsigned char *base, *ptr;
Name *name;
{
  unsigned cnt;
  unsigned char *p = ptr;

  /* find out how long the name is */
  while (1) {
    cnt = *p++;
    if (cnt == 0) {
      break;
    } else if ((cnt & 0xc0) == 0xc0) {
      p++;
      break;
    } else {
      p += cnt;
    }
  }
  if (name) {
    name->name.base = base;
    name->name.name = ptr;
    name->name.size = p - ptr;
    name->type = *(unsigned short *) p;
    name->class = *(unsigned short *) (p+2);
  }
  p += 4;
  return(p);
}

unsigned char *dns_getrr(base, ptr, rr)
unsigned char *base;
unsigned char *ptr;
ResRecord *rr;
{
  int dlen;
  ptr = dns_getname(base, ptr, &rr->name);
  dlen = ntohs(*(unsigned short *) (ptr + 4));
  if (rr) {
    rr->ttl = ntohl(*(long *) ptr);
    rr->dlen = dlen;
    rr->data = (ptr + 4 + 2);
  }
  ptr += 4 + 2 + dlen;
  return ptr;
}

dns_cleanup(rs,op,buf)
RSTATE *rs;
int op;
char *buf;
{
  int ans, i;
  RNS *t;
  ResRecord rr;
  Result *result;
  unsigned char *m;

  dns_freeiname(&rs->name.name);
  for (t = rs->list; t!=NULL; t=t->next) {
    dns_freeiname(&t->name);
    free((char *)t);
  }
  ans = -1;
  switch(op) {
    case(RRESOLVE):
    case(RESOLVE):
      if (rs->hdr.flags.flags.rcode == 0) {
	result = (Result *) buf;
	ans = 0;
	m = rs->msgbuf + RHDRLEN;
	m = dns_getname(rs->msgbuf, m, (Name *)NULL);
 	for (i = 0; i < rs->hdr.ancnt; i++) {
	  m = dns_getrr(rs->msgbuf, m, &rr);
	  TRACE1(dnsp, 5, "rr type is %d", rr.name.type);
	  switch (rr.name.type) {
	    case A:
	      result->addr[ans++] = *(IPhost *) rr.data;
	      break;
	    case CNAME:
	    case PTR:
	      (void) dns_itos(rr.data, rs->msgbuf, result->name);
	      break;
	    default:
	      break;
	  }
	}
	free((char *)rs->msgbuf);
      }
      free((char *)rs);
      break;
  }
  return ans;
}

#ifndef XSIMUL
#include "unixtime.h"
#endif

dns_rtime(current_time)
long *current_time;
{
#ifndef XSIMUL
  unsigned char buffer[8];
  UnixTime ut;

  ReadTime(buffer);
  ConvertTimeToUnix(buffer, &ut);
  *current_time = ut.sec;
#else
  long time;

  read_clock(&time);
  *current_time = (time / 1000);
#endif
}

dns_timeout(rs)
RSTATE *rs;
{
  if (rs->result == DNS_QUERYING) {
    rs->result = DNS_TIMEOUT;
    V(&rs->s);
  }
}

char *dns_itos(name, buffer, s)
unsigned char *name, *buffer;
char *s;
{
  int first = 1;
  unsigned int len;

  while (1) {
    len = name[0];
    if ((len & 0xc0) == 0xc0) {
      /* we have a pointer */
      name = buffer + ((len & 0x3f) << 8) + name[1];
    } else if (len == 0) {
      break;
    } else {
      if (first) {
	first = 0;
      } else {
	*s++ = '.';
      }
      bcopy((char *)&name[1], s, (int)len);
      s += len;
      name += len + 1;
    }
  }
  *s = '\0';
  return s;
}

dns_freeiname(in)
Iname *in;
{
  if (in->base == in->name) {
    /* it is a good bet that it is safe to free this */
    free((char *)in->base);
  } else {
    /* It is in a message, so wait for the message free */
  }
}

dns_makeiname(in, s)
Iname *in;
char *s;
{
  int len;
  unsigned char *fill;
  char *t;

  in->size = strlen(s) + 2;
  fill = (unsigned char *)malloc(in->size);
  in->name = fill;
  in->base = fill;
  while (*s) {
    for (t = s; *t && *t != '.'; t++) ;
    len = t - s;
    assert(len < 64);
    *fill++ = len;
    bcopy(s, (char *)fill, len);
    fill += len;
    s = t;
    if (*s) s++;
  }
  *fill = '\0';
}

printiname(l)
Iname *l;
{
  char buffer[256];
  printf("%s\n", dns_itos(l->name, l->base, buffer));
}

