/*
 * event.c
 *
 * x-kernel v3.2
 *
 * Copyright (c) 1991  Arizona Board of Regents
 *
 *
 * $Revision: 1.16 $
 * $Date: 1992/02/07 17:07:54 $
 */

/*
 * Event library for Mach 3.0 platform [mats]
 */


#include <mach/message.h>

#ifdef XKMACHKERNEL
#include <kern/thread.h>
#include <kern/lock.h>
#include <kern/syscall_subr.h>
#else
#include <cthreads.h>
#include <mach.h>
#endif XKMACHKERNEL

#include "xk_debug.h"
#include "upi.h"
#include "platform.h"
#include "assert.h"
#include "event.h"

#define E_RUNNING (1 << 0)
#define E_FINISHED (1 << 1)
#define E_DETACHED  (1 << 2)

typedef int (*intFunc)();

#define THIS_IS_THE_HEADER ((EvFunc)-42)


/* BIG_N should be a function of the estimated number of scheduled events,
 * such that queues don't get too long, but with as little overhead as
 * possible. 128 is probably good for up to some 500-1000 events. [mats]
 */

#define BIG_N 128 

static struct Event evhead[BIG_N];

static mach_port_t	evClock_port;

#ifdef XKMACHKERNEL
static thread_t		evClock_thread;
#else
static cthread_t	evClock_thread;
#endif XKMACHKERNEL

#ifdef XKMACHKERNEL
simple_lock_data_t	 eLock;
#define EVENT_LOCK()     simple_lock( &eLock )
#define EVENT_UNLOCK()   simple_unlock( &eLock )
#else
mutex_t			 event_mutex;
#define EVENT_LOCK()     mutex_lock(event_mutex)
#define EVENT_UNLOCK()   mutex_unlock(event_mutex)
#endif XKMACHKERNEL


extern int event_granularity;

int tickmod;
int tracetick;
int traceevent;

void evClock(int); /* Forward */

void
evInit(interval)
    int interval;
{
  int i;
  xTrace0(event,TR_GROSS_EVENTS,"evInit enter");
  for(i=0;i<BIG_N;i++)
    {
      evhead[i].next = &evhead[i];
      evhead[i].prev = &evhead[i];
      evhead[i].func = THIS_IS_THE_HEADER;
    }
  tickmod = 0;
 
/* initialize timers and semaphores */
#ifdef XKMACHKERNEL
  simple_lock_init( &eLock );
#else
  event_mutex = mutex_alloc();
  (void) mach_port_allocate(mach_task_self(), MACH_PORT_RIGHT_RECEIVE,
			    &evClock_port);
  (void) mach_port_insert_right(mach_task_self(),
				evClock_port, evClock_port,
				MACH_MSG_TYPE_MAKE_SEND);
#endif XKMACHKERNEL
  
/* Since evClock does not use any xkernel routines, this should be 
   allowed to remain a cthread, in spite of the sledgehammer concurrency
   control otherwise enforced in the xkernel.
 */

  xTrace0(event,TR_EVENTS,"evInit starting evClock thread");

#ifdef XKMACHKERNEL
  evClock_thread = kernel_thread( kernel_task, evClock );
#else
  evClock_thread = cthread_fork(evClock,interval);
  thread_priority(evClock_thread, STD_PRIO-1, FALSE);
  cthread_detach(evClock_thread);
#endif XKMACHKERNEL

  xTrace0(event,TR_EVENTS,"evInit exit");
}


static void
e_insque(q,e)
    int q;
    Event e;
{
  Event a;

  xTrace0(event,TR_FULL_TRACE,"e_insque enter");
  EVENT_LOCK();
  for (a = evhead[q].next; a != &evhead[q]; a = a->next) {
    if (a->deltat < e->deltat) {
      /* E goes after a */
      e->deltat -= a->deltat;
      continue;
    } else {
      /* E goes just before a */
      a->deltat -= e->deltat;
      insque(e, a->prev);
      EVENT_UNLOCK();
      return;
    }
  }
  /* E goes at the end */
  insque(e, evhead[q].prev);
  EVENT_UNLOCK();
}

static void
e_remque(e)
    Event e;
{
    xTrace0(event,TR_FULL_TRACE,"e_remque enter");
    xAssert(e);
    if (e->next->func != THIS_IS_THE_HEADER) {
	e->next->deltat += e->deltat;
    }
    remque(e);
}


static void
stub(e)
    Event e;
{
  xTrace0(event,TR_FULL_TRACE,"event stub enter");
  e->func(e->arg);
  EVENT_LOCK();
  e->flags &= ~E_RUNNING;
  e->flags |= E_FINISHED;
  if (e->flags & E_DETACHED) {
    xFree((char *)e);
  }
  EVENT_UNLOCK();
}

      
Event evSchedule( func, arg, time ) /* Time in us */
    EvFunc 	func;
    VOID	*arg;
    unsigned 	time; 	/* Time in us */
{
  Event e;
  int delt;
  bool  ret;

  xTrace0(event,TR_FULL_TRACE,"evSchedule enter");
  e = (Event)xMalloc(sizeof(struct Event));
  xTrace1(event, TR_EVENTS, "evSchedule allocates event %x", e);
  e->func = func;
  e->arg = arg;
  delt = (time+500*event_granularity)/(event_granularity*1000);
               /* time in us, delt in ticks, event_granularity in ms/tick */
  if (delt == 0 && time != 0) {
      delt = 1;
  }
  e->deltat = delt/BIG_N; /* If BIG_N is a power of 2, this could be a shift*/
  e->flags = 0;
  if (delt == 0) {
    e->flags |= E_RUNNING;
/*    cthread_detach(cthread_fork(stub, e)); */
/* Disallowed, use sledgehammer instead. */
    xTrace0(event,TR_EVENTS,"evSchedule starting event");
    ret = CreateProcess(stub,STD_PRIO,1,e);
  } else {
    /* e_insque will take care of locking */
    e_insque((tickmod+delt)%BIG_N,e); /* Mod could be AND if BIG_N power of 2*/
  }
  xTrace0(event,TR_EVENTS,"evSchedule exit");
  return e;
}


void evDetach(e)
    Event e;
{
  xTrace0(event, TR_FULL_TRACE, "evDetach");
  EVENT_LOCK();
  if (e->flags & E_FINISHED) {
    xTrace1(event, 5, "evDetach frees event %x", e);
    xFree((char *)e);
  } else {
    xTrace1(event, 5, "evDetach marks event %x", e);
    e->flags |= E_DETACHED;
  }
  EVENT_UNLOCK();
  xTrace0(event, TR_EVENTS, "evDetach exits");
}


/* cancel event e:
   returns -1 if it knows that the event has already executed to completion
   returns  0 if it can guarantee that either f is currently running or that
       it will run in the future
   returns  1 if it can guarantee that f has not run, and will not run in
       the future
 */
int evCancel(e)
    Event e;
{
    int ans;

    xAssert(e);
    EVENT_LOCK();
    xTrace1(event, TR_FULL_TRACE, "evCancel: event = %x", e);
    if (e->flags & E_RUNNING) {
	e->flags |= E_DETACHED;
	ans = 0;
    } else if (e->flags & E_FINISHED) {
	ans = -1;
	xFree((char *)e);
    } else {
	e_remque(e);
	ans = 1;
	xFree((char *)e);
    }
    EVENT_UNLOCK();
    return ans;
}


void evClock(interval)
    int interval;
{
  Event e;
  bool  ret;
#ifndef XKMACHKERNEL
  mach_msg_header_t m;
  cthread_set_name(cthread_self(), "evClock");
#endif XKMACHKERNEL

  xTrace0(event,TR_FULL_TRACE,"evClock enter");
  while(TRUE)
    {
#ifdef XKMACHKERNEL
      thread_will_wait_with_timeout( current_thread(), interval );
      thread_block((void (*)()) 0);
      if (1)
#else /* XKMACHKERNEL */
      cthread_yield();
      if (mach_msg(&m, MACH_RCV_MSG|MACH_RCV_TIMEOUT,
		   0, sizeof m, evClock_port,
		   interval, MACH_PORT_NULL) == MACH_RCV_TIMED_OUT)
#endif XKMACHKERNEL
	{
	  xTrace0(event,TR_EVENTS,"evClock tick");
	  EVENT_LOCK();
	  evhead[tickmod].next->deltat--;
	  tickmod = (tickmod+1)%BIG_N; /* optimize if BIG_N is power of 2*/
	  for (e = evhead[tickmod].next; e != &evhead[tickmod] 
	       && e->deltat == 0; e = e->next) {
	    remque(e);
	    e->flags |= E_RUNNING;
/*	    cthread_detach(cthread_fork(stub, e)); */
/* Disallowed, use sledgehammer instead. */
	    xTrace0(event,TR_EVENTS,"evClock starting event");
	    ret = CreateProcess(stub,STD_PRIO, 1, e);
	  }
	  EVENT_UNLOCK();
	}
#ifndef XKMACHKERNEL
      else
	{
	  printf("evClock: received a message!!!\n");
	  /* If we get a message. In this version we shouldn't! [mats] */
	}
#endif XKMACHKERNEL      
    }
}

