/*
 * prottbl.c
 *
 * x-kernel v3.2
 *
 * Copyright (c) 1991  Arizona Board of Regents
 *
 *
 * $Revision: 1.17 $
 * $Date: 1992/02/07 04:32:07 $
 */

/*
 * Management of the "protocol id / relative number" table
 */

#ifndef XKMACHKERNEL
#  include "x_stdio.h"
#  include "x_libc.h"
#endif ! XKMACHKERNEL
#include "prottbl.h"
#include "prottbl_i.h"
#include "idmap.h"
#include "xk_debug.h"
#include "assert.h"

typedef PtblEntry	Entry;

int	traceptbl;

Map	ptblNameMap = 0;	/* strings to Entry structures */
static Map	idMap = 0;
static char	errBuf[200];
static int	errorOccurred;

#define MAX_PROT_NAME	16
/* 
 * Error messages
 */
#define MULT_PROT_MSG   "prot table: multiple protocols with id %d declared"
#define NO_ID_MSG     "prot table: %s (declared as hlp to %s) has no id number"
#define NAME_REPEATED_MSG  "prot table: protocol %s is declared with different id's"
#define BIND_REPEATED_MSG  "prot table: binding for %s above %s defined multiple times"
#define NO_ENTRY_MSG "prot table: No entry exists for llp %s"

#define NUM_BOUND_MSG "prot table: relative number %d is already bound for llp %s"

static void
error(msg)
    char *msg;
{
    xError(msg);
    errorOccurred = 1;
}


static void
mkKey(key, name)
    char *key;
    char *name;
{
    bzero(key, MAX_PROT_NAME);
    strncpy(key, name, MAX_PROT_NAME);
}


/* 
 * This function is called for each entry in a lower protocol's temp
 * map (binding higher protocol name strings to relative numbers.)
 * We find the hlp's id number and bind it to the relative number,
 * signalling an error if we can't find the hlp's id number.
 */
static int
backPatchBinding(key, relNum, arg)
    VOID *key;
    int relNum;
    VOID *arg;
{
    Entry	*llpEntry = (Entry *)arg;
    Entry	*hlpEntry;
    
    if ( (hlpEntry = (Entry *)mapResolve(ptblNameMap, key)) == (Entry *)-1 ) {
	error(sprintf(errBuf, NO_ID_MSG, key, llpEntry->name));
    } else {
	xTrace3(ptbl, 3,
		"Backpatching binding of %s->%s using %d",
		hlpEntry->name, llpEntry->name, relNum);
	mapBind(llpEntry->idMap, &hlpEntry->id, relNum);
    }
    return 1;
}


/* 
 * This function is called for each entry (lower protocol) in the name
 * map.  If there are entries in the llp's tmpMap (higher protocols
 * which were bound with a string instead of an id), we backpatch the
 * id binding.
 */
static int
checkEntry(key, value, arg)
    VOID *key;
    int value;
    VOID *arg;
{
    Entry	*llpEntry = (Entry *)value;

    /* 
     * If there is anything in this entry's tmpMap, transfer it to the
     * idMap.  If we don't know the id number of the hlp in the
     * tmpMap, signal an error.
     */
    if ( llpEntry->tmpMap ) {
	mapForEach(llpEntry->tmpMap, backPatchBinding, llpEntry);
	mapClose(llpEntry->tmpMap);
	llpEntry->tmpMap = mapCreate(101, MAX_PROT_NAME);
    }
    return 1;
}


int
protTblBuild(filename)
    char *filename;
{
    xTrace1(ptbl, 3, "protTblBuild( %s )", filename);
    if ( ptblNameMap == 0 ) {
	ptblNameMap = mapCreate(101, MAX_PROT_NAME);
	idMap = mapCreate(101, sizeof(long));
    }
    if ( (protTblParse(filename)) != 0 || errorOccurred ) {
	return -1;
    }
    xTrace1(ptbl, 3, "protTblBuild( %s ) checking map consistency", filename);
    /* 
     * Check the consistency of the protocol map.  
     */
    mapForEach(ptblNameMap, checkEntry, 0);
    return errorOccurred;
}


long
protTblGetId( protocolName )
    char *protocolName;
{
    Entry	*e;
    char	key[MAX_PROT_NAME];
    
    xAssert(ptblNameMap);
    mkKey(key, protocolName);
    if ( (e = (Entry *)mapResolve(ptblNameMap, key)) == (Entry *)-1 ) {
	return -1;
    }
    return e->id;
}


char *
protIdToStr( id )
    long	id;
{
    Entry	*e;

    if ( (e = (Entry *)mapResolve(idMap, (char *)&id)) == (Entry *)-1 ) {
	return 0;
    }
    return e->name;
}


long
relProtNum(hlp, llp)
    XObj hlp;
    XObj llp;
{
    Entry	*llpEntry;
    
    xAssert(ptblNameMap);
    if ( ! ( xIsProtocol(hlp) && xIsProtocol(llp) ) ) {
	return -1;
    }
    if ( (llpEntry = (Entry *)mapResolve(idMap, &llp->id)) == (Entry *)-1 ) {
	return -1;
    }
    if ( llpEntry->idMap == 0 ) {
	/* 
	 * The lower protocol does not use relative naming.  Return the
	 * absolute id of the upper protocol.
	 */
	return hlp->id;
    }
    return mapResolve(llpEntry->idMap, &hlp->id);
}


void
protTblAddProt(name, id)
    char *name;
    long int id;
{
    Entry	*e;
    char	key[MAX_PROT_NAME];
    
    xTrace2(ptbl, 3, "protTblAddProt adding %s(%d)", name, id);
    mkKey(key, name);
    if ( (e = (Entry *)mapResolve(ptblNameMap, key)) == (Entry *)-1 ) {
	/* 
	 * Entry does not exist for this name
	 */
	if ( mapResolve(idMap, &id) != -1 ) {
	    error(sprintf(errBuf, MULT_PROT_MSG, id));
	    return;
	}
	e = (Entry *)xMalloc(sizeof(Entry));
	e->name = xMalloc(strlen(name) + 1);
	strcpy(e->name, name);
	e->id = id;
	e->idMap = e->tmpMap = 0;
	mapBind(ptblNameMap, key, e);
	mapBind(idMap, &e->id, e);
    } else {
	/* 
	 * Make sure that this declaration has the same id number as
	 * the previous declarations.
	 */
	if ( e->id != id ) {
	    error(sprintf(errBuf, NAME_REPEATED_MSG, name));
	}
    }
}



void
protTblAddBinding(llpName, hlpName, relNum)
    char *llpName;
    char *hlpName;
    long int relNum;
{
    Entry	*llpEntry, *hlpEntry;
    char	key[MAX_PROT_NAME];
    
    xTrace3(ptbl, 3, "protTblAddBinding adding %s->%s uses %d",
	    hlpName, llpName, relNum);
    mkKey(key, llpName);
    if ( (llpEntry = (Entry *)mapResolve(ptblNameMap, key)) == (Entry *)-1 ) {
	/* 
	 * No entry exists for the lower protocol
	 */
	error(sprintf(errBuf, NO_ENTRY_MSG, llpName));
	return;
    }
    if ( llpEntry->idMap == 0 ) {
	llpEntry->idMap = mapCreate(101, sizeof(long));
	llpEntry->tmpMap = mapCreate(101, MAX_PROT_NAME);
	llpEntry->revMap = mapCreate(101, sizeof(long));
    }
    if ( mapResolve(llpEntry->revMap, &relNum) != -1 ) {
	error(sprintf(errBuf, NUM_BOUND_MSG, relNum, llpName));
	return;
    }
    mkKey(key, hlpName);
    if ( (hlpEntry = (Entry *)mapResolve(ptblNameMap, key)) == (Entry *)-1 ) {
	/* 
	 * Entry for hlp doesn't exist yet.  Add the binding from the
	 * hlp name to the number in tmpMap.  The binding will be
	 * tranferred to idMap after the hlp entry has been added.
	 */
	if ( mapResolve(llpEntry->tmpMap, key) == -1 ) {
	    mapBind(llpEntry->tmpMap, key, relNum);
	    mapBind(llpEntry->revMap, &relNum, 0);
	} else {
	    error(sprintf(errBuf, BIND_REPEATED_MSG, hlpName, llpName));
	}
    } else {
	/* 
	 * Add the entry directly to the idMap
	 */
	if ( mapResolve(llpEntry->idMap, &hlpEntry->id) == -1 ) {
	    mapBind(llpEntry->idMap, &hlpEntry->id, relNum);
	    mapBind(llpEntry->revMap, &relNum, 0);
	} else {
	    error(sprintf(errBuf, BIND_REPEATED_MSG, hlpName, llpName));
	}
    }
}


#ifndef NDEBUG

static int
dispHlp(key, relNum, arg)
    VOID *key;
    int relNum;
    VOID *arg;
{
    int	*hlpId = (int *)key;

    xTrace2(ptbl, TR_ALWAYS,
	    "      hlp id %d uses rel num %d", *hlpId, relNum);
    return 1;
}


static int
dispEntry(key, value, arg)
    VOID *key;
    int value;
    VOID *arg;
{
    Entry	*e = (Entry *)value;

    xTrace2(ptbl, TR_ALWAYS, "%s (id %d)", e->name, e->id);
    if (e->idMap) {
	xTrace0(ptbl, TR_ALWAYS, "  upper protocols:");
	mapForEach(e->idMap, dispHlp, 0);
    }
    return 1;
}


void
protTblDisplayMap()
{
    xTrace0(ptbl, TR_ALWAYS, "protocol table:");
    mapForEach(ptblNameMap, dispEntry, 0);
}

#endif ! NDEBUG
