/* 
 * ethProtl.c 
 *
 * x-kernel v3.2
 *
 * Copyright (c) 1991  Arizona Board of Regents
 *
 *
 * $Revision: 1.33 $
 * $Date: 1992/02/11 23:28:49 $
 */

/*
 * The xkernel ethernet driver is structured in two layers.
 *
 * The ethernet protocol layer (this file) is independent of any
 * particular ethernet controller hardware.
 * It comprises the usual xkernel protocol functions,
 * e.g. eth_open, eth_push, etc.).
 * It knows about ethernet addresses and "types,"
 * but nothing about the Intel ethernet controller.
 *
 *
 * The interface between the this layer
 * and the ethernet controller (ethCtlr) layer is as follows:
 *
 *	From ethernet to ethCtlr:
 *	  ethCtlrInit( LocalEthAd ) - init controller
 *	  ethCtlrXmit( msgToXmit, destEthAd, msgEthType ) - transmit data
 *	  SetPromiscuous();
 *
 *	From ethCtlr to ethProtl:
 *		eth_demux()
 */


#include "xkernel.h"
#include "eth.h"
#include "eth_i.h"

typedef struct {
    u_short	type;
    ETHhost	host;
} Sstate, ActiveId;

typedef ETHtype  PassiveId;
typedef struct {
    Msg		msg;
    ethType_t	type;
    ETHhost	dst;
} RetMsg;

#define	ETH_Sessn_MAP_SZ	257

int traceethp;
ETHhost ethBcastHost = BCAST_ETH_AD;

static XObj    isPromis = 0;
static ETHhost LocalEthHost;

static bool EthInitted = FALSE;

static Map ActiveMap;
static Map PassiveMap;
static XObj ETH_SELF = 0;

#ifdef __STDC__

static XObj	ethCreateSessn( XObj, XObj, ActiveId * );
static void	ethSessnInit( XObj );
static int	ethControlProtl( XObj, int, char *, int );
static XObj	ethOpen( XObj, XObj, XObj, Part * );
static xkern_return_t	ethOpenEnable( XObj, XObj, XObj, Part * );
static xkern_return_t	ethOpenDisable( XObj, XObj, XObj, Part * );
static xkern_return_t	ethClose( XObj );
static xmsg_handle_t	ethPush( XObj, Msg * );
static xkern_return_t	ethPop( XObj, XObj, Msg * );
static int	ethControlSessn( XObj, int, char *, int );
static long	getRelProtNum( XObj, XObj, char * );

#else

static XObj	ethCreateSessn();
static void	ethSessnInit();
static int	ethControlProtl();
static XObj	ethOpen();
static xkern_return_t	ethOpenEnable();
static xkern_return_t	ethOpenDisable();
static xkern_return_t	ethClose();
static xmsg_handle_t	ethPush();
static xkern_return_t	ethPop();
static int	ethControlSessn();
static long	getRelProtNum();

#endif __STDC__


static long
getRelProtNum( hlp, llp, s )
    XObj	hlp, llp;
    char	*s;
{
    long	n;

    n = relProtNum(hlp, llp);
    if ( n == -1 ) {
	xTrace3(ethp, TR_ERRORS,
		"eth %s could not get prot num of %s relative to %s",
		s, hlp->name, llp->name);
    }
    if ( n < 0 || n > 0xffff ) {
	return -1;
    }
    return n;
}


int
eth_init(self)
     XObj self;
{
  
  xTrace0(ethp, TR_EVENTS, "eth_init");
  xAssert(xIsProtocol(self));
  if (ETH_SELF) {
    Kabort("eth_init: can't initialize more than one eth");
  }
  ETH_SELF = self;
  
  ActiveMap = mapCreate(ETH_Sessn_MAP_SZ, sizeof(ActiveId));
  PassiveMap = mapCreate(ETH_Sessn_MAP_SZ, sizeof(PassiveId));

  self->control = ethControlProtl;
  self->open = ethOpen;
  self->openenable = ethOpenEnable;
  self->opendisable = ethOpenDisable;
  
  getLocalEthHost( &LocalEthHost );
  ethCtlrInit(LocalEthHost);
  EthInitted = TRUE;
  return 0;
} /* end eth_init */



static XObj
ethOpen(self, hlpRcv, hlpType, part)
    XObj self, hlpRcv, hlpType;
    Part *part;
{
    ActiveId  	activeid;
    XObj 	ethSessn;
    ETHhost	*remoteHost;
    long	protNum;
    
    xAssert(xIsProtocol(self));
    xAssert(xIsProtocol(hlpRcv));
    xAssert(xIsProtocol(hlpType));
    xAssert(part && (partLen(part) >= 1));
    
    remoteHost = (ETHhost *)partPop(*part);
    xAssert(remoteHost);
    activeid.host = *remoteHost;
    if ( (protNum = getRelProtNum(hlpType, self, "open")) == -1 ) {
	return ERR_XOBJ;
    }
    activeid.type = protNum;
    xTrace2(ethp, TR_MAJOR_EVENTS, "eth_open: destination address = %s:%4x",
	    ethHostStr(&activeid.host), activeid.type);
    ethSessn = (XObj) mapResolve(ActiveMap, (char *) &activeid);
    
    if (ethSessn == ERR_XOBJ) {
	ethSessn = ethCreateSessn(self, hlpRcv, &activeid);
    }
    xTrace1(ethp, TR_MAJOR_EVENTS, "eth_open: returning %X", ethSessn);
    return (ethSessn);

} /* end eth_open */


static XObj
ethCreateSessn( self, hlp, key )
    XObj	self, hlp;
    ActiveId	*key;
{
    XObj	s;
    Sstate	*sstate;

    s = xCreateSessn(ethSessnInit, hlp, self, 0, NULL);
    s->binding = (Bind) mapBind(ActiveMap, (char *)key, (int)s);
    if ( s->binding == ERR_BIND ) {
	xTrace0(ethp, TR_ERRORS, "error binding in ethCreateSessn");
	xDestroy(s);
	return ERR_XOBJ;
    }
    sstate = X_NEW(Sstate);
    *sstate = *key;
    s->state = (char *)sstate;
    return s;
}


static xkern_return_t
ethOpenEnable(self, hlpRcv, hlpType, part)
    XObj self, hlpRcv, hlpType;
    Part *part;
{
    Enable	*e;
    PassiveId	passiveId;
    long	protNum;
    
    xAssert(xIsProtocol(self));
    xAssert(xIsProtocol(hlpRcv));
    xAssert(xIsProtocol(hlpType));
    if ( (protNum = getRelProtNum(hlpType, self, "openEnable")) == -1 ) {
	return XK_FAILURE;
    }
    passiveId = protNum;
    xTrace2(ethp, TR_GROSS_EVENTS, "eth_openenable: hlp=%d, protlNum=%x",
	    hlpRcv, passiveId);
    e = (Enable *)mapResolve(PassiveMap, (char *)&passiveId);
    if ( e == ERR_ENABLE ) {
	e = X_NEW(Enable);
	e->hlpRcv = hlpRcv;
	e->hlpType = hlpType;
	e->rcnt = 1;
	e->binding = mapBind(PassiveMap, (char *) &passiveId, (int) e);
	xAssert(e->binding != ERR_BIND);
    } else {
	if ( e->hlpRcv == hlpRcv && e->hlpType == hlpType ) {
	    e->rcnt++;
	} else {
	    xTrace0(ethp, TR_SOFT_ERROR, "ethOpenEnable -- existing bind");
	    return XK_FAILURE;
	}
    }
    return XK_SUCCESS;
} /* end eth_openenable */


static xkern_return_t
ethOpenDisable(self, hlpRcv, hlpType, part)
    XObj self, hlpRcv, hlpType;
    Part *part;
{
    Enable	*e;
    long	protNum;
    
    xAssert(xIsProtocol(self));
    xAssert(xIsProtocol(hlpRcv));
    xAssert(xIsProtocol(hlpType));
    
    /* delete the corresponding Sessn from the PassiveMap */
    if ( (protNum = getRelProtNum(hlpType, self, "opendisable")) == -1 ) {
	return XK_FAILURE;
    }
    e = (Enable *)mapResolve(PassiveMap, (char *)&protNum);
    if ( e == ERR_ENABLE || e->hlpRcv != hlpRcv || e->hlpType != hlpType ) {
	return XK_FAILURE;
    }
    if ( --e->rcnt <= 0 ) {
	mapRemoveBinding(PassiveMap, e->binding);
	xFree( (char *)e );
    }
    return XK_SUCCESS;
} /* end eth_opendisable */


#ifdef XKMACHKERNEL
/* see xklance.c for matching definition */
struct argblock {
	  Msg		msg;
	  ethType_t	type;
	  ETHhost	src, dest;
	};
#endif XKMACHKERNEL


xkern_return_t
eth_demux
#ifndef XKMACHKERNEL
         (incomingMsg, ethType, srcEthAd, destEthAd)
    Msg incomingMsg;
    ethType_t ethType;
    ETHhost srcEthAd;
    ETHhost destEthAd;
#else
         (args)
    struct argblock *args;
#endif ! XKMACHKERNEL
{
    ActiveId  activeid;
    PassiveId passiveid;
    XObj ethSessn;
    Enable *e;
#ifdef XKMACHKERNEL
    Msg incomingMsg;
    ethType_t ethType;
    ETHhost srcEthAd;
    ETHhost destEthAd;

    incomingMsg = args->msg;
    ethType = args->type;
    srcEthAd = args->src;
    destEthAd = args->dest;
    xFree(args);
#endif XKMACHKERNEL
    
    activeid.type = ethType;
    activeid.host = srcEthAd;
    
    xTrace0(ethp, TR_EVENTS, "eth_demux");
    xTrace1(ethp, TR_FUNCTIONAL_TRACE, "eth type: %x", ethType);
    xTrace2(ethp, TR_FUNCTIONAL_TRACE, "src: %s  dst: %s",
	    ethHostStr(&srcEthAd), ethHostStr(&destEthAd));
    xIfTrace(ethp, TR_DETAILED) msgShow(&incomingMsg);
    if (isPromis) {
	Msg      	p;
	ETHhdr	hdr;
	
	xTrace0(ethp, TR_EVENTS, "eth_demux: passing msg to promiscuous session");
	msgConstructCopy(&p, &incomingMsg);
	hdr.dst = destEthAd;
	hdr.src = srcEthAd;
	hdr.type = ethType;
	msgSetAttr(&p, (VOID *)&hdr);
	xDemux(isPromis, &p);
    }
    /* verify that msg is for this host */
    if ((ETH_ADS_EQUAL(destEthAd, LocalEthHost) ||
	 ETH_ADS_EQUAL(destEthAd, ethBcastHost))) {
	xTrace0(ethp, TR_EVENTS, "eth_demux: msg is for this host");
	ethSessn = (XObj) mapResolve(ActiveMap, (char *) &activeid);
	if (ethSessn != ERR_XOBJ) {
	    /* found the corresponding Sessn */
	    xTrace0(ethp, TR_EVENTS, "eth_demux: sending message to active session");
	    xPop(ethSessn, 0, &incomingMsg);
	} else {
	    /* Sessn doesn't exist; check for openenable */
	    passiveid = activeid.type;
	    e = (Enable *) mapResolve(PassiveMap, (char *) &passiveid);
	    if (e != ERR_ENABLE) {
		/* there is an openenable for this msg */
		Part	p;
		
		xTrace1(ethp, TR_EVENTS,
			"eth_demux: openenable exists for msg type %x",
			passiveid);
		if ( ethType != relProtNum(e->hlpType, ETH_SELF) ) {
		    xTrace2(ethp, TR_ERRORS,
    "eth map inconsistency -- lookup type %d found enable with type %d",
			    ethType, relProtNum(e->hlpType, ETH_SELF));
		    return XK_FAILURE;
		}
		partInit(&p, 1);
		partPush(p, &srcEthAd);
		ethSessn = ethCreateSessn(ETH_SELF, e->hlpRcv, &activeid);
		xOpenDone(ethSessn, e->hlpType);
		xTrace0(ethp, TR_EVENTS, "eth_demux: sending message to new session");
		xPop(ethSessn, 0, &incomingMsg);
	    } else {
		xTrace1(ethp, TR_EVENTS,
			"eth_demux: openenable does not exist for msg type %x",
			passiveid);
	    }
	}
    } else {
	xTrace0(ethp, TR_EVENTS, "eth_demux: msg is not for this host");
    }
    msgDestroy(&incomingMsg);
    return XK_SUCCESS;
} /* end eth_demux */


static xkern_return_t
ethClose(ethSessn)
    XObj ethSessn;
{
    xTrace1(ethp, TR_MAJOR_EVENTS, "eth closing session %x", ethSessn);
    xAssert( xIsSession( ethSessn ) );
    xAssert( ethSessn->rcnt <= 0 );
    mapRemoveBinding( ActiveMap, ethSessn->binding );
    xDestroy( ethSessn );
    return XK_SUCCESS;
} /* end eth_close */


static void
demuxStub(r)
    VOID *r;
{
    eth_demux(((RetMsg *)r)->msg, ((RetMsg *)r)->type,
	     LocalEthHost, ((RetMsg *)r)->dst);
    xFree((char *)r);
}


void
ethSendUp(m, dst, type)
    Msg *m;
    ETHhost *dst;
    ethType_t type;
{
    RetMsg	*r;

    r = X_NEW(RetMsg);
    r->type = type;
    msgConstructCopy(&r->msg, m);
    r->dst = *dst;
    evDetach( evSchedule(demuxStub, r, 0) );
}



static xmsg_handle_t
ethPush(ethSessn, msg)
    XObj ethSessn;
    Msg *msg;
{
    Sstate *dstAddr;
    
    xTrace0(ethp, TR_EVENTS, "eth_push");
    xAssert(xIsSession(ethSessn));
    
    dstAddr = (Sstate *) (ethSessn->state);
    xTrace2(ethp, TR_EVENTS, "eth_push state: %s:%4x",
	    ethHostStr(&dstAddr->host), dstAddr->type);

    if (ETH_ADS_EQUAL(dstAddr->host, LocalEthHost)) {
	/*
	 * local msg; goes back up only
	 */
	ethSendUp(msg, &dstAddr->host, dstAddr->type);
    } else {
	ethCtlrXmit(msg, &dstAddr->host, dstAddr->type);
    }
    
    return XMSG_NULL_HANDLE;
    
} /* end eth_push */


static xkern_return_t
ethPop(ethSessn, deliverySessn, inMsg)
    XObj ethSessn;
    XObj deliverySessn;
    Msg *inMsg;
{
  
  xAssert(xIsSession(ethSessn));
  return xDemux(ethSessn, inMsg);
  
} /* end eth_pop */


static int
ethControlSessn(ethSessn, opcode, bufPtr, bufLen)
    XObj ethSessn;
    int opcode;
    char *bufPtr;
    int bufLen;
{
    Sstate *sstate;
    
    xAssert(xIsSession(ethSessn));
    
    sstate = (Sstate *) ethSessn->state;
    switch (opcode) {
      case GETMYHOST:
	return ethControlProtl(ethSessn->myprotl, opcode, bufPtr, bufLen);
	
      case GETMAXPACKET:
      case GETOPTPACKET:
	checkLen(bufLen, sizeof(int));
	*(int *) bufPtr = MAX_ETH_DATA_SZ;
	return (sizeof(int));
	
      case GETPEERHOST:
	checkLen(bufLen, sizeof(ETHhost));
	bcopy((char *) &sstate->host, bufPtr, sizeof(ETHhost));
	return (sizeof(ETHhost));
	
      case GETMYHOSTCOUNT:
      case GETPEERHOSTCOUNT:
	checkLen(bufLen, sizeof(int));
	*(int *)bufPtr = 1;
	return sizeof(int);
	
      case GETMYPROTO:
      case GETPEERPROTO:
	checkLen(bufLen, sizeof(long));
	*(long *) bufPtr = sstate->type;
	return sizeof(long);
	
      case GETPARTICIPANTS:
	{
	    Part	*p = (Part *)bufPtr;
	    ETHhost	*h;

	    checkLen(bufLen, sizeof(Part));
	    partInit(p, 1);
	    /* 
	     * Remote host
	     */
	    h = X_NEW(ETHhost);
	    *h = sstate->host;
	    partPush(p[0], h);
	    return sizeof(Part);
	}

      case ETH_SETPROMISCUOUS:
	checkLen(bufLen, sizeof(int));
	isPromis = ethSessn;
	return ( SetPromiscuous() == 0 ) ? sizeof(int) : -1;
	
      default:
	return -1;
    }
} /* end eth_controlSessn */


static int
ethControlProtl(self, opcode, bufPtr, bufLen)
    XObj self;
    int opcode;
    char *bufPtr;
    int bufLen;
{
  xAssert(xIsProtocol(self));
  
  switch (opcode) {
  case GETMYHOST:
      checkLen(bufLen, sizeof(ETHhost));
      bcopy((char *) &LocalEthHost, bufPtr, sizeof(ETHhost));
      return (sizeof(ETHhost));

  default:
    return -1;
  }
  
} /* end eth_controlprotl */


static void
ethSessnInit(s)
    XObj s;
{
  s->push = ethPush;
  s->pop = ethPop;
  s->close = ethClose;
  s->control = ethControlSessn;
}


