/*     
 * ip.c
 *
 * x-kernel v3.2
 *
 * Copyright (c) 1991  Arizona Board of Regents
 *
 *
 * $Revision: 1.79 $
 * $Date: 1992/02/11 23:29:13 $
 */


#include "xkernel.h"
#include "gc.h"
#include "eth.h"
#include "ip.h"
#include "ip_i.h"
#include "arp.h"
#include "route.h"


#ifdef __STDC__

extern void	scheduleIpFragCollector( PSTATE *pstate );
static long	getRelProtNum( XObj, XObj, char * );
static void	displayFragTable( Fragtable * );
static void 	dump_header(IPheader *);
static int 	get_ident(IPhost *, int );
static void 	getnet(IPhost *, IPhost *);
static void 	hole_create(Fraginfo *, int, int);
static xkern_return_t 	ipForward( XObj, Msg *, IPheader * );
static void 	ip_init_sessn(XObj);
static XObj	ipOpenInterface( XObj, XObj, IPhost * );
static xkern_return_t 	ip_reassemble(XObj, XObj, Msg *, IPheader *);
static xmsg_handle_t	ip_send( XObj s, Msg *msg, IPheader *hdr );
static int 	is_my_addr(PSTATE *, IPhost *);
static void	ipHdrStore(void *internal, char *external, long len, void *);
static int 	ipHandleRedirect( XObj );
static void	ipDestroySessn( XObj s );
static XObj	ipCreateSessn( XObj, XObj, ActiveId *, IPhost * );
static int	routeChangeFilter( void *, int, void * );

#else

extern void	scheduleIpFragCollector();
static long	getRelProtNum();
static void	displayFragTable();
static void 	dump_header();
static int 	get_ident();
static void 	getnet();
static void 	hole_create();
static xkern_return_t 	ipForward();
static void 	ip_init_sessn();
static XObj	ipOpenInterface();
static xkern_return_t 	ip_reassemble();
static xmsg_handle_t	ip_send();
static int 	is_my_addr();
static void	ipHdrStore();
static int 	ipHandleRedirect();
static void	ipDestroySessn();
static XObj	ipCreateSessn();
static int	routeChangeFilter();

#endif __STDC__

static xkern_return_t 	ipCloseProtl();
static xkern_return_t 	ipCloseSessn();
static xkern_return_t 	ipOpenDisable();
static xkern_return_t 	ipOpenEnable();
static xkern_return_t 	ipDemux();
static XObj		ipOpen();
static xkern_return_t	ipPop();
static xmsg_handle_t 	ipPush();


int traceipp;

static IPhost	siteIpGateway = SITE_IP_GTW ;

#define SESSN_COLLECT_INTERVAL	20 * 1000 * 1000	/* 20 seconds */
#define IP_MAX_PROT	0xff

#define IP_BCAST( bcast, host, mask ) {		\
	(bcast).a = (host).a | ~ (mask).a;	\
	(bcast).b = (host).b | ~ (mask).b;	\
	(bcast).c = (host).c | ~ (mask).c;	\
	(bcast).d = (host).d | ~ (mask).d;	\
     }


static long
getRelProtNum( hlp, llp, s )
    XObj	hlp, llp;
    char	*s;
{
    long	n;

    n = relProtNum(hlp, llp);
    if ( n == -1 ) {
	xTrace3(ipp, TR_SOFT_ERROR,
	       "%s: couldn't get prot num of %s relative to %s",
	       s, hlp->name, llp->name);
	return -1;
    }
    if ( n < 0 || n > 0xff ) {
	xTrace4(ipp, TR_SOFT_ERROR,
	       "%s: prot num of %s relative to %s (%d) is out of range",
	       s, hlp->name, llp->name, n);
	return -1;
    }
    return n;
}

/*
 * Note:   *hdr will be modified
 */
static void
ipHdrStore(hdr, dst, len, arg)
    VOID *hdr;
    char *dst;
    long int len;
    VOID *arg;
{
    /*
     * XXX -- this needs to be revised to include options
     */
    xAssert(len == sizeof(IPheader));
    ((IPheader *)hdr)->dlen = htons(((IPheader *)hdr)->dlen);
    ((IPheader *)hdr)->ident = htons(((IPheader *)hdr)->ident);
    ((IPheader *)hdr)->frag = htons(((IPheader *)hdr)->frag);
    ((IPheader *)hdr)->checksum = 0;
    ((IPheader *)hdr)->checksum = ~ocsum((u_short *)hdr, sizeof(IPheader) / 2);
    xAssert(! (~ ocsum( (u_short *)hdr, sizeof(IPheader) / 2 ) & 0xFFFF ));
    bcopy ( (char *)hdr, dst, sizeof(IPheader) );
}


/*
 * checksum over the the header is written into the checksum field of
 * *hdr.
 */
static long
ipStdHdrLoad(hdr, src, len, arg)
    VOID *hdr;
    char *src;
    long int len;
    VOID *arg;
{
    xAssert(len == sizeof(IPheader));
    bcopy(src, (char *)hdr, sizeof(IPheader));
    ((IPheader *)hdr)->checksum =
      ~ ocsum((u_short *)hdr, sizeof(IPheader) / 2) & 0xFFFF;
    ((IPheader *)hdr)->dlen = ntohs(((IPheader *)hdr)->dlen);
    ((IPheader *)hdr)->ident = ntohs(((IPheader *)hdr)->ident);
    ((IPheader *)hdr)->frag = ntohs(((IPheader *)hdr)->frag);
    return sizeof(IPheader);
}


static long
ipOptionsLoad(hdr, netHdr, len, arg)
    VOID *hdr;
    char *netHdr;
    long int len;
    VOID *arg;
{
    bcopy(netHdr, (char *)hdr, len);
    *(u_short *)arg = ~ocsum((u_short *)hdr, len / 2);
    return len;
}


/*
 * ip_init: main entry point to IP
 */
void
ip_init(self)
    XObj self;
{
  PSTATE	*pstate;
  Part		part;
  Interfaceid	myinterfaces[ARP_MAXINTERFACES];
  int		arpifnum = -1, i;
  
  xTrace0(ipp, 1, "IP init");
  xAssert(xIsProtocol(self));
  
  initNetMaskMap();
#ifdef IP_SIM_DELAYS
  xError("Warning: IP is simulating delayed packets");
#endif
#ifdef IP_SIM_DROPS
  xError("Warning: IP is simulating dropped packets");
#endif
  /* set up function pointers for IP protocol object */
  self->open = ipOpen;
  self->close = ipCloseProtl;
  self->control = ipControlProtl;
  self->openenable = ipOpenEnable;
  self->opendisable = ipOpenDisable;
  self->demux = ipDemux;

  /* initialize protocol-specific state */
  pstate = (PSTATE *) xMalloc(sizeof(PSTATE));
  self->state = (char *) pstate;
  pstate->activeMap = mapCreate(101, sizeof(ActiveId));
  pstate->passiveMap = mapCreate(23, sizeof(PassiveId));
  pstate->passiveSpecMap = mapCreate(23, sizeof(PassiveSpecId));
  pstate->fragMap = mapCreate(23, sizeof(FragId));
  xTrace1(ipp, 2, "IP has %d protocols below\n", self->numdown);

  /* search for ARP protocol */
  for( i = 0; i < self->numdown; i++ ) {
      xTrace1(ipp,1,"IP asking protocol %d for interface info",i);
      if ( (pstate->n_interfaces = xControl(xGetDown(self,i),
        ARP_GETIPINTERFACES, (char *)(myinterfaces),
        ARP_MAXINTERFACES * sizeof(Interfaceid))) != -1) {
 	arpifnum = i;
	break;
      }
      xTrace1(ipp,1,"IP got bogus reply from protocol %d",i);
  }
  if (arpifnum == -1) {
    	xTrace0(ipp,1,"Could not find ARP protocol");
  	return;
  }
  xTrace1(ipp,1,"IP found ARP at protocol number %d",arpifnum);
  xTrace1(ipp, 1, "IP: # of interfaces = %d", pstate->n_interfaces);

  for( i = 0; i < pstate->n_interfaces; i++ ) {
      IPinterfaceinfo	*ifp = &pstate->interface[i];

      partInit(&part, 1);
      ifp->arpprotl = xGetDown(self,arpifnum);
      ifp->ethprotl = myinterfaces[i].ifprotl;
      ifp->myipaddr = myinterfaces[i].ipaddr;
      ipNetMask(&ifp->netmask, &ifp->myipaddr);
      IP_BCAST(ifp->ipbcast, ifp->myipaddr, ifp->netmask);
      pstate->interface[i].myethhost = myinterfaces[i].ethaddr;
      /* 
       * Openenable with ANY_HOST to pick up both host-specific and broadcast 
       */
      partPush(part, ANY_HOST);	
      /* openenable ethernet */
      xTrace1(ipp,1,"IP calls xOpenEnable on ETH for interface %d",i);
      if ( xOpenEnable(self, self, pstate->interface[i].ethprotl, &part) ==
	   XK_FAILURE ) {
	  xTrace0(ipp,1,"ip_init : can't openenable net protocol");
	  xFree((char *)pstate);
	  return;
      }
      xTrace1(ipp,1,"IP xOpenEnable on ETH for interface %d OK",i);
  }
  
  /* initialize route table and set up default route */
  /* if (pstate->n_interfaces > 1) { */
  rt_init( pstate, &siteIpGateway );
  scheduleIpFragCollector(pstate);
  initSessionCollector(pstate->activeMap, SESSN_COLLECT_INTERVAL,
		       ipDestroySessn, "ip");

  xTrace0(ipp, 1, "IP init done");
}


/*
 * ipOpen
 */
static XObj
ipOpen(self, hlpProxy, hlpType, p)
    XObj self, hlpProxy, hlpType;
    Part *p;
{
    XObj	ip_s;
    IPhost      *remoteHost;
    IPhost      *localHost = 0;
    ActiveId    activeid;
    long	hlpNum;
    
    
    xTrace0(ipp, 3, "IP open");
    xAssert(xIsProtocol(self));
    xAssert(xIsProtocol(hlpProxy));
    xAssert(xIsProtocol(hlpType));

    if ( !p || (partLen(p) < 1) || partLen(p) > 2 ) {
	xTrace0(ipp, TR_MAJOR_EVENTS, "ipOpen: participant list error");
	return ERR_XOBJ;
    }
    remoteHost = (IPhost *)partPop(p[0]);
    if ( remoteHost == 0 ) {
	xTrace0(ipp, TR_MAJOR_EVENTS, "ipOpen: empty participant stack");
	return ERR_XOBJ;
    }
    if ( (hlpNum = getRelProtNum(hlpType, self, "open")) == -1 ) {
	return ERR_XOBJ;
    }
    if ( partLen(p) > 1 ) {
	/* 
	 * Local participant has been explicitly specified
	 */
	localHost = (IPhost *)partPop(p[1]);
	if ( localHost == (IPhost *)ANY_HOST ) {
	    localHost = 0;
	}
    }
    xTrace2(ipp, 5, "IP sends to %s, %d", ipHostStr(remoteHost), hlpNum);
    
    /*
     * key on hlp prot number, destination addr, and local addr (if given)
     */
    activeid.protNum = hlpNum;
    activeid.remote = *remoteHost;
    ip_s = ipCreateSessn( self, hlpProxy, &activeid, localHost );
    if ( ip_s != ERR_XOBJ ) {
	ip_s->idle = FALSE;
    }
    xTrace1(ipp, 3, "IP open returns %x", ip_s);
    return ip_s;
}


/* 
 * Create an IP session which sends to remote host key->dest.  The
 * 'rem' and 'prot' fields of 'key' will be used as passed in.
 * If localHost is non-null, use it as the localHost for this connection.
 * If localHost is null, pick an appropriate localHost.
 */
static XObj
ipCreateSessn(self, hlp, key, localHost)
    XObj self;
    XObj hlp;
    ActiveId *key;
    IPhost *localHost;
{
    XObj	ip_s;
    SSTATE	*sstate;
    PSTATE	*pstate;
    IPheader	*iph;
    IPinterfaceinfo	*ifInfo;
    
    pstate = (PSTATE *)self->state;
    ip_s = xCreateSessn(ip_init_sessn, hlp, self, 0, 0);
    sstate = X_NEW(SSTATE);
    ip_s->state = (VOID *)sstate;
    bzero((char *)sstate, sizeof(SSTATE));
    sstate->dest = key->remote;

    if ( ipHandleRedirect(ip_s) ) {
	xTrace0(ipp, 3, "IP open fails");
	ipDestroySessn(ip_s);
	return ERR_XOBJ;
    }
    xAssert(xIsProtocol(sstate->llp));
    /*
     * Determine my host address
     */
    if ( localHost ) {
	if ( is_my_addr(pstate, localHost) == -1 ) {
	    xTrace1(ipp, TR_SOFT_ERROR, "%s is not a local IP host",
		    ipHostStr(localHost));
	    return ERR_XOBJ;
	}
    } else {
	ifInfo = ipLookupIfInfo(pstate, sstate->llp);
	if ( ifInfo == 0 ) {
	    xTrace0(ipp, TR_SOFT_ERROR,
		    "IP open could not get interface info for remote host");
	    ipDestroySessn(ip_s);
	    return ERR_XOBJ;
	}
	localHost = &ifInfo->myipaddr;
    }
    iph = &(sstate->hdr);
    key->local = iph->source = *localHost;

    ip_s->binding = mapBind(pstate->activeMap, (char *)key, (int)ip_s);
    if ( ip_s->binding == ERR_BIND ) {
	xTrace0(ipp, TR_MAJOR_EVENTS, "IP open -- session already existed");
	ipDestroySessn(ip_s);
	return (XObj)mapResolve(pstate->activeMap, (char *)key);
    }
    xTrace3(ipp,5,"IP open bind key l: %s r: %s %d",
	    ipHostStr(&key->local), ipHostStr(&key->remote), key->protNum);

    /* fill in session template header */
    iph->vers_hlen = IPVERS;
    iph->vers_hlen |= 5;	/* default hdr length */
    iph->type = 0;
    iph->time = IPDEFAULTDGTTL;
    iph->prot = key->protNum;
    iph->dest = key->remote;
    xTrace1(ipp, 4, "IP open: my ip address is %s",
	    ipHostStr(&iph->source));
    return ip_s;
}


static void
ip_init_sessn(self)
     XObj self;
{
  self->push = ipPush;
  self->pop = ipPop;
  self->control = ipControlSessn;
  self->close = ipCloseSessn;
}


/* 
 * Binds an Enable object to this key
 */
static xkern_return_t
passiveBind(map, key, hlpRcv, hlpType)
    Map map;
    VOID *key;
    XObj hlpRcv, hlpType;
{
    Enable	*e;

    e = (Enable *) mapResolve(map, (char *)key);
    if ( e != ERR_ENABLE ) {
	if ( e->hlpRcv == hlpRcv && e->hlpType == hlpType ) {
	    e->rcnt++;
	    xTrace1(ipp, TR_EVENTS,
		    "ipOpenEnable increasing ref count of existing bind to %d",
		    e->rcnt);
	} else {
	    xTrace0(ipp, TR_EVENTS,
		    "ipOpenenable error -- existing bind");
	    return XK_FAILURE;
	}
    } else {
	xTrace0(ipp, TR_EVENTS, "ipOpenEnable -- new bind");
	e = X_NEW(Enable);
	e->hlpRcv = hlpRcv;
	e->hlpType = hlpType;
	e->rcnt = 1;
	e->binding = mapBind(map, (char *)key, (int)e);
    }
    return XK_SUCCESS;
}


/*
 * ipOpenEnable
 */
static xkern_return_t
ipOpenEnable(self, hlpRcv, hlpType, p)
    XObj self, hlpRcv, hlpType;
    Part *p;
{
    PSTATE 	*pstate;
    IPhost	*localHost;
    long	protNum;
    
    xTrace0(ipp, TR_MAJOR_EVENTS, "IP open enable");
    xAssert(xIsProtocol(self));
    xAssert(xIsProtocol(hlpRcv));
    xAssert(xIsProtocol(hlpType));
    
    pstate = (PSTATE *)self->state;
    if ( (localHost = (IPhost *)partPop(*p)) == 0 ) {
	return XK_FAILURE;
    }
    if ( (protNum = getRelProtNum(hlpType, self, "ipOpenEnable")) == -1 ) {
	return XK_FAILURE;
    }
    if ( localHost == (IPhost *)ANY_HOST ) {
	xTrace1(ipp, TR_MAJOR_EVENTS, "ipOpenEnable binding protocol %d",
		protNum);
	return passiveBind(pstate->passiveMap, &protNum, hlpRcv, hlpType);
    } else {
	PassiveSpecId	key;

	if ( is_my_addr(pstate, localHost) == -1 ) {
	    xTrace1(ipp, TR_MAJOR_EVENTS,
		    "ipOpenEnable -- %s is not one of my hosts",
		    ipHostStr(localHost));
	    return XK_FAILURE;
	}
	key.host = *localHost;
	key.prot = protNum;
	xTrace2(ipp, TR_MAJOR_EVENTS,
		"ipOpenEnable binding protocol %d, host %s",
		key.prot, ipHostStr(&key.host));
	return passiveBind(pstate->passiveSpecMap, &key, hlpRcv, hlpType);
    }
}


static xkern_return_t
passiveUnbind(m, key)
    Map 	m;
    VOID	*key;
{
    Enable	*e;

    if ( (e = (Enable *)mapResolve(m, (char *)key)) == ERR_ENABLE ) {
	xTrace0(ipp, TR_MAJOR_EVENTS,
		"ipOpenDisable -- could not find enable object");
	return XK_FAILURE;
    }
    if ( --e->rcnt <= 0 ) {
	xTrace0(ipp, TR_EVENTS, "ipOpenDisable -- removed object");
	return ( mapUnbind(m, (char *)key) == 0 ) ? XK_SUCCESS : XK_FAILURE;
    }
    xTrace1(ipp, TR_EVENTS, "ipOpenDisable -- decreased rcnt to %d",
	    e->rcnt);
    return XK_SUCCESS;
}


/*
 * ipOpenDisable
 */
static xkern_return_t
ipOpenDisable(self, hlpRcv, hlpType, p)
    XObj self, hlpRcv, hlpType;
    Part *p;
{
    PSTATE      *pstate;
    IPhost	*localHost;
    long	protNum;
    
    xTrace0(ipp, 3, "IP open disable");
    xAssert(xIsProtocol(self));
    xAssert(self->state);
    xAssert(xIsProtocol(hlpRcv));
    xAssert(xIsProtocol(hlpType));
    xAssert(p);

    if ( (protNum = getRelProtNum(hlpType, self, "ipOpenDisable")) == -1 ) {
	return XK_FAILURE;
    }
    if ( (localHost = (IPhost *)partPop(*p)) == 0 ) {
	return XK_FAILURE;
    }
    pstate = (PSTATE *)self->state;
    if ( localHost == (IPhost *)ANY_HOST ) {
	xTrace1(ipp, TR_MAJOR_EVENTS,
		"ipOpenDisable unbinding protocol %d", protNum);
	return passiveUnbind(pstate->passiveMap, &protNum);
    } else {
	PassiveSpecId	key;

	key.host = *localHost;
	key.prot = protNum;
	xTrace2(ipp, TR_MAJOR_EVENTS,
		"ipOpenDisable unbinding protocol %d, host %s",
		key.prot, ipHostStr(&key.host));
	return passiveUnbind(pstate->passiveSpecMap, &key);
    }
}


/*
 * ipCloseSessn
 */
static xkern_return_t
ipCloseSessn(s)
    XObj s;
{
  xTrace1(ipp, 3, "IP close of session %x (does nothing)", s);
  xAssert(xIsSession(s));
  xAssert( s->rcnt == 0 );
  return XK_SUCCESS;
}


static void
ipDestroySessn(s)
    XObj s;
{
    PSTATE 	*pstate;
    int		i;
    
    xTrace1(ipp, 3, "IP DestroySessn %x", s);
    xAssert(xIsSession(s));
    pstate = (PSTATE *)s->myprotl->state;
    if ( s->binding && s->binding != ERR_BIND ) {
	mapRemoveBinding(pstate->activeMap, s->binding);
    }
    for (i=0; i < s->numdown; i++ ) {
	xClose(xGetDown(s,i));
    }
    xDestroy(s);
}    
  


/*
 * ipCloseProtl
 */
static xkern_return_t
ipCloseProtl(self)
    XObj self;
{
  PSTATE        *pstate;
  
  xAssert(xIsProtocol(self));
  xAssert(self->rcnt==1);
  
  pstate = (PSTATE *) self->state;
  mapClose(pstate->activeMap);
  mapClose(pstate->passiveMap);
  mapClose(pstate->fragMap);
  xFree((char *) pstate);
  xDestroy(self);
  return XK_SUCCESS;
}

/*
 * ipPush
 */
static xmsg_handle_t
ipPush(s, msg)
    XObj s;
    Msg *msg;
{
  PSTATE	*pstate;
  SSTATE	*sstate;
  IPheader	hdr;

  xAssert(xIsSession(s));
  sstate = (SSTATE *) s->state;
  pstate = (PSTATE *) s->myprotl->state;

  hdr = sstate->hdr;
  hdr.ident = get_ident(&(sstate->dest),hdr.prot);
#ifdef OLD_FLAGS
  hdr.frag &= 0x0004;    
#endif
  /*
   * Clear DONTFRAGMENT and MOREFRAGS
   */
  hdr.frag = 0;

  hdr.dlen = msgLen(msg) + (GET_HLEN(&hdr) * 4);
  return ip_send(s, msg, &hdr);
}


/*
 * Send the msg over the ip session's down session, fragmenting if necessary.
 * All header fields not directly related to fragmentation should already
 * be filled in.  We only reference the 'mtu' field of s->state.
 */
static xmsg_handle_t
ip_send(s, msg, hdr)
    XObj s;
    Msg *msg;
    IPheader *hdr;
{
    int 	hdrLen;
    int		len;
    SSTATE	*sstate;

    sstate = (SSTATE *)s->state;
    len = msgLen(msg);
    hdrLen = GET_HLEN(hdr);
    if ( len + hdrLen * 4 <= sstate->mtu ) {
	/*
	 * No fragmentation
	 */
	xTrace0(ipp,5,"IP send : message requires no fragmentation");
#ifdef OLD_FLAGS
	hdr->frag &= 0x0003; 
#else
	hdr->frag &= ~FRAGOFFMASK;		/* Set offset to 0 */
	hdr->frag &= ~MOREFRAGMENTS;  	/* Clear MOREFRAGMENTS */
	/*
	 * Note: DONTFRAGMENT flag is preserved
	 */
#endif    
	msgPush(msg, ipHdrStore, hdr, hdrLen * 4, NULL);
	xIfTrace(ipp,5) {
	    xTrace0(ipp,5,"IP send unfragmented datagram header: \n");
	    dump_header(hdr);
	}
	return xPush(xGetDown(s, 0), msg);
    } else {
	/*
	 * Fragmentation required
	 */
	int 	fragblks;
	int	fragsize;
	Msg	fragmsg;
	int	offset;
	int	fraglen;
	xmsg_handle_t handle = XMSG_NULL_HANDLE;
	
	if ( hdr->frag & DONTFRAGMENT ) {
	    xTrace0(ipp,5,
		    "IP send: fragmentation needed, but NOFRAG bit set");
	    return XMSG_NULL_HANDLE;  /* drop it */
	}
	fragblks = (sstate->mtu - (hdrLen * 4)) / 8;
	fragsize = fragblks * 8;
	xTrace0(ipp,5,"IP send : datagram requires fragmentation");
	xIfTrace(ipp,5) {
	    xTrace0(ipp,5,"IP original datagram header :");
	    dump_header(hdr);
	}
	/* fragmsg = msg; */
	msgConstructEmpty(&fragmsg);
	for( offset = 0; len > 0; len -= fragsize, offset += fragblks) {
	    IPheader  	hdrToPush;
	    XObj	lls;
	    
	    hdrToPush = *hdr;
	    fraglen = len > fragsize ? fragsize : len;
	    msgChopOff(msg, &fragmsg, fraglen);
	    /*
	     * eventually going to need to selectively copy options
	     */
	    hdrToPush.frag = offset;
	    if ( fraglen == len ) {/* last frag */
		hdrToPush.frag &= ~MOREFRAGMENTS;
	    } else {
		hdrToPush.frag |= MOREFRAGMENTS;
	    }
	    hdrToPush.dlen = hdrLen * 4 + fraglen;
	    xIfTrace(ipp,5) {
		xTrace0(ipp,5,"IP datagram fragment header: \n");
		dump_header(&hdrToPush);
	    }
	    msgPush(&fragmsg, ipHdrStore, &hdrToPush, hdrLen * 4, NULL);
	    lls = xGetDown(s, 0);
	    xAssert(xIsSession(lls));
	    if ( (handle =  xPush(lls, &fragmsg)) == XMSG_ERR_HANDLE ) {
		break;
	    }
	}
	msgDestroy(&fragmsg);
	return ( handle == XMSG_ERR_HANDLE ) ? handle : XMSG_NULL_HANDLE;
    }
}


/*
 * This is a bit ugly.  The checksum and network-to-host byte conversion
 * can't nicely take place in the load function because options are possible,
 * the checksum is calculated over the entire header (including options),
 * and the checksum is done over the data in network-byte order.
 */
static int
getHdr(msg, h, options)
    Msg *msg;
    IPheader *h;
    char *options;
{
    u_char hdrLen;

    if (! msgPop(msg, ipStdHdrLoad, h, IPHLEN, NULL)) {
	xTrace0(ipp, 3, "ip getHdr: msg too short!");
	return -1;
    }
    xIfTrace(ipp, 7) {
	xTrace0(ipp, 7, "ip getHdr: received header:");
	dump_header(h);
    }
    hdrLen = GET_HLEN(h);
    if (hdrLen == 5) {
	/*
	 * No options
	 */
	if (h->checksum) {
	    xTrace0(ipp, 3, "ip getHdr: bad checksum!");
	    return -1;
	}
	return 0;
    }
    if (hdrLen > 5) {
	/*
	 * IP options
	 */
	u_short cksum[2];
	int optBytes;
	
	optBytes = (hdrLen - 5) * 4;
	cksum[0] = h->checksum;
	if (! msgPop(msg, ipOptionsLoad, options, optBytes, &cksum[1])) {
	    xTrace0(ipp, 3, "ip getHdr: options component too short!");
	    return -1;
	}
	/*
	 * Add the regular header checksum with the options checksum
	 */
	if ( ~ocsum( cksum, 2 ) ) {
	    xTrace0(ipp, 3, "ip getHdr: bad checksum (with options)!");
	    return -1;
	}
	return 0;
    }
    xTrace1(ipp, 3, "ip getHdr: hdr length (%d) < 5 !!", hdrLen);
    return -1;
}


static XObj
createPassiveSessn(self, actKey, hdr)
    XObj self;
    ActiveId *actKey;
    IPheader *hdr;
{
    PSTATE		*pstate = (PSTATE *)self->state;
    PassiveId		key;
    PassiveSpecId	specKey;
    Enable		*e;
    XObj		s;

    key = hdr->prot;
    e = (Enable *) mapResolve(pstate->passiveMap, (char *)&key);
    if ( e != ERR_ENABLE ) {
	    xTrace1(ipp, TR_MAJOR_EVENTS,
		    "Found an enable object for prot %d", key);
    } else {
	specKey.prot = hdr->prot;
	specKey.host = hdr->dest;
	e = (Enable *) mapResolve(pstate->passiveSpecMap, (char *)&key);
	if ( e != ERR_ENABLE ) {
	    xTrace2(ipp, TR_MAJOR_EVENTS,
		    "Found an enable object for prot %d host %s",
		    specKey.prot, ipHostStr(&specKey.host));
	}
    }
    if ( e == ERR_ENABLE ) {
	return ERR_XOBJ;
    }
    s = ipCreateSessn(self, e->hlpRcv, actKey, &hdr->dest);
    if ( s != ERR_XOBJ ) {
	xOpenDone(s, e->hlpType);
    }
    return s;
}


/*
 * ipDemux
 */
static xkern_return_t
ipDemux(self, transport_s, dg)
    XObj self;
    XObj transport_s;
    Msg *dg;
{
    IPheader	hdr;
    XObj        s;
    ActiveId    activeid;
    PSTATE	*pstate;
    int		dataLen;
    char	options[40];
    
    xTrace1(ipp, TR_EVENTS,
	    "IP demux called with datagram of len %d", msgLen(dg));
    xAssert(xIsSession(transport_s));
    xAssert(xIsProtocol(self));
    
    pstate = (PSTATE *) self->state;
    if (getHdr(dg, &hdr, options)) {
	xTrace0(ipp, TR_SOFT_ERRORS,
		"IP demux : getHdr problems, dropping message\n"); 
	return XK_SUCCESS;
    }
    xTrace3(ipp, TR_MORE_EVENTS,
	    "ipdemux: seq=%d,frag=%d, len=%d", hdr.ident, hdr.frag,
	    msgLen(dg));
    if (GET_HLEN(&hdr) > 5) {
	xTrace0(ipp, TR_SOFT_ERRORS,
		"IP demux: I don't understand options!  Dropping msg");
	return XK_SUCCESS;
    }
    dataLen = hdr.dlen - IPHLEN;
    if ( dataLen < msgLen(dg) ) {
	xTrace1(ipp, TR_MORE_EVENTS,
		"IP demux : truncating right at byte %d", dataLen);
	msgTruncate(dg, dataLen);
    }
    xIfTrace(ipp, TR_DETAILED)
      dump_header(&hdr);
    if ( is_my_addr(pstate, &(hdr.dest)) == -1 ) {
	xTrace0(ipp, TR_EVENTS ,"IP demux - forwarding this datagram");
	return ipForward(self, dg, &hdr);
    }
    /* 
     * We know the message is either addressed to this host or is broadcast
     */
    activeid.protNum = hdr.prot;
    activeid.remote = hdr.source;
    activeid.local = hdr.dest;
    xTrace2(ipp, TR_MORE_EVENTS, "IP demux lookup activeid %s, %d",
	    ipHostStr(&activeid.remote), activeid.protNum);
    s = (XObj) mapResolve(pstate->activeMap, (char *) &activeid);
    if ( s == ERR_XOBJ ) {
	s = createPassiveSessn(self, &activeid, &hdr);
	if ( s == ERR_XOBJ ) {
	    xTrace0(ipp, TR_EVENTS, "...dropping the message");
	    return XK_SUCCESS;
	}
    } else {
	xTrace0(ipp, TR_EVENTS, "ipDemux found existing session");
    }
    xTrace1(ipp, TR_EVENTS, "Popping to session %x", s);
    s->idle = FALSE;
#ifdef IP_SIM_DROPS
    /*
     * Simulate a dropped packet
     */
    {
	static int c = 0;
	
	if (++c % ( 234 + 4 * hdr.dest.d) == 0) {
	    xTrace0(ipp, TR_ALWAYS, "IP simulates dropped packet");
	    return XK_SUCCESS;
	} else {
	    xTrace2(ipp, TR_FULL_TRACE,
		    "Not dropping packet, c = %d, mod = %d",
		    c, 234 + 4 * hdr.dest.d);
	}
    }
#endif
#ifdef IP_SIM_DELAYS
    /*
     * Simulate a delayed packet
     */
    {
	static int c = 0;
	
	if (++c % ( 234 + 4 * hdr.dest.d) == 0) {
	    xTrace0(ipp, TR_ALWAYS, "IP simulates delayed packet");
	    Delay(50);
	}
    }
#endif
    if (COMPLETEPACKET(hdr)) {
	msgSetAttr(dg, (VOID *)&hdr);
	return xPop(s, transport_s, dg);
    } else {
	return ip_reassemble(s, transport_s, dg, &hdr);
    }
}


static xkern_return_t
ipForward(self, msg, h)
    XObj self;
    Msg *msg;
    IPheader *h;
{
    ActiveId	activeId;
    XObj	s;
    PSTATE	*pstate;

    xAssert(xIsProtocol(self));
    pstate = (PSTATE *)self->state;
    bzero((char *)&activeId, sizeof(activeId));
    /*
     * Look for an active session explicitly for the destination host
     */
    activeId.remote = h->dest;
    s = (XObj)mapResolve(pstate->activeMap, (char *)&activeId);
    if (s == ERR_XOBJ) {
	/*
	 * Look for a session which forwards packets to the destination
	 * network
	 */
	getnet(&activeId.remote, &h->dest);
	s = (XObj)mapResolve(pstate->activeMap, (char *)&activeId);
 	if (s == ERR_XOBJ) {
	    s = xCreateSessn(ip_init_sessn, self, self, 0, 0);
	    s->state = xMalloc(sizeof(SSTATE));
	    ((SSTATE *)s->state)->dest = h->dest;
	    if ( ipHandleRedirect(s) ) {
		return XK_FAILURE;
	    }
	    s->binding = mapBind(pstate->activeMap, (char *)&activeId,
				   (int)s);
	}
    }
    return ( ip_send(s, msg, h) == XMSG_ERR_HANDLE ) ? XK_FAILURE : XK_SUCCESS;
}


/*
 * ipHandleRedirect -- called when the ip session's lower session needs
 * to be (re)opened.  This could be when the ip session is first created
 * and the lower session is noneistent, or when a redirect is received
 * for this session's remote network.  The router is
 * consulted for the best interface.  The new session is assigned to
 * the first position in the ip session's down vector.  The old session,
 * if it existed, is freed.
 *
 * Note that the local IPhost field of the header doesn't change even
 * if the route changes.
 * 
 * preconditions: 
 * 	s->state should be allocated
 * 	s->state->dest should contain the ultimate remote address or net
 *
 * return values:
 *	0 if lower session was succesfully opened and assigned
 *	1 if lower session could not be opened -- old lower session is
 *		not affected
 */
static int
ipHandleRedirect(s)
    XObj s;
{
    IPhost 	host;	
    XObj	llp, lls, llsOld;
    SSTATE	*state;
    PSTATE	*pstate;
    route	*rt;
    int		res;

    /*
     * 'host' is the remote host to which this session sends packets,
     * not necessarily the final destination
     */
    xAssert(xIsSession(s));
    state = (SSTATE *)s->state;
    pstate = (PSTATE *)s->myprotl->state;
    if ((llp = ipGetInterface(pstate, &state->dest)) != NULL) {
	xTrace0(ipp, 3, "ipHandleRedirect got direct interface");
	host = state->dest;
    } else {
	rt = rt_get(&state->dest);
	llp = rt->interface;
	host = rt->gw;
	rt_free(rt);
    }
    if ( (lls = ipOpenInterface(s->myprotl, llp, &host)) == ERR_XOBJ ) {
	xTrace0(ipp, 3, "ipHandleRedirect could not get a lower session");
	return 1;
    }
    xTrace0(ipp, 5, "Successfully opened lls");
    /*
     * Determine mtu for this interface
     */
    res = xControl(lls, GETMAXPACKET, (char *)&state->mtu, sizeof(int));
    if (res < 0 || state->mtu <= 0) {
	xTrace0(ipp, 3, "Could not determine interface mtu");
	state->mtu = IPOPTPACKET;
    }
    if ( xIsSession(llsOld = xGetDown(s, 0)) ) {
	xClose(llsOld);
    }
    xSetDown(s, 0, lls);
    state->llp = llp;
    return 0;
}

/*
 * ipOpenInterface -- open a session with interface llp, connecting to
 * the host 'ipRemote.'  return ERR_XOBJ if 'ipRemote' can not be directly
 * contacted on this interface
 */
static XObj
ipOpenInterface(self, llp, ipRemote)
    XObj self;
    XObj llp;
    IPhost *ipRemote;
{
    ETHhost		remEthHost;
    IPinterfaceinfo 	*info;
    Part		p;

    if ((info = ipLookupIfInfo((PSTATE *)self->state, llp)) == 0) {
	return ERR_XOBJ;
    }
    bcopy((char *)ipRemote, (char *)&remEthHost, sizeof(IPhost));
    if (xControl(info->arpprotl, RESOLVE, (char *)&remEthHost,
		 sizeof(remEthHost)) == -1) {
	xTrace0(ipp, 3, "ipOpenInterface couldn't resolve remote addr");
	return ERR_XOBJ;
    }
    partInit(&p, 1);
    partPush(p, &remEthHost);
    return xOpen(self, self, info->ethprotl, &p);
}


/*
 * ipPop
 */
/*ARGSUSED*/
static xkern_return_t
ipPop(s, down_s, dg)
    XObj s;
    XObj down_s;
    Msg *dg;
{
    IPheader *h;
    IPpseudoHdr ph;
    
    h = (IPheader *)msgGetAttr(dg);
    xAssert(h);
    ph.src = h->source;
    ph.dst = h->dest;
    ph.zero = 0;
    ph.prot = h->prot;
    ph.len = htons( msgLen(dg) );
    msgSetAttr(dg, (VOID *)&ph);
    xTrace1(ipp, TR_EVENTS, "IP pop, length = %d", msgLen(dg));
    xAssert(xIsSession(s));
    return xDemux(s, dg);
}


/* 
 * Misc. routines 
 */
static int
get_ident(dhost, protNum)
    IPhost *dhost;
    u_char protNum;
{
  static int n = 1;
  return n++;
}


static void
getnet(new, old)
    IPhost *new;
    IPhost *old;
{
  new->a = old->a;
  new->b = new->c = new->d = 0;
  if ( CLASSC(*old) )
    new->b = old->b; new->c = old->c;
  if ( CLASSB(*old) )
    new->b = old->b; 
}


/*
 * interface lookup routines
 */

/*
 * is_my_addr:  is this IP address one of mine?
 *
 * return values:
 *	interface index if ipaddr is one of my addresses (either the host
 *	or broadcast address on that interface.)
 *
 *	0 (index of the default interface) if the address is the broadcast
 *	address 0x ff.ff.ff.ff
 *
 *	-1 if ipaddr is some other host
 */
static int
is_my_addr(pstate, ipaddr)
    PSTATE *pstate;
    IPhost *ipaddr;
{
    int i;
    
    for( i = 0; i < pstate->n_interfaces; i++ ) {
	if ( IP_EQUAL(pstate->interface[i].myipaddr, *ipaddr) ||
	    IP_EQUAL(pstate->interface[i].ipbcast, *ipaddr) ) {
	    return i;
	}
    }
    if ( ipaddr->a == 0xff && ipaddr->b == 0xff &&
	 ipaddr->c == 0xff && ipaddr->d == 0xff ) {
	return 0;
    }
    return -1;
}


IPinterfaceinfo *
ipLookupIfInfo(pstate, interface)
    PSTATE *pstate;
    XObj interface;
{
  int i;
  for ( i = 0; i < pstate->n_interfaces; i++ )
    if ( pstate->interface[i].ethprotl == interface )
      return &(pstate->interface[i]);
  return NULL;
}

/*
 * ipGetInterface -- Find an interface through which 'host' can be directly
 * reached.  If it can't, return NULL.
 */
XObj
ipGetInterface(pstate, host)
    PSTATE *pstate;
    IPhost *host;
{
  /* need to look at this, is checking for equal nets correct in the 
     presence of possible subnetting ? */

  int i;
  for ( i = 0; i < pstate->n_interfaces; i++ ) 
    if ( IP_NETEQ(pstate->interface[i].myipaddr,*host) )
      return pstate->interface[i].ethprotl;
  return NULL;
}


/*
 * FRAGMENTATION / REASSEMBLY ROUTINES
 */
static xkern_return_t
ip_reassemble(s, down_s, dg, hdr)
    XObj s;
    XObj down_s;
    Msg *dg;
    IPheader *hdr;
{
  PSTATE *pstate;
  FragId fragid;
  Fragtable *fragtable;
  Fraginfo *fraginfo;
  Hole_ent *hole;
  u_short offset, len;
  xkern_return_t retVal;
  
  pstate = (PSTATE *) s->myprotl->state;
  offset = FRAGOFFSET(hdr->frag)*8;
  len = hdr->dlen - GET_HLEN(hdr) * 4;
  xTrace3(ipp,4,"IP reassemble, seq=%d, off=%d, len=%d",
	  hdr->ident, offset, len);

  fragid.source = hdr->source;
  fragid.dest = hdr->dest;	/* might be multiple IP addresses for me! */
  fragid.prot = hdr->prot;
  fragid.pad  = 0;
  fragid.seqid = hdr->ident;

  fragtable = (Fragtable *) mapResolve(pstate->fragMap, (char *) &fragid);
  if ( fragtable == ERR_FRAG ) {
    xTrace0(ipp,5,"IP reassemble, allocating new Fragtable");
    fragtable = (Fragtable *) xMalloc(sizeof(Fragtable));
    fragtable->binding = mapBind(pstate->fragMap, (char *)&fragid,
 				 (int)fragtable );
    fragtable->listhead =  (Fraginfo *) xMalloc(sizeof(Fraginfo));
    fragtable->nholes = 1;
    fragtable->gcMark = FALSE;
    fraginfo = (Fraginfo *) xMalloc(sizeof(Fraginfo));
    fragtable->listhead->next = fragtable->listhead->prev = fraginfo;
    fraginfo->type = RHOLE;
    fraginfo->data.hole = hole = (Hole_ent *) xMalloc(sizeof(Hole_ent));
    fraginfo->next = fraginfo->prev = fragtable->listhead;
    hole->first = 0;
    hole->last = INFINITE_OFFSET;
  } else {
    xTrace1(ipp,5,"IP reassemble - found fragtable == %x",fragtable);
    fragtable->gcMark = FALSE;
  }
  
  xIfTrace(ipp, 7) {
      xTrace0(ipp, 7, "frag table before adding");
      displayFragTable(fragtable);
  }
  for( fraginfo = fragtable->listhead->next; fraginfo != fragtable->listhead;
      fraginfo = fraginfo->next ) {
    if ( fraginfo->type == RFRAG )
      continue;
    hole = fraginfo->data.hole;
    if ( (offset < hole->last) && ((offset + len) > hole->first) ) {
      xTrace0(ipp, 5, "IP reassemble, found hole for datagram");
      xTrace2(ipp, 6, "hole->first: %d  hole->last: %d",
	      hole->first, hole->last);
      /* check to see if frag overlaps previously received frags */
      if ( offset < hole->first ) {
	xTrace0(ipp,5,"Truncating message from left");
	msgPopDiscard(dg, hole->first - offset);
	offset = hole->first;
      }
      if ( (offset + len) > hole->last ) {
	xTrace0(ipp,5,"Truncating message from right");
	/* msg_truncateright(dg,(hole->last - offset)); */
	msgTruncate(dg, hole->last - offset); 
	len = hole->last - offset;
      }
      /* now check to see if new hole(s) need to be made */
      if ( ((offset + len) < hole->last) && (hdr->frag & MOREFRAGMENTS) ) {
	  /* This hole is not created if this is the last fragment */
	  xTrace0(ipp, 6, "Creating new hole above");
	  hole_create(fraginfo,(offset+len),hole->last);
	  fragtable->nholes++;
      }
      if ( offset > hole->first ) {
	  xTrace0(ipp, 6, "Creating new hole below");
	  hole_create(fraginfo->prev,hole->first,(offset));
	  fragtable->nholes++;
      }
      /* update this fraginfo structure to be an RFRAG */
      xFree((char *)fraginfo->data.hole);
      fragtable->nholes--;
      fraginfo->type = RFRAG;
      /* fraginfo->data.frag = dg; */
      msgConstructCopy(&fraginfo->data.frag, dg); 
      break;
    } /* if found a hole */
  } /* for loop */
  xIfTrace(ipp, 7) {
      xTrace0(ipp, 7, "frag table after adding");
      displayFragTable(fragtable);
  }
  
  /* check to see if we're done */
  if ( fragtable->nholes == 0 ) {
    Msg fullMsg;

    Fraginfo *list;
    xTrace0(ipp,5,"IP reassemble : done collecting frags for datagram");
    /* msg_clear(fullMsg); */
    msgConstructEmpty(&fullMsg);
    for( list = fragtable->listhead->next;
	list != fragtable->listhead; list = list->next ) {
      if (list->type == RHOLE) {
	xTrace0(ipp, 1, "Hole in reassembly process...\n");
	continue;
      }
      msgJoin(&fullMsg, &fullMsg, &list->data.frag);
    }
    if (mapRemoveBinding(pstate->fragMap, fragtable->binding) == -1) {
	xTrace1(ipp, 2, "Error removing entry for seq %d", fragid.seqid);
    } else {
	xTrace1(ipp, 5, "Successfully removed seq %d", fragid.seqid);
    }
    free_fragtable(fragtable);
    msgSetAttr(&fullMsg, (VOID *)hdr);
    xTrace1(ipp,4,"IP reassemble popping up message of length %d",
	    msgLen(&fullMsg));
    retVal = xPop(s, down_s, &fullMsg);
    msgDestroy(&fullMsg);
  } else {
    retVal = XK_SUCCESS;
  }
  return retVal;
}

/* hole_create :
    insert a new hole frag after the given list with the given 
    first and last hole values
*/
static void
hole_create(fraglist, first, last)
    Fraginfo *fraglist;
    u_short first;
    u_short last;
{
  Fraginfo *newfrag;
  Hole_ent *newhole;
  
  xTrace2(ipp,5,"IP hole_create : creating new hole from %d to %d",
	  first,last);
  newfrag = (Fraginfo *) xMalloc(sizeof(Fraginfo));
  newfrag->type = RHOLE;
  newfrag->data.hole = newhole = (Hole_ent *) xMalloc(sizeof(Hole_ent));
  newfrag->prev = fraglist;
  newfrag->next = fraglist->next;
  fraglist->next->prev = newfrag;
  fraglist->next = newfrag;
  
  newhole->first = first;
  newhole->last = last;
}

void
free_fragtable(fragtable)
    Fragtable *fragtable;
{
  Fraginfo *list, *next;

  for( list = fragtable->listhead->next;
      list != fragtable->listhead; list = next ) {
    next = list->next;
    if (list->type == RFRAG) {
      msgDestroy(&list->data.frag);
    }
    xFree((char *)list);
  }
  xFree((char *)fragtable->listhead);
  xFree((char *)fragtable);
}


static void
dump_header(hdr)
    IPheader *hdr;
{
    xTrace5(ipp, TR_ALWAYS,
	    "    version=%d,hlen=%d,type=%d,dlen=%d,ident=%d",
	   GET_VERS(hdr), GET_HLEN(hdr), hdr->type,hdr->dlen,hdr->ident);
    xTrace2(ipp, TR_ALWAYS, "    flags:  DF: %d  MF: %d",
	    (hdr->frag & DONTFRAGMENT) ? 1 : 0, 
	    (hdr->frag & MOREFRAGMENTS) ? 1 : 0);
    xTrace4(ipp, TR_ALWAYS, "    fragoffset=%d,time=%d,prot=%d,checksum=%d",
	   FRAGOFFSET(hdr->frag), hdr->time, hdr->prot, hdr->checksum);
    xTrace2(ipp, TR_ALWAYS, "    source address = %s  dest address %s", 
	    ipHostStr(&hdr->source), ipHostStr(&hdr->dest));
}



static void
displayFragTable(t)
    Fragtable *t;
{
    Fraginfo 	*info;

    xTrace2(ipp, 7, "Table has %d hole%c", t->nholes,
	    t->nholes == 1 ? ' ' : 's');
    for ( info = t->listhead->next; info != t->listhead; info = info->next ) {
	if ( info->type == RHOLE ) {
	    xTrace2(ipp, 7, "hole  first == %d  last == %d",
		    info->data.hole->first, info->data.hole->last);
	} else {
	    xTrace0(ipp, 7, "frag");
	}
    } 
}


static void
callRedirect(s)
    VOID *s;
{
    xTrace1(ipp, 4, "ip: callRedirect runs with session %x", s);
    ipHandleRedirect((XObj) s);
    xClose((XObj)s);
    return;
}


typedef struct {
    int 	(* affected)(
#ifdef __STDC__
			     PSTATE *, IPhost *, route *
#endif
			     );
    route	*rt;
    PSTATE	*pstate;
} RouteChangeInfo;

/* 
 * ipRouteChanged -- For each session in the active map, determine if a
 * change in the given route affects that session.  If it does, the
 * session is reconfigured appropriately.
 */
void
ipRouteChanged(pstate, rt, routeAffected)
    PSTATE *pstate;
    route *rt;
    int (*routeAffected)(
#ifdef __STDC__
			 PSTATE *, IPhost *, route *
#endif			 
			 );
{
    RouteChangeInfo	rInfo;

    rInfo.affected = routeAffected;
    rInfo.rt = rt;
    rInfo.pstate = pstate;
    mapForEach(pstate->activeMap, routeChangeFilter, &rInfo);
}
  

static int
routeChangeFilter(key, value, arg)
    VOID *key;
    int value;
    VOID *arg;
{
    RouteChangeInfo	*rInfo = (RouteChangeInfo *)arg;
    XObj		s = (XObj)value;
    SSTATE		*state;
    
    xAssert(xIsSession(s));
    state = (SSTATE *)s->state;
    xTrace3(ipp, 4, "ipRouteChanged does net %s affect ses %x, dest %s?",
	    ipHostStr(&rInfo->rt->net), s, ipHostStr(&state->dest));
    if ( rInfo->affected(rInfo->pstate, &state->dest, rInfo->rt) ) {
	xTrace1(ipp, 4,
		"session %x affected -- reopening lower session", s);
	xDuplicate(s);
	evDetach( evSchedule(callRedirect, s, 0) );
    } else {
	xTrace1(ipp, 4, "session %x unaffected by routing change", s);
    }
    return 1;
}

	/*
	 * Functions used as arguments to ipRouteChanged
	 */
/*
 * Return true if the remote host is not connected to the local net
 */
int
ipRemoteNet(pstate, dest, rt)
    PSTATE *pstate;
    IPhost *dest;
    route *rt;
{
    return ! ipGetInterface(pstate, dest);
}


/*
 * Return true if the remote host is on the network described by the
 * route.
 */
int
ipSameNet(pstate, dest, rt)
    PSTATE *pstate;
    IPhost *dest;
    route *rt;
{
    return ( (dest->a & rt->mask.a) == (rt->net.a & rt->mask.a) &&
	     (dest->b & rt->mask.b) == (rt->net.b & rt->mask.b) &&
	     (dest->c & rt->mask.c) == (rt->net.c & rt->mask.c) &&
	     (dest->d & rt->mask.d) == (rt->net.d & rt->mask.d) );
}
