/* Copyright (c) 1988 P. A. Buhr */

/*
  This is a monitor implementation of the shortest seek time first (SSTF) disk
  scheduling algorithm.  The SSTF algorithm schedules movement of the disk head to
  the next disk request that is closest to the current location of the disk head.
  */

#include <uMonitor.h>

uMonitor {
#   define NoOfTracks 100
    
    uCondition Tracks[NoOfTracks];
    int NoRequest = 0;
    int CurrentPos = 0, Right, Left;
    
    uEntry void init( ) {
	int i;
	
	for ( i = 0; i < NoOfTracks; i += 1 ) {
	    Tracks[i] = U_CONDITION;
	} /* for */
    } /* init */
    
    uEntry void StartDiskRequest( int TrackNo ) {
	NoRequest += 1;					/* request for disk usage */
	if ( NoRequest > 1 ) {				/* disk in use => wait */
	    uWait Tracks[TrackNo];			/* wait on the condition variable for this track */
	} /* if */
	CurrentPos = TrackNo;
    } /* StartDiskRequest */
    
    uEntry void EndDiskRequest(  ) {
	NoRequest -= 1;
	if ( NoRequest > 0 ) {				/* any processs waiting ? */
	    Right = Left = CurrentPos;
	    for ( ;; ) {				/* scan along tracks in looking for the nearest process to execute */
	    if ( uCondLength( &Tracks[Right] ) != 0 ) { /* found a waiting process ? */
		    uSignal Tracks[Right];		/* signal next request */
		    break;
		} /* exit */
	    if ( uCondLength( &Tracks[Left] ) != 0 ) {	/* found a waiting process ? */
		    uSignal Tracks[Left];		/* signal next request */
		    break;
		} /* exit */
		if ( Right < NoOfTracks - 1 ) Right += 1;
		if ( Left  > 0 ) Left -= 1;
	    } /* for */
	} /* if */
    } /* EndDiskRequest */
} /* monitor */

void DiskUser( int trackno ) {
    StartDiskRequest( trackno );
    uPrintf( "processing request:%d\n", trackno );
    EndDiskRequest( );
    uDie( NULL, 0 );
} /* DiskUser */

#define NoOfTests 10

int tester[NoOfTests] = { 20, 99, 0, 15, 30, 4, 16, 80, 85, 2 };

void uMain( ) {
    uTask process[NoOfTests];
    int i;

    init( );						/* initialize the monitor */

    for ( i = 0; i < NoOfTests; i += 1 ) {
	process[i] = uEmit( DiskUser, tester[i] );
	uDelay( 2 );
    } /* for */
    
    for ( i = 0; i < NoOfTests; i += 1 ) {
	uAbsorb( process[i], NULL, 0 );
    } /* for */
} /* uMain */
