/* Copyright (c) 1989 P. A. Buhr */

/*
  Dining Philosophers Problem
  
  Philosophers are eating around a dining table; however, the table is
  set so that there is only one chop stick between each philosopher. Since
  two chop sticks are necessary to eat, not all philosophers can eat at the
  same time. Fortunately, a philosopher is not always eating, he may be
  just hungry or thinking. Hence, a philosopher can be eating when the
  philosophers on the left and right are not eating.
  */

#include <uSystem.h>

#define NoOfPhil 5
#define NoOfChips 100

#define THINKING 0
#define HUNGRY 1
#define EATING 2

inline int RightOf( int me ) {
    return( ( me + 1 ) % NoOfPhil );
} /* RightOf */

inline int LeftOf( int me ) {
    return( ( me != 0 ) ? me - 1 : NoOfPhil - 1 );
} /* LeftOf */

int state[NoOfPhil] = { THINKING, THINKING, THINKING, THINKING, THINKING };
uSemaphore self[NoOfPhil] = { U_SEMAPHORE( 0 ), U_SEMAPHORE( 0 ), U_SEMAPHORE( 0 ), U_SEMAPHORE( 0 ), U_SEMAPHORE( 0 ) };
uSemaphore lock = U_SEMAPHORE( 1 );

void TestBeside( int me ) {
    if ( state[LeftOf( me )] != EATING && state[me] == HUNGRY && state[RightOf( me )] != EATING ) {
	state[me] = EATING;
	uV( &self[me] );
    } /* if */
} /* TestBeside */

void pickup( int me ) {
    uP( &lock );
    state[me] = HUNGRY;
    TestBeside( me );
    uV( &lock );
    uP( &self[me] );
} /* pickup */

void putdown( int me ) {
    uP( &lock );
    state[me] = THINKING;
    TestBeside( LeftOf( me ) );
    TestBeside( RightOf( me ) );
    uV( &lock );
} /* putdown */

void philosopher( int i, int chips ) {
    int bite;
    int random( void );
    
    uPrintf( "philosopher%d will eat %d chips\n", i, chips );
    for ( ;; ) {
	pickup( i );					/* pick up both chop sticks */
	bite = random(  ) % 5 + 1;			/* take a bite */
	if ( bite >= chips ) {				/* finished eating ? */
	    uPrintf( "philosopher%d finished eating the last %d chips\n", i, chips );
	    putdown( i );
	    break;
	} /* exit */
	chips -= bite;					/* reduce chips */
	uPrintf( "philosopher%d is eating %d chips leaving %d chips\n", i, bite, chips );
	putdown( i );					/* put down both chop sticks */
    } /* for */
    uDie( NULL, 0 );
} /* philosopher */

void uMain( ) {
    int i;
    uTask phil[NoOfPhil];
    
    for ( i = 0; i < NoOfPhil; i += 1 ) {		/* create philosophers */
	phil[i] = uEmit( philosopher, i, NoOfChips );
    } /* for */
    
    for ( i = 0; i < NoOfPhil; i += 1 ) {		/* wait for completion of meal */
	uAbsorb( phil[i], NULL, 0 );
    } /* for */
    uPrintf( "successful completion\n" );
} /* uMain */

/* Local Variables: */
/* compile-command: "concc -O DiningPhil.c" */
/* End: */
